package ru.yandex.direct.core.entity.uac.repository.stat

import java.math.BigDecimal
import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.campaign_content.AssetStat
import ru.yandex.direct.grid.schema.yt.Tables
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.longNodeBigDecimal
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode

@Repository
class AssetStatRepository(
    private val contextProvider: AssetsStatContextProvider,
) {
    companion object {
        private val ASSETS_STAT = Tables.ASSETSSTAT_BS.`as`("AssetsStat")
        private val SUM_SHOWS = DSL.sum(ASSETS_STAT.SHOWS).`as`("shows")
        private val SUM_CLICKS = DSL.sum(ASSETS_STAT.CLICKS).`as`("clicks")
        private val SUM_CLICKS_COST = DSL.sum(ASSETS_STAT.CLICKS_COST).`as`("clicksCost")
        private val SUM_CLICKS_COST_CUR = DSL.sum(ASSETS_STAT.CLICKS_COST_CUR).`as`("clicksCostCur")
        private val SUM_COST = DSL.sum(ASSETS_STAT.COST).`as`("cost")
        private val SUM_COST_CUR = DSL.sum(ASSETS_STAT.COST_CUR).`as`("costCur")
        private val SUM_COST_TAX_FREE = DSL.sum(ASSETS_STAT.COST_TAX_FREE).`as`("costTaxFree")
        private val SUM_CONVERSIONS = DSL.sum(ASSETS_STAT.CONVERSIONS).`as`("conversions")
        private val SUM_WITH_SHOWS_CONVERSIONS = DSL.sum(ASSETS_STAT.WITH_SHOWS_CONVERSIONS).`as`("withShowsConversions")
        private val SUM_INSTALLS = DSL.sum(ASSETS_STAT.INSTALLS).`as`("installs")
        private val SUM_WITH_SHOWS_INSTALLS = DSL.sum(ASSETS_STAT.WITH_SHOWS_INSTALLS).`as`("withShowsInstalls")
        private val ASSET_TYPE = ASSETS_STAT.ASSET_TYPE.`as`("assetType")
        private val ASSET_HASH = ASSETS_STAT.ASSET_HASH.`as`("assetHash")
    }

    fun getStatsByAssetIds(
        assetIds: Set<String>,
        orderId: Long,
        dateTimeFrom: Long,
        dateTimeTo: Long,
    ): List<AssetStat> {
        val query = YtDSL.ytContext()
            .select(
                ASSET_TYPE,
                ASSET_HASH,
                SUM_SHOWS,
                SUM_CLICKS,
                SUM_CLICKS_COST,
                SUM_CLICKS_COST_CUR,
                SUM_COST,
                SUM_COST_CUR,
                SUM_COST_TAX_FREE,
                SUM_CONVERSIONS,
                SUM_WITH_SHOWS_CONVERSIONS,
                SUM_INSTALLS,
                SUM_WITH_SHOWS_INSTALLS,
            )
            .from(ASSETS_STAT)
            .where(ASSET_HASH.`in`(assetIds))
            .and(ASSETS_STAT.ORDER_ID.eq(orderId))
            .and(ASSETS_STAT.DATE.greaterOrEqual(dateTimeFrom))
            .and(ASSETS_STAT.DATE.lessOrEqual(dateTimeTo))
            .groupBy(ASSET_TYPE, ASSET_HASH, ASSETS_STAT.ORDER_ID)

        return contextProvider.getContext()
            .executeSelect(query)
            .yTreeRows
            .map { convertFromYt(it) }
    }

    private fun convertFromYt(
        node: YTreeMapNode
    ) = AssetStat(
        id = node.getLong(ASSET_HASH.name).toULong().toString(),
        mediaType = MediaType.fromId(node.getInt(ASSET_TYPE.name)),
        shows = longNodeBigDecimal(node, SUM_SHOWS.name) ?: BigDecimal.ZERO,
        clicks = longNodeBigDecimal(node, SUM_CLICKS.name) ?: BigDecimal.ZERO,
        clicksCost = longNodeBigDecimal(node, SUM_CLICKS_COST.name) ?: BigDecimal.ZERO,
        clicksCostCur = longNodeBigDecimal(node, SUM_CLICKS_COST_CUR.name) ?: BigDecimal.ZERO,
        cost = longNodeBigDecimal(node, SUM_COST.name) ?: BigDecimal.ZERO,
        costCur = longNodeBigDecimal(node, SUM_COST_CUR.name) ?: BigDecimal.ZERO,
        costTaxFree = longNodeBigDecimal(node, SUM_COST_TAX_FREE.name) ?: BigDecimal.ZERO,
        conversions = longNodeBigDecimal(node, SUM_CONVERSIONS.name) ?: BigDecimal.ZERO,
        installs = longNodeBigDecimal(node, SUM_INSTALLS.name) ?: BigDecimal.ZERO,
        postViewConversions = longNodeBigDecimal(node, SUM_WITH_SHOWS_CONVERSIONS.name) ?: BigDecimal.ZERO,
        postViewInstalls = longNodeBigDecimal(node, SUM_WITH_SHOWS_INSTALLS.name) ?: BigDecimal.ZERO,
    )
}
