package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.fromEpochSecond
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAccount
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.ACCOUNT
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.AccountTable
import ru.yandex.direct.utils.JsonUtils.toJson
import ru.yandex.direct.utils.fromJson
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable

@Lazy
@Repository
class UacYdbAccountRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {
    fun getAccountByClientId(clientId: Long): UacYdbAccount? {
        val accountClientIdIndex = ACCOUNT.withIndex(ACCOUNT.ACCOUNT_CLIENT_ID_INDEX) as AccountTable
        val queryBuilder = select(
            accountClientIdIndex.ID,
            accountClientIdIndex.FEATURES,
            accountClientIdIndex.CREATED_AT,
            accountClientIdIndex.UID,
            accountClientIdIndex.DIRECT_CLIENT_ID,
        )
            .from(accountClientIdIndex)
            .where(accountClientIdIndex.DIRECT_CLIENT_ID.eq(clientId))

        return fetchAccount(queryBuilder, accountClientIdIndex)
    }

    fun getAccountById(id: String): UacYdbAccount? {
        val queryBuilder = select(
            ACCOUNT.ID,
            ACCOUNT.FEATURES,
            ACCOUNT.CREATED_AT,
            ACCOUNT.UID,
            ACCOUNT.DIRECT_CLIENT_ID,
        )
            .from(ACCOUNT)
            .where(ACCOUNT.ID.eq(id.toIdLong()))

        return fetchAccount(queryBuilder, ACCOUNT)
    }

    private fun fetchAccount(queryBuilder: QueryBuilder, accountTable: AccountTable): UacYdbAccount? {
        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        if (!result.next()) {
            return null
        }

        return UacYdbAccount(
            id = result.getValueReader(accountTable.ID).uint64.toIdString(),
            features = result.getValueReaderOrNull(accountTable.FEATURES)?.json?.let { fromJson(it) },
            createdAt = result.getValueReader(accountTable.CREATED_AT).uint64.let { fromEpochSecond(it) },
            uid = result.getValueReader(accountTable.UID).uint64,
            directClientId = result.getValueReader(accountTable.DIRECT_CLIENT_ID).uint64,
        )
    }

    fun saveAccount(account: UacYdbAccount) {
        val insertValues = buildTempTable {
            value(ACCOUNT.ID, account.id.toIdLong())
            value(ACCOUNT.FEATURES, toJson(account.features))
            value(ACCOUNT.CREATED_AT, UacYdbUtils.toEpochSecond(account.createdAt))
            value(ACCOUNT.UID, account.uid)
            value(ACCOUNT.DIRECT_CLIENT_ID, account.directClientId)
        }

        val queryAndParams = InsertBuilder.insertInto(ACCOUNT)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }
}
