package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toId
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.APP_INFO
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.APP_INFO_BUNDLE_ID_INDEX
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.AppInfoBundleIdIndexTable
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable

@Lazy
@Repository
class UacYdbAppInfoRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {
    fun getAppInfoById(id: String): UacYdbAppInfo? {
        return getAppInfoByIds(listOf(id)).firstOrNull()
    }

    fun getAppInfoByIds(ids: Collection<String>): List<UacYdbAppInfo> {
        val queryBuilder = select(
            APP_INFO.ID,
            APP_INFO.PK_VERSION,
            APP_INFO.APP_ID,
            APP_INFO.BUNDLE_ID,
            APP_INFO.LANGUAGE,
            APP_INFO.REGION,
            APP_INFO.PLATFORM,
            APP_INFO.SOURCE,
            APP_INFO.DATA,
            APP_INFO.UPDATED_AT
        )
            .from(APP_INFO)
            .where(APP_INFO.ID.`in`(ids.toIdsLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        val appInfos = mutableListOf<UacYdbAppInfo>()
        while (result.next()) {
            appInfos.add(readAppInfo(result))
        }
        return appInfos
    }

    private fun getAppInfoIdsByBundleIds(bundleIds: Collection<String>): List<String> {
        val indexTable = APP_INFO_BUNDLE_ID_INDEX
            .withIndex(APP_INFO_BUNDLE_ID_INDEX.BUNDLE_ID_INDEX) as AppInfoBundleIdIndexTable
        val queryBuilder = select(
            indexTable.ID
        )
            .from(indexTable)
            .where(indexTable.BUNDLE_ID.`in`(bundleIds))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        val ids = mutableListOf<String>()

        while (result.next()) {
            ids.add(result.getValueReader(indexTable.ID).uint64.toIdString())
        }
        return ids
    }

    fun getAppInfoByBundleIds(bundleIds: Collection<String>): List<UacYdbAppInfo> {
        return getAppInfoByIds(getAppInfoIdsByBundleIds(bundleIds))
    }

    fun getAppInfoByUrl(url: MobileAppStoreUrl): UacYdbAppInfo? {
        val queryBuilder = select(
            APP_INFO.ID,
            APP_INFO.PK_VERSION,
            APP_INFO.APP_ID,
            APP_INFO.BUNDLE_ID,
            APP_INFO.LANGUAGE,
            APP_INFO.REGION,
            APP_INFO.PLATFORM,
            APP_INFO.SOURCE,
            APP_INFO.DATA,
            APP_INFO.UPDATED_AT
        )
            .from(APP_INFO)
            .where(
                APP_INFO.ID.eq(url.toId())
            )
            .limit(1)

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        return if (result.next()) readAppInfo(result) else null
    }

    private fun readAppInfo(result: ResultSetReaderWrapped): UacYdbAppInfo =
        UacYdbAppInfo(
            id = result.getValueReader(APP_INFO.ID).uint64.toIdString(),
            pkVersion = result.getValueReader(APP_INFO.PK_VERSION).uint32.toInt(),
            appId = result.getValueReader(APP_INFO.APP_ID).utf8,
            bundleId = result.getValueReader(APP_INFO.BUNDLE_ID).utf8,
            language = result.getValueReader(APP_INFO.LANGUAGE).utf8,
            region = result.getValueReader(APP_INFO.REGION).utf8,
            platform = Platform.fromId(result.getValueReader(APP_INFO.PLATFORM).uint32.toInt()),
            source = Store.fromId(result.getValueReader(APP_INFO.SOURCE).uint32.toInt()),
            data = result.getValueReader(APP_INFO.DATA).jsonDocument,
            updatedAt = UacYdbUtils.fromEpochSecond(result.getValueReader(APP_INFO.UPDATED_AT).uint64),
        )

    private fun saveAppInfoInCommonTable(appInfo: UacYdbAppInfo) {
        val insertValues = buildTempTable {
            value(APP_INFO.ID, appInfo.id.toIdLong())
            value(APP_INFO.PK_VERSION, appInfo.pkVersion)
            value(APP_INFO.APP_ID, appInfo.appId)
            value(APP_INFO.BUNDLE_ID, appInfo.bundleId)
            value(APP_INFO.LANGUAGE, appInfo.language)
            value(APP_INFO.REGION, appInfo.region)
            value(APP_INFO.PLATFORM, appInfo.platform.id)
            value(APP_INFO.SOURCE, appInfo.source.id)
            value(APP_INFO.DATA, appInfo.data)
            value(APP_INFO.UPDATED_AT, UacYdbUtils.toEpochSecond(appInfo.updatedAt))
        }

        val queryAndParams = InsertBuilder.upsertInto(APP_INFO)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    private fun saveAppInfoInBundleIdIndexTable(appInfo: UacYdbAppInfo) {
        val insertValues = buildTempTable {
            value(APP_INFO_BUNDLE_ID_INDEX.ID, appInfo.id.toIdLong())
            value(APP_INFO_BUNDLE_ID_INDEX.BUNDLE_ID, appInfo.bundleId)
        }

        val queryAndParams = InsertBuilder.upsertInto(APP_INFO_BUNDLE_ID_INDEX)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    fun saveAppInfo(appInfo: UacYdbAppInfo) {
        saveAppInfoInCommonTable(appInfo)
        saveAppInfoInBundleIdIndexTable(appInfo)
    }
}
