package ru.yandex.direct.core.entity.uac.repository.ydb

import java.time.LocalDateTime
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toEpochSecond
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CAMPAIGN_CONTENT
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CAMPAIGN_CONTENT_TEST
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CampaignContentTable
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_CAMPAIGN
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DirectCampaignTable
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.utils.JsonUtils.toJson
import ru.yandex.direct.utils.fromJson
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.upsertInto
import ru.yandex.direct.ydb.builder.querybuilder.JoinBuilder
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.set
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.update
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable

@Lazy
@Repository
class UacYdbCampaignContentRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath
) {
    fun getCampaignContents(campaignId: String): List<UacYdbCampaignContent> {
        val table = CAMPAIGN_CONTENT.withIndex(CAMPAIGN_CONTENT.CAMPAIGN_CONTENT_CAMPAIGN_ID_INDEX) as CampaignContentTable
        val queryBuilder = select(
            table.ID,
            table.CAMPAIGN_ID,
            table.CONTENT_ID,
            table.TYPE,
            table.ORDER,
            table.TEXT,
            table.CREATED_AT,
            table.REMOVED_AT,
            table.STATUS,
            table.REJECT_REASONS,
            table.SITELINK,
        )
            .from(table)
            .where(table.CAMPAIGN_ID.eq(campaignId.toIdLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)
        val contents = mutableListOf<UacYdbCampaignContent>()
        while (result.next()) {
            val content = UacYdbCampaignContent(
                id = result.getValueReader(table.ID).uint64.toIdString(),
                campaignId = campaignId,
                contentId = result.getValueReaderOrNull(table.CONTENT_ID)?.uint64?.toIdString(),
                type = MediaType.fromId(result.getValueReader(table.TYPE).uint32.toInt()),
                order = result.getValueReader(table.ORDER).uint32.toInt(),
                text = result.getValueReaderOrNull(table.TEXT)?.utf8,
                createdAt = UacYdbUtils.fromEpochSecond(result.getValueReader(table.CREATED_AT).uint64),
                removedAt = result.getValueReaderOrNull(table.REMOVED_AT)
                    ?.uint64
                    ?.let { UacYdbUtils.fromEpochSecond(it) },
                status = CampaignContentStatus.fromId(result.getValueReader(table.STATUS).uint32.toInt()),
                rejectReasons = result.getValueReaderOrNull(table.REJECT_REASONS)?.json?.let { fromJson(it) },
                sitelink = result.getValueReaderOrNull(table.SITELINK)?.json?.let { fromJson(it) },
            )
            contents.add(content)
        }
        return contents
    }

    fun getCampaignsContents(campaignIds: Collection<String>): Map<String, List<UacYdbCampaignContent>> {
        val table = CAMPAIGN_CONTENT.withIndex(CAMPAIGN_CONTENT.CAMPAIGN_CONTENT_CAMPAIGN_ID_INDEX) as CampaignContentTable
        val queryBuilder = select(
            table.ID,
            table.CAMPAIGN_ID,
            table.CONTENT_ID,
            table.TYPE,
            table.ORDER,
            table.TEXT,
            table.CREATED_AT,
            table.REMOVED_AT,
            table.STATUS,
            table.REJECT_REASONS,
            table.SITELINK,
        )
            .from(table)
            .where(table.ID.gt(0).and(table.CAMPAIGN_ID.`in`(campaignIds.toIdsLong())))
            .orderBy(table.ID)

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)
        val contents = mutableMapOf<String, MutableList<UacYdbCampaignContent>>()
        while (result.next()) {
            val content = UacYdbCampaignContent(
                id = result.getValueReader(table.ID).uint64.toIdString(),
                campaignId = result.getValueReader(table.CAMPAIGN_ID).uint64.toIdString(),
                contentId = result.getValueReaderOrNull(table.CONTENT_ID)?.uint64?.toIdString(),
                type = MediaType.fromId(result.getValueReader(table.TYPE).uint32.toInt()),
                order = result.getValueReader(table.ORDER).uint32.toInt(),
                text = result.getValueReaderOrNull(table.TEXT)?.utf8,
                createdAt = UacYdbUtils.fromEpochSecond(result.getValueReader(table.CREATED_AT).uint64),
                removedAt = result.getValueReaderOrNull(table.REMOVED_AT)
                    ?.uint64
                    ?.let { UacYdbUtils.fromEpochSecond(it) },
                status = CampaignContentStatus.fromId(result.getValueReader(table.STATUS).uint32.toInt()),
                rejectReasons = result.getValueReaderOrNull(table.REJECT_REASONS)?.json?.let { fromJson(it) },
                sitelink = result.getValueReaderOrNull(table.SITELINK)?.json?.let { fromJson(it) },
            )
            contents.computeIfAbsent(content.campaignId) { _ -> mutableListOf() }
            contents[content.campaignId]!!.add(content)
        }
        return contents
    }

    /**
     * Возвращает соответствия вида тип контента - текст, для видов TEXT и TITLE, с любым статусом кроме DELETED
     */
    fun getTypeToTextsByCampaignId(campaignId: String): Map<MediaType, List<String>> {
        val table = CAMPAIGN_CONTENT.withIndex(CAMPAIGN_CONTENT.CAMPAIGN_CONTENT_CAMPAIGN_ID_INDEX) as CampaignContentTable
        val query = select(table.TEXT, table.TYPE)
            .from(table)
            .where(table.CAMPAIGN_ID.eq(campaignId.toIdLong())
                .and(table.TYPE.`in`(MediaType.TEXT.id, MediaType.TITLE.id))
                .and(table.STATUS.neq(CampaignContentStatus.DELETED.id))
                .and(table.REMOVED_AT.isNull))
            .queryAndParams(path)
        val result = ydbClient.executeQuery(query, true)
            .getResultSet(0)
        val typeToTexts: MutableMap<MediaType, MutableList<String>> = mutableMapOf()
        while (result.next()) {

            val type = MediaType.fromId(result.getValueReader(table.TYPE).uint32.toInt())
            val text = result.getValueReader(table.TEXT).utf8

            typeToTexts.computeIfAbsent(type) { mutableListOf() }
                .add(text)
        }
        return typeToTexts
    }


    private fun getIdsForCampaignId(campaignId: String): List<Long> {
        val campaignContentCampaignIndex = CAMPAIGN_CONTENT.withIndex(CAMPAIGN_CONTENT.CAMPAIGN_CONTENT_CAMPAIGN_ID_INDEX) as CampaignContentTable
        val campaignContentIdsQuery = select(campaignContentCampaignIndex.ID)
            .from(campaignContentCampaignIndex)
            .where(campaignContentCampaignIndex.CAMPAIGN_ID.eq(campaignId.toIdLong()))
            .queryAndParams(path)

        val result = ydbClient.executeQuery(campaignContentIdsQuery, true)
            .getResultSet(0)
        val ids = mutableListOf<Long>()
        while (result.next()) {
            ids.add(result.getValueReader(campaignContentCampaignIndex.ID).uint64)
        }
        return ids
    }

    /**
     * Возвращает contentId всех не удаленных ассетов кампании по переданному типу [mediaType]
     */
    fun getContentIdsByDirectCampaignId(
        directCampaignId: Long,
        mediaType: MediaType,
    ): List<String> {
        val directCampaignTable =
            DIRECT_CAMPAIGN.withIndex(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID_INDEX) as DirectCampaignTable
        val campaignContentTable =
            CAMPAIGN_CONTENT.withIndex(CAMPAIGN_CONTENT.CAMPAIGN_CONTENT_CAMPAIGN_ID_INDEX) as CampaignContentTable

        val query = select(campaignContentTable.CONTENT_ID)
            .from(directCampaignTable)
            .join(
                campaignContentTable,
                JoinBuilder.JoinStatement.on(campaignContentTable.CAMPAIGN_ID, directCampaignTable.ID)
            )
            .where(
                directCampaignTable.DIRECT_CAMPAIGN_ID.eq(directCampaignId)
                    .and(campaignContentTable.TYPE.eq(mediaType.id))
                    .and(campaignContentTable.REMOVED_AT.isNull)
            )
            .queryAndParams(path)

        val result = ydbClient.executeQuery(query, true)
            .getResultSet(0)
        val contentIds = mutableListOf<String>()
        while (result.next()) {
            contentIds.add(result.getValueReader(campaignContentTable.CONTENT_ID).uint64.toIdString())
        }
        return contentIds
    }

    fun deleteByCampaignId(campaignId: String) {
        val ids = getIdsForCampaignId(campaignId)
        if (ids.isEmpty()) {
            return
        }
        val queryAndParams = deleteFrom(CAMPAIGN_CONTENT)
            .where(CAMPAIGN_CONTENT.ID.`in`(ids))
            .queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }

    fun delete(ids: Collection<String>) {
        val queryAndParams = deleteFrom(CAMPAIGN_CONTENT)
            .where(CAMPAIGN_CONTENT.ID.`in`(ids.toIdsLong()))
            .queryAndParams(path)
        ydbClient.executeQuery(queryAndParams)
    }

    fun addCampaignContents(contents: Collection<UacYdbCampaignContent>) {
        if (contents.isEmpty()) {
            return
        }

        val insertValues = buildTempTable {
            contents.forEach { content ->
                value(CAMPAIGN_CONTENT.ID, content.id.toIdLong())
                value(CAMPAIGN_CONTENT.CAMPAIGN_ID, content.campaignId.toIdLong())
                value(CAMPAIGN_CONTENT.CONTENT_ID, content.contentId?.toIdLong())
                value(CAMPAIGN_CONTENT.TYPE, content.type!!.id)
                value(CAMPAIGN_CONTENT.ORDER, content.order)
                value(CAMPAIGN_CONTENT.TEXT, content.text)
                value(CAMPAIGN_CONTENT.CREATED_AT, toEpochSecond(content.createdAt))
                value(CAMPAIGN_CONTENT.REMOVED_AT, content.removedAt?.let { toEpochSecond(it) })
                value(CAMPAIGN_CONTENT.STATUS, content.status.id)
                value(CAMPAIGN_CONTENT.REJECT_REASONS, toJson(content.rejectReasons))
                value(CAMPAIGN_CONTENT.SITELINK, toJson(content.sitelink))
                newRecord()
            }
        }

        val queryAndParams = InsertBuilder.insertInto(CAMPAIGN_CONTENT)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    fun update(
        ids: Collection<String>,
        newRemovedAt: LocalDateTime?,
        newStatus: CampaignContentStatus,
    ) {
        val queryBuilder = update(CAMPAIGN_CONTENT,
            set(CAMPAIGN_CONTENT.REMOVED_AT, toEpochSecond(newRemovedAt))
                .set(CAMPAIGN_CONTENT.STATUS, newStatus.id)
        ).where(CAMPAIGN_CONTENT.ID.`in`(ids.toIdsLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateStatusForCampaignContents(
        campaignContentIds: Collection<String>,
        status: CampaignContentStatus,
    ) {
        val builder = update(CAMPAIGN_CONTENT,
            set(CAMPAIGN_CONTENT.STATUS, status.id))
            .where(CAMPAIGN_CONTENT.ID.`in`(campaignContentIds.map { it.toIdLong() }))

        val queryAndParams = builder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateStatusForCampaignContentsForTestTable(
        campaignContentIds: Collection<String>,
        status: CampaignContentStatus,
    ) {
        val insertValues = buildTempTable {
            for (campaignContentId in campaignContentIds) {
                value(CAMPAIGN_CONTENT_TEST.ID, campaignContentId.toIdLong())
                value(CAMPAIGN_CONTENT_TEST.STATUS, status.id)
                newRecord()
            }
        }

        val queryAndParams = upsertInto(CAMPAIGN_CONTENT_TEST)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateOrderForCampaignContent(
        id: String,
        order: Int,
    ) {
        val builder: QueryBuilder = update(CAMPAIGN_CONTENT,
            set(CAMPAIGN_CONTENT.ORDER, order)
        ).where(CAMPAIGN_CONTENT.ID.eq(id.toIdLong()))

        val queryAndParams = builder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateRejectReasons(ktModelChanges: KtModelChanges<String, UacYdbCampaignContent>) {
        var updateStatement: UpdateBuilder.SetStatement? = null
        updateStatement = updateStatement
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.REJECT_REASONS, UacYdbCampaignContent::rejectReasons,
                { rejectReasons -> if (rejectReasons != null) toJson(rejectReasons) else null })
        if (updateStatement == null) {
            return
        }
        val query = update(CAMPAIGN_CONTENT, updateStatement)
            .where(CAMPAIGN_CONTENT.ID.eq(ktModelChanges.id.toIdLong()))
            .queryAndParams(path)
        ydbClient.executeQuery(query, true)
    }

    fun updateRejectReasonsForTestTable(id: String, rejectReasons: List<ModerationDiagData>?) {
        val dbRejectedReasons = if (rejectReasons != null) toJson(rejectReasons) else null
        val insertValues = buildTempTable {
            value(CAMPAIGN_CONTENT_TEST.ID, id.toIdLong())
            value(CAMPAIGN_CONTENT_TEST.REJECT_REASONS, dbRejectedReasons)
            newRecord()
        }

        val queryAndParams = upsertInto(CAMPAIGN_CONTENT_TEST)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    fun update(ktModelChanges: KtModelChanges<String, UacYdbCampaignContent>) {
        var updateStatement: UpdateBuilder.SetStatement? = null

        updateStatement = updateStatement
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.CREATED_AT, UacYdbCampaignContent::createdAt) { toEpochSecond(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.ORDER, UacYdbCampaignContent::order)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.TEXT, UacYdbCampaignContent::text)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.REMOVED_AT, UacYdbCampaignContent::removedAt) { toEpochSecond(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.STATUS, UacYdbCampaignContent::status) { it.id }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.REJECT_REASONS, UacYdbCampaignContent::rejectReasons) { toJson(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN_CONTENT.SITELINK, UacYdbCampaignContent::sitelink) { toJson(it) }

        if (updateStatement == null) {
            return
        }
        val query = update(CAMPAIGN_CONTENT, updateStatement)
            .where(CAMPAIGN_CONTENT.ID.eq(ktModelChanges.id.toIdLong()))
            .queryAndParams(path)

        ydbClient.executeQuery(query, true)
    }
}
