package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration.Companion.UAC_YDB_CLIENT_BEAN
import ru.yandex.direct.common.configuration.UacYdbConfiguration.Companion.UAC_YDB_PATH_BEAN
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.BaseUacTrackingInfo
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.model.TargetStatus
import ru.yandex.direct.core.entity.uac.model.TargetType
import ru.yandex.direct.core.entity.uac.model.UacStrategyPlatform
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.fromEpochSecond
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toEpochSecond
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.APP_INFO
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CAMPAIGN
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CAMPAIGN_REGIONS
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CAMPAIGN_TEST
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CampaignTable
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_CAMPAIGN
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.utils.JsonUtils.toJson
import ru.yandex.direct.utils.JsonUtils.toJsonNullable
import ru.yandex.direct.utils.fromJson
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.insertInto
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.upsertInto
import ru.yandex.direct.ydb.builder.querybuilder.JoinBuilder.JoinStatement.on
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.selectDistinct
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.set
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.update
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable
import java.time.LocalDateTime.now

data class CampaignByAccountRequest(
    val accountId: Long,
    val name: String? = null,
    val advTypes: List<Int>? = null,
    val statuses: List<Int>? = null,
    val appPlatform: Int? = null,
    val apps: List<Long>? = null,
    val regions: List<Long>? = null,
)

@Lazy
@Repository
class UacYdbCampaignRepository(
    @Qualifier(UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UAC_YDB_PATH_BEAN) var path: YdbPath
) {

    /**
     * Добавила пока этот метод для примера как пользоваться индексами и подзапросами, доделаю в рамках https://st.yandex-team.ru/DIRECT-145304
     */
    fun getCampaignIdsByAccount(request: CampaignByAccountRequest) {
        val campaignAccountIndex = CAMPAIGN.withIndex(CAMPAIGN.CAMPAIGN_ACCOUNT_INDEX) as CampaignTable
        var accountSubQueryPredicate = campaignAccountIndex.ACCOUNT_ID.eq(request.accountId)
        if (request.name != null) {
            accountSubQueryPredicate = accountSubQueryPredicate.and(campaignAccountIndex.NAME.ilike(request.name))
        }
        val advTypes = request.advTypes ?: AdvType.values().map { it.id }

        accountSubQueryPredicate = accountSubQueryPredicate.and(campaignAccountIndex.ADV_TYPE.`in`(advTypes))
        if (request.apps != null) {
            accountSubQueryPredicate = accountSubQueryPredicate.and(campaignAccountIndex.APP_ID.`in`(request.apps))
        }
        if (request.statuses != null) {
            accountSubQueryPredicate = accountSubQueryPredicate.and(campaignAccountIndex.STATUS.`in`(request.statuses))
        }

        var query = select()
            .from(campaignAccountIndex)
            .where(accountSubQueryPredicate)
        var campaignIdColumn = campaignAccountIndex.ID.`as`("id")
        if (request.appPlatform != null) {
            val accountSubQuery = query.asSubQuery().`as`("accountSubQuery")
            campaignIdColumn = accountSubQuery.getColumn(campaignIdColumn).`as`("id")
            query = selectDistinct(campaignIdColumn)
                .from(accountSubQuery)
                .join(
                    APP_INFO,
                    on(campaignIdColumn, APP_INFO.ID)
                )
                .where(APP_INFO.PLATFORM.eq(request.appPlatform))
        }
        if (request.regions != null && request.regions.isNotEmpty()) {
            val subQuery = query.asSubQuery().`as`("accountSecondSubQuery")
            campaignIdColumn = subQuery.getColumn(campaignIdColumn).`as`("id")
            query = selectDistinct(campaignIdColumn)
                .from(subQuery)
                .join(CAMPAIGN_REGIONS, on(campaignIdColumn, CAMPAIGN_REGIONS.ID))
                .where(CAMPAIGN_REGIONS.REGION.`in`(request.regions))
        }
        val queryWithOrderBy = query.orderBy(campaignIdColumn)
        // todo pagination?
        val queryAndParams = queryWithOrderBy.queryAndParams(path)
        ydbClient.executeOnlineRoQuery(queryAndParams, true)
    }

    fun getAppIdsByAccount(request: CampaignByAccountRequest): List<Long> {
        val campaignAccountIndex = CAMPAIGN.withIndex(CAMPAIGN.CAMPAIGN_ACCOUNT_INDEX) as CampaignTable

        val query = selectDistinct(campaignAccountIndex.APP_ID)
            .from(campaignAccountIndex)
            .where(
                campaignAccountIndex.ACCOUNT_ID.eq(request.accountId).and(
                    campaignAccountIndex.APP_ID.isNotNull
                )
            )

        val queryAndParams = query.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        val appIds = mutableListOf<Long>()
        while (result.next()) {
            appIds.add(result.getValueReader(campaignAccountIndex.APP_ID).uint64)
        }
        return appIds
    }

    fun getTrackingInfoByAccountAndAppId(accountId: String, appId: String): List<BaseUacTrackingInfo> {
        val campaignAccountIndex = CAMPAIGN.withIndex(CAMPAIGN.CAMPAIGN_ACCOUNT_INDEX) as CampaignTable

        val query = select(
            campaignAccountIndex.TRACKING_URL,
            campaignAccountIndex.IMPRESSION_URL,
            DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID,
            campaignAccountIndex.NAME,
            campaignAccountIndex.CREATED_AT,
        )
            .from(campaignAccountIndex)
            .join(DIRECT_CAMPAIGN, on(campaignAccountIndex.ID, DIRECT_CAMPAIGN.ID))
            .where(
                campaignAccountIndex.ACCOUNT_ID.eq(accountId.toIdLong())
                    .and(campaignAccountIndex.APP_ID.eq(appId.toIdLong()))
            )

        val queryAndParams = query.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        val trackingInfos = mutableListOf<BaseUacTrackingInfo>()
        while (result.next()) {
            trackingInfos.add(BaseUacTrackingInfo(
                trackingUrl = result.getValueReaderOrNull(campaignAccountIndex.TRACKING_URL)?.utf8,
                impressionUrl = result.getValueReaderOrNull(campaignAccountIndex.IMPRESSION_URL)?.utf8,
                id = result.getValueReader(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID).uint64,
                name = result.getValueReader(campaignAccountIndex.NAME).utf8,
                createdTime = result.getValueReader(campaignAccountIndex.CREATED_AT).uint64.let { fromEpochSecond(it) },
            ))
        }
        return trackingInfos
    }

    fun getCampaign(id: String): UacYdbCampaign? {
        val gotCampaign = getCampaigns(listOf(id));
        if (gotCampaign.isEmpty()) {
            return null
        }
        return gotCampaign[0];
    }

    fun getCampaigns(campaignIds: Collection<String>): List<UacYdbCampaign> {
        val queryBuilder = select(
            CAMPAIGN.ID,
            CAMPAIGN.ADV_TYPE,
            CAMPAIGN.NAME,
            CAMPAIGN.STORE_URL,
            CAMPAIGN.APP_ID,
            CAMPAIGN.APP_TITLE,
            CAMPAIGN.REGIONS,
            CAMPAIGN.MINUS_REGIONS,
            CAMPAIGN.CONTENT_FLAGS,
            CAMPAIGN.STATE_REASONS,
            CAMPAIGN.TRACKING_URL,
            CAMPAIGN.IMPRESSION_URL,
            CAMPAIGN.TARGET_ID,
            CAMPAIGN.CPA,
            CAMPAIGN.WEEK_LIMIT,
            CAMPAIGN.STATUS,
            CAMPAIGN.TARGET_STATUS,
            CAMPAIGN.CREATED_AT,
            CAMPAIGN.UPDATED_AT,
            CAMPAIGN.STARTED_AT,
            CAMPAIGN.ACCOUNT_ID,
            CAMPAIGN.OPTIONS,
            CAMPAIGN.SKAD_NETWORK_ENABLED,
            CAMPAIGN.ADULT_CONTENT_ENABLED,
            CAMPAIGN.HYPERGEO_ID,
            CAMPAIGN.KEYWORDS,
            CAMPAIGN.MINUS_KEYWORDS,
            CAMPAIGN.SOCDEM,
            CAMPAIGN.DEVICE_TYPES,
            CAMPAIGN.GOALS,
            CAMPAIGN.COUNTERS,
            CAMPAIGN.PERMALINK_ID,
            CAMPAIGN.PHONE_ID,
            CAMPAIGN.CALLTRACKING_SETTINGS_ID,
            CAMPAIGN.TIME_TARGET,
            CAMPAIGN.STRATEGY,
            CAMPAIGN.RETARGETING_CONDITION,
            CAMPAIGN.VIDEOS_ARE_NON_SKIPPABLE,
            CAMPAIGN.BRAND_SURVEY_ID,
            CAMPAIGN.BRIEF_SYNCED,
            CAMPAIGN.SHOWS_FREQUENCY_LIMIT,
            CAMPAIGN.STRATEGY_PLATFORM,
            CAMPAIGN.IS_ECOM,
            CAMPAIGN.CRR,
            CAMPAIGN.FEED_ID,
            CAMPAIGN.FEED_FILTERS,
            CAMPAIGN.CPM_CAMPAIGN_MEASURERS,
            CAMPAIGN.CPM_ASSETS,
            CAMPAIGN.BRAND_SAFETY,
            CAMPAIGN.DISABLED_PLACES,
            CAMPAIGN.RELEVANCE_MATCH,
        )
            .from(CAMPAIGN)
            .where(CAMPAIGN.ID.`in`(campaignIds.toIdsLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        val resultCampaigns = mutableListOf<UacYdbCampaign>()
        while (result.next()) {
            resultCampaigns.add(getCampaignFromResultSet(result))
        }
        return resultCampaigns;
    }


    private fun getCampaignFromResultSet(result: ResultSetReaderWrapped): UacYdbCampaign {
        return UacYdbCampaign(
            id = result.getValueReader(CAMPAIGN.ID).uint64.toIdString(),
            name = result.getValueReader(CAMPAIGN.NAME).utf8,
            cpa = result.getValueReaderOrNull(CAMPAIGN.CPA)?.uint64?.let { UacYdbUtils.moneyFromDb(it) },
            weekLimit = result.getValueReader(CAMPAIGN.WEEK_LIMIT).uint64.let { UacYdbUtils.moneyFromDb(it) },
            regions = result.getValueReader(CAMPAIGN.REGIONS).json.let { fromJson(it) },
            minusRegions = result.getValueReaderOrNull(CAMPAIGN.MINUS_REGIONS)?.json?.let { fromJson(it) },
            storeUrl = result.getValueReader(CAMPAIGN.STORE_URL).utf8,
            appId = result.getValueReaderOrNull(CAMPAIGN.APP_ID)?.uint64?.toIdString(),
            targetId = result.getValueReaderOrNull(CAMPAIGN.TARGET_ID)?.uint64?.let { TargetType.fromId(it) },
            trackingUrl = result.getValueReaderOrNull(CAMPAIGN.TRACKING_URL)?.utf8,
            account = result.getValueReader(CAMPAIGN.ACCOUNT_ID).uint64.toIdString(),
            impressionUrl = result.getValueReaderOrNull(CAMPAIGN.IMPRESSION_URL)?.utf8,
            createdAt = result.getValueReader(CAMPAIGN.CREATED_AT).uint64.let { fromEpochSecond(it) },
            updatedAt = result.getValueReader(CAMPAIGN.UPDATED_AT).uint64.let { fromEpochSecond(it) },
            startedAt = result.getValueReaderOrNull(CAMPAIGN.STARTED_AT)?.uint64?.let { fromEpochSecond(it) },
            advType = result.getValueReader(CAMPAIGN.ADV_TYPE).uint32.toInt().let { AdvType.fromId(it) },
            targetStatus = result.getValueReader(CAMPAIGN.TARGET_STATUS).uint32.toInt().let { TargetStatus.fromId(it) },
            contentFlags = result.getValueReaderOrNull(CAMPAIGN.CONTENT_FLAGS)?.json?.let { fromJson(it) },
            options = result.getValueReaderOrNull(CAMPAIGN.OPTIONS)?.json?.let { fromJson(it) },
            skadNetworkEnabled = result.getValueReaderOrNull(CAMPAIGN.SKAD_NETWORK_ENABLED)?.bool,
            adultContentEnabled = result.getValueReaderOrNull(CAMPAIGN.ADULT_CONTENT_ENABLED)?.bool,
            hyperGeoId = result.getValueReaderOrNull(CAMPAIGN.HYPERGEO_ID)?.uint64,
            keywords = result.getValueReaderOrNull(CAMPAIGN.KEYWORDS)?.json?.let { fromJson(it) },
            minusKeywords = result.getValueReaderOrNull(CAMPAIGN.MINUS_KEYWORDS)?.json?.let { fromJson(it) },
            socdem = result.getValueReaderOrNull(CAMPAIGN.SOCDEM)?.json?.let { fromJson(it) },
            deviceTypes = result.getValueReaderOrNull(CAMPAIGN.DEVICE_TYPES)?.json?.let { fromJson(it) },
            inventoryTypes = null,
            goals = result.getValueReaderOrNull(CAMPAIGN.GOALS)?.json?.let { fromJson(it) },
            counters = result.getValueReaderOrNull(CAMPAIGN.COUNTERS)?.json?.let { fromJson(it) },
            permalinkId = result.getValueReaderOrNull(CAMPAIGN.PERMALINK_ID)?.uint64,
            phoneId = result.getValueReaderOrNull(CAMPAIGN.PHONE_ID)?.uint64,
            calltrackingSettingsId = result.getValueReaderOrNull(CAMPAIGN.CALLTRACKING_SETTINGS_ID)?.uint64,
            timeTarget = result.getValueReaderOrNull(CAMPAIGN.TIME_TARGET)?.json?.let { fromJson(it) },
            strategy = result.getValueReaderOrNull(CAMPAIGN.STRATEGY)?.json?.let { fromJson(it) },
            retargetingCondition = result.getValueReaderOrNull(CAMPAIGN.RETARGETING_CONDITION)?.json?.let { fromJson(it) },
            videosAreNonSkippable = result.getValueReaderOrNull(CAMPAIGN.VIDEOS_ARE_NON_SKIPPABLE)?.bool,
            brandSurveyId = result.getValueReaderOrNull(CAMPAIGN.BRAND_SURVEY_ID)?.utf8,
            showsFrequencyLimit = result.getValueReaderOrNull(CAMPAIGN.SHOWS_FREQUENCY_LIMIT)?.json?.let { fromJson(it) },
            briefSynced = result.getValueReaderOrNull(CAMPAIGN.BRIEF_SYNCED)?.bool,
            strategyPlatform = result.getValueReaderOrNull(CAMPAIGN.STRATEGY_PLATFORM)?.uint32?.let { UacStrategyPlatform.fromId(it.toInt()) },
            isEcom = result.getValueReaderOrNull(CAMPAIGN.IS_ECOM)?.bool,
            crr = result.getValueReaderOrNull(CAMPAIGN.CRR)?.uint32,
            feedId = result.getValueReaderOrNull(CAMPAIGN.FEED_ID)?.uint64,
            feedFilters = result.getValueReaderOrNull(CAMPAIGN.FEED_FILTERS)?.json?.let { fromJson(it) },
            cpmAssets = result.getValueReaderOrNull(CAMPAIGN.CPM_ASSETS)?.json?.let { fromJson(it) },
            campaignMeasurers = result.getValueReaderOrNull(CAMPAIGN.CPM_CAMPAIGN_MEASURERS)?.json?.let{ fromJson(it)},
            uacBrandsafety = result.getValueReaderOrNull(CAMPAIGN.BRAND_SAFETY)?.json?.let { fromJson(it) },
            uacDisabledPlaces = result.getValueReaderOrNull(CAMPAIGN.DISABLED_PLACES)?.json?.let { fromJson(it) },
            relevanceMatch = result.getValueReaderOrNull(CAMPAIGN.RELEVANCE_MATCH)?.json?.let { fromJson(it) },
            searchLift = null,
        )
    }

    fun addCampaign(campaign: UacYdbCampaign) {
        val insertValues = buildTempTable {
            value(CAMPAIGN.ID, campaign.id.toIdLong())
            value(CAMPAIGN.ADV_TYPE, campaign.advType.id)
            value(CAMPAIGN.NAME, campaign.name)
            value(CAMPAIGN.STORE_URL, campaign.storeUrl)
            value(CAMPAIGN.APP_ID, campaign.appId?.toIdLong())
            value(CAMPAIGN.REGIONS, toJson(campaign.regions))
            value(CAMPAIGN.MINUS_REGIONS, toJson(campaign.minusRegions))
            value(CAMPAIGN.TRACKING_URL, campaign.trackingUrl)
            value(CAMPAIGN.IMPRESSION_URL, campaign.impressionUrl)
            value(CAMPAIGN.TARGET_ID, campaign.targetId?.id)
            value(CAMPAIGN.CPA, campaign.cpa?.let { UacYdbUtils.moneyToDb(it) })
            value(CAMPAIGN.WEEK_LIMIT, campaign.weekLimit?.let { UacYdbUtils.moneyToDb(it) })
            value(CAMPAIGN.TARGET_STATUS, campaign.targetStatus.id)
            value(CAMPAIGN.CREATED_AT, toEpochSecond(campaign.createdAt))
            value(CAMPAIGN.UPDATED_AT, toEpochSecond(campaign.updatedAt))
            value(CAMPAIGN.STARTED_AT, campaign.startedAt?.let { toEpochSecond(it) })
            value(CAMPAIGN.ACCOUNT_ID, campaign.account.toIdLong())
            value(CAMPAIGN.OPTIONS, toJson(campaign.options))
            value(CAMPAIGN.SKAD_NETWORK_ENABLED, campaign.skadNetworkEnabled)
            value(CAMPAIGN.ADULT_CONTENT_ENABLED, campaign.adultContentEnabled)
            value(CAMPAIGN.HYPERGEO_ID, campaign.hyperGeoId)
            value(CAMPAIGN.KEYWORDS, toJson(campaign.keywords))
            value(CAMPAIGN.MINUS_KEYWORDS, toJson(campaign.minusKeywords))
            value(CAMPAIGN.SOCDEM, toJson(campaign.socdem))
            value(CAMPAIGN.DEVICE_TYPES, toJson(campaign.deviceTypes))
            value(CAMPAIGN.GOALS, toJson(campaign.goals))
            value(CAMPAIGN.COUNTERS, toJson(campaign.counters))
            value(CAMPAIGN.PERMALINK_ID, campaign.permalinkId)
            value(CAMPAIGN.PHONE_ID, campaign.phoneId)
            value(CAMPAIGN.CALLTRACKING_SETTINGS_ID, campaign.calltrackingSettingsId)
            value(CAMPAIGN.CONTENT_FLAGS, toJson(campaign.contentFlags))
            value(CAMPAIGN.TIME_TARGET, toJson(campaign.timeTarget))
            value(CAMPAIGN.STRATEGY, toJson(campaign.strategy))
            value(CAMPAIGN.RETARGETING_CONDITION, toJson(campaign.retargetingCondition))
            value(CAMPAIGN.VIDEOS_ARE_NON_SKIPPABLE, campaign.videosAreNonSkippable)
            value(CAMPAIGN.BRAND_SURVEY_ID, campaign.brandSurveyId)
            value(CAMPAIGN.BRIEF_SYNCED, campaign.briefSynced)
            value(CAMPAIGN.SHOWS_FREQUENCY_LIMIT, campaign.showsFrequencyLimit?.let { toJson(it) })
            value(CAMPAIGN.STRATEGY_PLATFORM, campaign.strategyPlatform?.id)
            value(CAMPAIGN.IS_ECOM, campaign.isEcom)
            value(CAMPAIGN.CRR, campaign.crr?.toInt())
            value(CAMPAIGN.FEED_ID, campaign.feedId)
            value(CAMPAIGN.FEED_FILTERS, campaign.feedFilters?.let { toJson(it) })
            value(CAMPAIGN.CPM_ASSETS, campaign.cpmAssets?.let { toJson(it) })
            value(CAMPAIGN.CPM_CAMPAIGN_MEASURERS, campaign.campaignMeasurers?.let { toJson(it) })
            value(CAMPAIGN.BRAND_SAFETY, campaign.uacBrandsafety?.let { toJson(it) })
            value(CAMPAIGN.DISABLED_PLACES, campaign.uacDisabledPlaces?.let { toJson(it) })
            value(CAMPAIGN.RELEVANCE_MATCH, campaign.relevanceMatch?.let { toJson(it) })
        }

        val queryAndParams = insertInto(CAMPAIGN)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateBriefSynced(campaignId: String, briefSynced: Boolean) {
        val queryBuilder = update(
            CAMPAIGN,
            set(CAMPAIGN.BRIEF_SYNCED, briefSynced)
        ).where(CAMPAIGN.ID.eq(campaignId.toIdLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateCampaignStatusAndTimes(
        campaignId: String,
        status: Status,
    ) {
        val builder: QueryBuilder = update(
            CAMPAIGN,
            set(CAMPAIGN.STATUS, status.id)
                .set(CAMPAIGN.STARTED_AT, toEpochSecond(now()))
                .set(CAMPAIGN.UPDATED_AT, toEpochSecond(now()))
        )
            .where(CAMPAIGN.ID.eq(campaignId.toIdLong()))
        val queryAndParams = builder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams)
    }

    fun updateCampaign(uacCampaign: UacYdbCampaign) {
        val queryBuilder = update(
            CAMPAIGN,
            set(CAMPAIGN.UPDATED_AT, toEpochSecond(now()))
                .set(CAMPAIGN.ADV_TYPE, uacCampaign.advType.id)
                .set(CAMPAIGN.NAME, uacCampaign.name)
                .set(CAMPAIGN.STORE_URL, uacCampaign.storeUrl)
                .set(CAMPAIGN.APP_ID, uacCampaign.appId?.toIdLong())
                .set(CAMPAIGN.REGIONS, toJson(uacCampaign.regions))
                .set(CAMPAIGN.MINUS_REGIONS, toJson(uacCampaign.minusRegions))
                .set(CAMPAIGN.TRACKING_URL, uacCampaign.trackingUrl)
                .set(CAMPAIGN.IMPRESSION_URL, uacCampaign.impressionUrl)
                .set(CAMPAIGN.TARGET_ID, uacCampaign.targetId?.id)
                .set(CAMPAIGN.CPA, uacCampaign.cpa?.let { UacYdbUtils.moneyToDb(it) })
                .set(CAMPAIGN.WEEK_LIMIT, uacCampaign.weekLimit?.let { UacYdbUtils.moneyToDb(it) })
                .set(CAMPAIGN.TARGET_STATUS, uacCampaign.targetStatus.id)
                .set(CAMPAIGN.ACCOUNT_ID, uacCampaign.account.toIdLong())
                .set(CAMPAIGN.SKAD_NETWORK_ENABLED, uacCampaign.skadNetworkEnabled)
                .set(CAMPAIGN.ADULT_CONTENT_ENABLED, uacCampaign.adultContentEnabled)
                .set(CAMPAIGN.OPTIONS, uacCampaign.options?.let { toJson(it) })
                .set(CAMPAIGN.HYPERGEO_ID, uacCampaign.hyperGeoId)
                .set(CAMPAIGN.KEYWORDS, toJson(uacCampaign.keywords))
                .set(CAMPAIGN.MINUS_KEYWORDS, toJson(uacCampaign.minusKeywords))
                .set(CAMPAIGN.SOCDEM, toJson(uacCampaign.socdem))
                .set(CAMPAIGN.DEVICE_TYPES, toJson(uacCampaign.deviceTypes))
                .set(CAMPAIGN.GOALS, toJson(uacCampaign.goals))
                .set(CAMPAIGN.COUNTERS, toJson(uacCampaign.counters))
                .set(CAMPAIGN.PERMALINK_ID, uacCampaign.permalinkId)
                .set(CAMPAIGN.PHONE_ID, uacCampaign.phoneId)
                .set(CAMPAIGN.CALLTRACKING_SETTINGS_ID, uacCampaign.calltrackingSettingsId)
                .set(CAMPAIGN.TIME_TARGET, toJson(uacCampaign.timeTarget))
                .set(CAMPAIGN.STRATEGY, uacCampaign.strategy?.let { toJson(it) })
                .set(CAMPAIGN.RETARGETING_CONDITION, uacCampaign.retargetingCondition?.let { toJson(it) })
                .set(CAMPAIGN.VIDEOS_ARE_NON_SKIPPABLE, uacCampaign.videosAreNonSkippable)
                .set(CAMPAIGN.BRAND_SURVEY_ID, uacCampaign.brandSurveyId)
                .set(CAMPAIGN.BRIEF_SYNCED, uacCampaign.briefSynced)
                .set(CAMPAIGN.SHOWS_FREQUENCY_LIMIT, uacCampaign.showsFrequencyLimit?.let { toJson(it) })
                .set(CAMPAIGN.STRATEGY_PLATFORM, uacCampaign.strategyPlatform?.id)
                .set(CAMPAIGN.IS_ECOM, uacCampaign.isEcom)
                .set(CAMPAIGN.CRR, uacCampaign.crr?.toInt())
                .set(CAMPAIGN.FEED_ID, uacCampaign.feedId)
                .set(CAMPAIGN.FEED_FILTERS, uacCampaign.feedFilters?.let { toJson(it) })
                .set(CAMPAIGN.CPM_ASSETS, uacCampaign.cpmAssets?.let { toJson(it) })
                .set(CAMPAIGN.CPM_CAMPAIGN_MEASURERS, uacCampaign.campaignMeasurers?.let { toJson(it) })
                .set(CAMPAIGN.BRAND_SAFETY, uacCampaign.uacBrandsafety?.let { toJson(it) })
                .set(CAMPAIGN.DISABLED_PLACES, uacCampaign.uacDisabledPlaces?.let { toJson(it) })
                .set(CAMPAIGN.RELEVANCE_MATCH, uacCampaign.relevanceMatch?.let { toJson(it) })
        ).where(CAMPAIGN.ID.eq(uacCampaign.id.toIdLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams)
    }

    fun update(ktModelChanges: KtModelChanges<String, UacYdbCampaign>) { //не обновляю id, accountId, advType
        var updateStatement: UpdateBuilder.SetStatement? = null

        updateStatement = updateStatement
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.UPDATED_AT, UacYdbCampaign::updatedAt) { toEpochSecond(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.CREATED_AT, UacYdbCampaign::createdAt) { toEpochSecond(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.STARTED_AT, UacYdbCampaign::startedAt) { toEpochSecond(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.NAME, UacYdbCampaign::name)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.STORE_URL, UacYdbCampaign::storeUrl)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.APP_ID, UacYdbCampaign::appId) { it?.toIdLong() }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.REGIONS, UacYdbCampaign::regions) { toJson(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.MINUS_REGIONS, UacYdbCampaign::minusRegions) { toJson(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.TRACKING_URL, UacYdbCampaign::trackingUrl)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.IMPRESSION_URL, UacYdbCampaign::impressionUrl)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.TARGET_ID, UacYdbCampaign::targetId) { it?.id }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.CPA, UacYdbCampaign::cpa) { c -> c?.let { UacYdbUtils.moneyToDb(it) } }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.WEEK_LIMIT, UacYdbCampaign::weekLimit) { c -> c?.let { UacYdbUtils.moneyToDb(it) } }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.TARGET_STATUS, UacYdbCampaign::targetStatus) { it.id }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.OPTIONS, UacYdbCampaign::options) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.SKAD_NETWORK_ENABLED, UacYdbCampaign::skadNetworkEnabled)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.ADULT_CONTENT_ENABLED, UacYdbCampaign::adultContentEnabled)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.KEYWORDS, UacYdbCampaign::keywords) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.MINUS_KEYWORDS, UacYdbCampaign::minusKeywords) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.BRIEF_SYNCED, UacYdbCampaign::briefSynced)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.STRATEGY_PLATFORM, UacYdbCampaign::strategyPlatform) { it?.id }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.STRATEGY, UacYdbCampaign::strategy) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.BRAND_SURVEY_ID, UacYdbCampaign::brandSurveyId)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.VIDEOS_ARE_NON_SKIPPABLE, UacYdbCampaign::videosAreNonSkippable)
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.RETARGETING_CONDITION, UacYdbCampaign::retargetingCondition) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.SHOWS_FREQUENCY_LIMIT, UacYdbCampaign::showsFrequencyLimit) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.CPM_ASSETS, UacYdbCampaign::cpmAssets) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.CPM_CAMPAIGN_MEASURERS, UacYdbCampaign::campaignMeasurers) { toJsonNullable(it) }
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.RELEVANCE_MATCH, UacYdbCampaign::relevanceMatch) { toJsonNullable(it) }

        if (updateStatement == null) {
            return
        }

        val query = update(CAMPAIGN, updateStatement)
            .where(CAMPAIGN.ID.eq(ktModelChanges.id.toIdLong()))
            .queryAndParams(path)

        ydbClient.executeQuery(query, true)
    }

    fun updateContentFlags(ktModelChanges: KtModelChanges<String, UacYdbCampaign>) {
        var updateStatement: UpdateBuilder.SetStatement? = null

        updateStatement = updateStatement
            .setFromKtModelChanges(ktModelChanges, CAMPAIGN.CONTENT_FLAGS, UacYdbCampaign::contentFlags, { toJson(it) })

        if (updateStatement == null) {
            return
        }
        val query = update(CAMPAIGN, updateStatement)
            .where(CAMPAIGN.ID.eq(ktModelChanges.id.toIdLong()))
            .queryAndParams(path)

        ydbClient.executeQuery(query, true)
    }

    fun updateContentFlagsForCampaignTest(campaignId: String, flags: Map<String, Any>) {
        val insertValues = buildTempTable {
            value(CAMPAIGN_TEST.ID, campaignId.toIdLong())
            value(CAMPAIGN_TEST.CONTENT_FLAGS, toJson(flags))
            newRecord()
        }

        val query = upsertInto(CAMPAIGN_TEST)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(query, true)
    }

    fun delete(id: String) {
        val queryAndParams = deleteFrom(CAMPAIGN).where(CAMPAIGN.ID.eq(id.toIdLong())).queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }
}
