package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.CONTENT
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.ContentTable
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.utils.JsonUtils.toJson
import ru.yandex.direct.utils.fromJson
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable

@Lazy
@Repository
class UacYdbContentRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath
) {
    fun getContents(contentIds: Collection<String>): List<UacYdbContent> {
        val queryBuilder = selectAllContentFields(CONTENT)
            .from(CONTENT)
            .where(CONTENT.ID.`in`(contentIds.map { it.toIdLong() }))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)
        val contents = mutableListOf<UacYdbContent>()
        while (result.next()) {
            val content = convertResultToYdbContent(result)
            contents.add(content)
        }
        return contents
    }

    fun getContentByHash(hash: String): UacYdbContent? {
        val table = CONTENT.withIndex(CONTENT.DIRECT_IMAGE_HASH_INDEX) as ContentTable
        val queryBuilder = selectAllContentFields(table)
            .from(table)
            .where(table.DIRECT_IMAGE_HASH.eq(hash))
            .limit(1)
        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        if (!result.next()) {
            return null
        }

        return convertResultToYdbContent(result)
    }

    fun convertResultToYdbContent(result: ResultSetReaderWrapped) = UacYdbContent(
        id = result.getValueReader(CONTENT.ID).uint64.toIdString(),
        ownerId = result.getValueReaderOrNull(CONTENT.OWNER_ID)?.uint64?.toIdString(),
        type = MediaType.fromId(result.getValueReader(CONTENT.TYPE).uint32.toInt()),
        thumb = result.getValueReader(CONTENT.THUMB).utf8,
        sourceUrl = result.getValueReader(CONTENT.SOURCE_URL).utf8,
        mdsUrl = result.getValueReaderOrNull(CONTENT.MDS_URL)?.utf8,
        meta = fromJson(result.getValueReader(CONTENT.META).json),
        videoDuration = result.getValueReaderOrNull(CONTENT.VIDEO_DURATION)?.uint32?.toInt(),
        filename = result.getValueReaderOrNull(CONTENT.FILENAME)?.utf8,
        accountId = result.getValueReader(CONTENT.ACCOUNT_ID)?.uint64?.toIdString(),
        directImageHash = result.getValueReader(CONTENT.DIRECT_IMAGE_HASH)?.utf8,
    )

    fun selectAllContentFields(table: ContentTable) = SelectBuilder.select(
        table.ID,
        table.OWNER_ID,
        table.TYPE,
        table.THUMB,
        table.SOURCE_URL,
        table.MDS_URL,
        table.META,
        table.VIDEO_DURATION,
        table.FILENAME,
        table.ACCOUNT_ID,
        table.DIRECT_IMAGE_HASH,
    )

    fun saveContents(contents: List<UacYdbContent>) {
        val insertValues = buildTempTable {
            for (content in contents) {
                value(CONTENT.ID, content.id.toIdLong())
                value(CONTENT.OWNER_ID, content.ownerId?.toIdLong())
                value(CONTENT.TYPE, content.type.id)
                value(CONTENT.THUMB, content.thumb)
                value(CONTENT.SOURCE_URL, content.sourceUrl)
                value(CONTENT.MDS_URL, content.mdsUrl)
                value(CONTENT.META, toJson(content.meta))
                value(CONTENT.VIDEO_DURATION, content.videoDuration)
                value(CONTENT.FILENAME, content.filename)
                value(CONTENT.ACCOUNT_ID, content.accountId?.toIdLong())
                value(CONTENT.DIRECT_IMAGE_HASH, content.directImageHash)
                newRecord()
            }
        }

        val queryAndParams = InsertBuilder.insertInto(CONTENT)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams)
    }

    fun updateMetaAndImageHash(ktModelChanges: Collection<KtModelChanges<String, UacYdbContent>>) {
        ktModelChanges.forEach { updateMetaAndImageHash(it) }
    }

    fun updateMetaAndImageHash(ktModelChanges: KtModelChanges<String, UacYdbContent>) {
        var updateStatement: UpdateBuilder.SetStatement? = null

        updateStatement = updateStatement
            .setFromKtModelChanges(ktModelChanges, CONTENT.DIRECT_IMAGE_HASH, UacYdbContent::directImageHash)
            .setFromKtModelChanges(ktModelChanges, CONTENT.META, UacYdbContent::meta) { toJson(it) }

        if (updateStatement == null) {
            return
        }
        val query = UpdateBuilder.update(CONTENT, updateStatement)
            .where(CONTENT.ID.eq(ktModelChanges.id.toIdLong()))
            .queryAndParams(path)

        ydbClient.executeQuery(query, true)
    }

    fun deleteContent(id: String) {
        val queryAndParams =
            deleteFrom(CONTENT).where(CONTENT.ID.eq(id.toIdLong())).queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }
}
