package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.model.direct_ad_group.DirectAdGroupStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAdGroup
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_AD_GROUP
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DirectAdGroupTable
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder

@Lazy
@Repository
class UacYdbDirectAdGroupRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {

    fun getDirectAdGroupsByCampaignId(uacCampaignId: String): List<UacYdbDirectAdGroup> {
        val adGroupTable = DIRECT_AD_GROUP.withIndex(DIRECT_AD_GROUP.DIRECT_CAMPAIGN_ID_INDEX) as DirectAdGroupTable
        val queryBuilder = select(
            adGroupTable.ID,
            adGroupTable.DIRECT_CAMPAIGN_ID,
            adGroupTable.DIRECT_AD_GROUP_ID,
            adGroupTable.STATUS,
        )
            .from(adGroupTable)
            .where(adGroupTable.DIRECT_CAMPAIGN_ID.eq(uacCampaignId.toIdLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)

        val reader = ydbClient.executeQuery(queryAndParams).getResultSet(0)

        val uacYdbDirectAdGroup = mutableListOf<UacYdbDirectAdGroup>()
        while (reader.next()) {
            uacYdbDirectAdGroup.add(fetchUacYdbDirectAdGroup(reader, adGroupTable))
        }
        return uacYdbDirectAdGroup
    }

    fun getDirectAdGroupByDirectAdGroupId(adGroupId: Long): UacYdbDirectAdGroup? {
        val adGroupTable = DIRECT_AD_GROUP.withIndex(DIRECT_AD_GROUP.DIRECT_AD_GROUP_ID_INDEX) as DirectAdGroupTable
        val queryBuilder = select(
            adGroupTable.ID,
            adGroupTable.DIRECT_CAMPAIGN_ID,
            adGroupTable.DIRECT_AD_GROUP_ID,
            adGroupTable.STATUS,
        )
            .from(adGroupTable)
            .where(adGroupTable.DIRECT_AD_GROUP_ID.eq(adGroupId))

        val queryAndParams = queryBuilder.queryAndParams(path)

        val reader = ydbClient.executeQuery(queryAndParams).getResultSet(0)
        if (!reader.next()) {
            return null
        }

        return fetchUacYdbDirectAdGroup(reader, adGroupTable)
    }

    private fun fetchUacYdbDirectAdGroup(reader: ResultSetReaderWrapped, table: DirectAdGroupTable): UacYdbDirectAdGroup {
        return UacYdbDirectAdGroup(
            id = reader.getValueReader(table.ID).uint64.toIdString(),
            status = DirectAdGroupStatus.fromId(reader.getValueReader(table.STATUS).uint32.toInt()),
            directAdGroupId = reader.getValueReader(table.DIRECT_AD_GROUP_ID).uint64,
            directCampaignId = reader.getValueReader(table.DIRECT_CAMPAIGN_ID).uint64.toIdString(),
        )
    }

    /**
     * Возвращает директовые id групп по их uac id
     */
    fun getDirectAdGroupIdByUacAdGroupId(
        uacAdGroupIds: Collection<String>,
    ): Map<String, Long> {
        val queryBuilder = select(
            DIRECT_AD_GROUP.DIRECT_AD_GROUP_ID,
            DIRECT_AD_GROUP.ID,
        )
            .from(DIRECT_AD_GROUP)
            .where(DIRECT_AD_GROUP.ID.`in`(uacAdGroupIds.map { it.toIdLong() }))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeQuery(queryAndParams).getResultSet(0)
        if (result.isEmpty) {
            return mapOf()
        }

        val resultMap = HashMap<String, Long>()
        while (result.next()) {
            val uacAdGroupId = result.getValueReader(DIRECT_AD_GROUP.ID).uint64.toIdString()
            val directAdGroupId = result.getValueReader(DIRECT_AD_GROUP.DIRECT_AD_GROUP_ID).uint64

            resultMap[uacAdGroupId] = directAdGroupId
        }
        return resultMap
    }

    fun saveDirectAdGroup(uacYdbDirectAdGroup: UacYdbDirectAdGroup) {
        val insertValues = TempTableBuilder.buildTempTable {
            value(DIRECT_AD_GROUP.ID, uacYdbDirectAdGroup.id.toIdLong())
            value(DIRECT_AD_GROUP.DIRECT_AD_GROUP_ID, uacYdbDirectAdGroup.directAdGroupId)
            value(DIRECT_AD_GROUP.DIRECT_CAMPAIGN_ID, uacYdbDirectAdGroup.directCampaignId.toIdLong())
            value(DIRECT_AD_GROUP.STATUS, uacYdbDirectAdGroup.status.id)
        }

        val queryAndParams = InsertBuilder.insertInto(DIRECT_AD_GROUP)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams)
    }

    fun delete(adGroupId: String) {
        val queryAndParams = deleteFrom(DIRECT_AD_GROUP).where(DIRECT_AD_GROUP.ID.eq(adGroupId.toIdLong())).queryAndParams(path)
        ydbClient.executeQuery(queryAndParams)
    }
}
