package ru.yandex.direct.core.entity.uac.repository.ydb

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.bs.export.model.AssetHashes
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.YDB_MAX_ROWS_COUNT
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_AD
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_AD_GROUP
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_CAMPAIGN
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DirectAdGroupTable
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DirectAdTable
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DirectCampaignTable
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.expression.AggregateExpression
import ru.yandex.direct.ydb.builder.predicate.Predicate
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.ExtendedWhereBuilder
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.JoinBuilder.JoinStatement.on
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.builder.querybuilder.UnionLike
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.exceptions.YdbExecutionQueryException
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder

@Lazy
@Repository
class UacYdbDirectAdRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {
    private val logger = LoggerFactory.getLogger(UacYdbDirectAdRepository::class.java)

    fun getByDirectAdGroupId(
        uacDirectAdGroupIds: Collection<String>,
        fromUacAdId: Long,
        limit: Long,
    ): List<UacYdbDirectAd> {
        if (uacDirectAdGroupIds.isEmpty()) {
            return emptyList()
        }
        val directAdGroupIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_DIRECT_AD_GROUP_INDEX) as DirectAdTable

        val queryBuilder = select(
            directAdGroupIndex.ID,
            directAdGroupIndex.STATUS,
            directAdGroupIndex.TITLE_CONTENT_ID,
            directAdGroupIndex.TEXT_CONTENT_ID,
            directAdGroupIndex.DIRECT_CONTENT_ID,
            directAdGroupIndex.DIRECT_AD_GROUP_ID,
            directAdGroupIndex.DIRECT_AD_ID,
            directAdGroupIndex.DIRECT_IMAGE_CONTENT_ID,
            directAdGroupIndex.DIRECT_VIDEO_CONTENT_ID,
            directAdGroupIndex.DIRECT_HTML5_CONTENT_ID,
        )
            .from(directAdGroupIndex)
            .where(directAdGroupIndex.DIRECT_AD_GROUP_ID.`in`(uacDirectAdGroupIds.toIdsLong())
                .and(directAdGroupIndex.ID.gt(fromUacAdId)))
            .orderBy(directAdGroupIndex.ID)
            .limit(limit)

        return fetchUacYdbDirectAds(queryBuilder)
    }

    fun getMinDirectBannerIdInCampaign(
        uacCampaignId: String,
    ): Long? {
        val directAdGroupTable = DIRECT_AD_GROUP.withIndex(DIRECT_AD_GROUP.DIRECT_CAMPAIGN_ID_INDEX) as DirectAdGroupTable
        val directAdTable = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_DIRECT_AD_GROUP_INDEX) as DirectAdTable
        val minDirectAdIdField = AggregateExpression.min(directAdTable.DIRECT_AD_ID).`as`("min_direct_ad_id")

        val queryBuilder = select(minDirectAdIdField)
            .from(directAdGroupTable)
            .join(directAdTable, on(directAdGroupTable.ID, directAdTable.DIRECT_AD_GROUP_ID))
            .where(directAdGroupTable.DIRECT_CAMPAIGN_ID.eq(uacCampaignId.toIdLong())
                .and(directAdTable.DIRECT_AD_ID.isNotNull)
                .and(directAdTable.STATUS.neq(DirectAdStatus.DELETED.id))
            )

        val queryAndParams = queryBuilder.queryAndParams(path)

        val reader = ydbClient.executeQuery(queryAndParams).getResultSet(0)
        if (!reader.next()) {
            return null
        }

        return reader.getValueReaderOrNull(minDirectAdIdField)?.uint64
    }

    fun getByDirectAdId(uacDirectAdIds: Collection<Long>): List<UacYdbDirectAd> {
        if (uacDirectAdIds.isEmpty()) {
            return emptyList()
        }

        val result = mutableListOf<UacYdbDirectAd>()
        uacDirectAdIds.chunked(YDB_MAX_ROWS_COUNT)
            .forEach { result.addAll(getByDirectAdIdChunked(it)) }
        return result.toList()
    }

    private fun getByDirectAdIdChunked(uacDirectAdIds: Collection<Long>): List<UacYdbDirectAd> {
        val directAdIdIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_DIRECT_AD_ID_INDEX) as DirectAdTable
        val queryBuilder = select(
            directAdIdIndex.ID,
            directAdIdIndex.STATUS,
            directAdIdIndex.TITLE_CONTENT_ID,
            directAdIdIndex.TEXT_CONTENT_ID,
            directAdIdIndex.DIRECT_CONTENT_ID,
            directAdIdIndex.DIRECT_AD_GROUP_ID,
            directAdIdIndex.DIRECT_AD_ID,
            directAdIdIndex.DIRECT_IMAGE_CONTENT_ID,
            directAdIdIndex.DIRECT_VIDEO_CONTENT_ID,
            directAdIdIndex.DIRECT_HTML5_CONTENT_ID,
        )
            .from(directAdIdIndex)
            .where(directAdIdIndex.DIRECT_AD_ID.`in`(uacDirectAdIds))

        return fetchUacYdbDirectAds(queryBuilder)
    }

    /**
     * Функция возвращает банеры с 4-мя компонентами (текст + заголовок + картинка + видео)
     * Аналог python функции get_ad_ids_with_contents_v2 из tablets.py
     */
    fun getNotDeletedDirectAdsWithoutAdGroupsByContentIds(
        titleIds: Collection<String>,
        textIds: Collection<String>,
        imageIds: Collection<String>,
        creativeIds: Collection<String>,
    ): List<UacYdbDirectAd> {
        val directAdTitleIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_TITLE_CONTENT_INDEX) as DirectAdTable
        val directAdTextIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_TEXT_CONTENT_INDEX) as DirectAdTable
        val directAdImageIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_IMAGE_CONTENT_INDEX) as DirectAdTable
        val directAdVideoIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_VIDEO_CONTENT_INDEX) as DirectAdTable
        val directAdHtml5Index = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_HTML5_CONTENT_INDEX) as DirectAdTable

        val result = mutableListOf<UacYdbDirectAd>()
        if (titleIds.isNotEmpty()) {
            var titleContentQuery = getNotDeletedDirectAdsWithNullAdGroupByContentIdsQuery(
                directAdTitleIndex,
                directAdTitleIndex.TITLE_CONTENT_ID.`in`(titleIds.distinct().toIdsLong())
            )
            result += fetchUacYdbDirectAds(titleContentQuery)
        }
        if (textIds.isNotEmpty()) {
            val textContentQuery = getNotDeletedDirectAdsWithNullAdGroupByContentIdsQuery(
                directAdTextIndex,
                directAdTextIndex.TEXT_CONTENT_ID.`in`(textIds.distinct().toIdsLong())
            )
            result += fetchUacYdbDirectAds(textContentQuery)
        }
        if (imageIds.isNotEmpty()) {
            val imageContentQuery = getNotDeletedDirectAdsWithNullAdGroupByContentIdsQuery(
                directAdImageIndex,
                directAdImageIndex.DIRECT_IMAGE_CONTENT_ID.`in`(imageIds.distinct().toIdsLong())
            )
            result += fetchUacYdbDirectAds(imageContentQuery)
        }
        if (creativeIds.isNotEmpty()) {
            val videoContentQuery = getNotDeletedDirectAdsWithNullAdGroupByContentIdsQuery(
                directAdVideoIndex,
                directAdVideoIndex.DIRECT_VIDEO_CONTENT_ID.`in`(creativeIds.distinct().toIdsLong())
            )
            result += fetchUacYdbDirectAds(videoContentQuery)
            val html5ContentQuery = getNotDeletedDirectAdsWithNullAdGroupByContentIdsQuery(
                directAdHtml5Index,
                directAdHtml5Index.DIRECT_HTML5_CONTENT_ID.`in`(creativeIds.distinct().toIdsLong())
            )
            result += fetchUacYdbDirectAds(html5ContentQuery)
        }

        return result.distinctBy { it.id }
    }

    private fun getNotDeletedDirectAdsWithNullAdGroupByContentIdsQuery(
        table: DirectAdTable,
        predicate: Predicate
    ): ExtendedWhereBuilder {
        return select(
            table.ID,
            table.STATUS,
            table.TITLE_CONTENT_ID,
            table.TEXT_CONTENT_ID,
            table.DIRECT_CONTENT_ID,
            table.DIRECT_AD_GROUP_ID,
            table.DIRECT_AD_ID,
            table.DIRECT_IMAGE_CONTENT_ID,
            table.DIRECT_VIDEO_CONTENT_ID,
            table.DIRECT_HTML5_CONTENT_ID,
        )
            .from(table)
            .where(predicate.and(table.STATUS.neq(DirectAdStatus.DELETED.id))
                .and(table.DIRECT_AD_GROUP_ID.isNull))
    }

    /**
     * Функция возвращает староформатные банеры с 3-мя компонентами (текст + заголовок + картинка/видео).
     * Необходим для удаления старых банеров при перегенерации
     * Аналог python функции get_ad_ids_with_contents_v1 из tablets.py
     */
    fun getOldNotDeletedDirectAdsByContentIds(
        imageIds: Collection<String>,
        videoIds: Collection<String>,
    ): List<UacYdbDirectAd> {
        val directAdContentIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_CONTENT_INDEX) as DirectAdTable
        val queryBuilder = select(
            directAdContentIndex.ID,
            directAdContentIndex.STATUS,
            directAdContentIndex.TITLE_CONTENT_ID,
            directAdContentIndex.TEXT_CONTENT_ID,
            directAdContentIndex.DIRECT_CONTENT_ID,
            directAdContentIndex.DIRECT_AD_GROUP_ID,
            directAdContentIndex.DIRECT_AD_ID,
            directAdContentIndex.DIRECT_IMAGE_CONTENT_ID,
            directAdContentIndex.DIRECT_VIDEO_CONTENT_ID,
            directAdContentIndex.DIRECT_HTML5_CONTENT_ID,
        )
            .from(directAdContentIndex)
            .where(directAdContentIndex.DIRECT_CONTENT_ID.`in`(imageIds.toIdsLong() + videoIds.toIdsLong())
                .and(directAdContentIndex.STATUS.neq(DirectAdStatus.DELETED.id)))

        return fetchUacYdbDirectAds(queryBuilder)
    }

    /**
     * Возвращает id ассетов по баннерам и переданным id кампаниям
     * Применяется в пагинации по adId
     * Возвращается сортированный по adId список
     */
    fun getContentIdsByDirectCampaignIds(
        directCampaignIds: Collection<Long>,
        fromAdId: Long,
        limit: Long,
    ): List<AssetHashes> {

        val directCampaignTable = DIRECT_CAMPAIGN
            .withIndex(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID_INDEX) as DirectCampaignTable
        val directAdGroupTable = DIRECT_AD_GROUP
            .withIndex(DIRECT_AD_GROUP.DIRECT_CAMPAIGN_ID_INDEX) as DirectAdGroupTable
        val directAdTable = DIRECT_AD
            .withIndex(DIRECT_AD.DIRECT_AD_DIRECT_AD_GROUP_INDEX) as DirectAdTable

        val queryBuilder = select(
            directCampaignTable.DIRECT_CAMPAIGN_ID,
            directAdTable.DIRECT_AD_ID,
            directAdTable.TITLE_CONTENT_ID,
            directAdTable.TEXT_CONTENT_ID,
            directAdTable.DIRECT_IMAGE_CONTENT_ID,
            directAdTable.DIRECT_VIDEO_CONTENT_ID,
            directAdTable.DIRECT_HTML5_CONTENT_ID,
        )
            .from(directCampaignTable)
            .join(directAdGroupTable,
                on(directAdGroupTable.DIRECT_CAMPAIGN_ID, directCampaignTable.ID))
            .join(directAdTable,
                on(directAdTable.DIRECT_AD_GROUP_ID, directAdGroupTable.ID))
            .where(directCampaignTable.DIRECT_CAMPAIGN_ID.`in`(directCampaignIds)
                .and(directAdTable.STATUS.neq(DirectAdStatus.DELETED.id))
                .and(directAdTable.DIRECT_AD_ID.gt(fromAdId)))
            .orderBy(directAdTable.DIRECT_AD_ID)
            .limit(limit)

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)
        val assetsData: MutableList<AssetHashes> = mutableListOf()
        while (result.next()) {

            val bannerId = result.getValueReader(directAdTable.DIRECT_AD_ID)?.uint64
                ?: throw IllegalArgumentException("bannerId must not be null")

            val assetHashes = AssetHashes(
                campaignId = result.getValueReader(directCampaignTable.DIRECT_CAMPAIGN_ID).uint64,
                bannerId = bannerId,
                titleContentId = result.getValueReaderOrNull(directAdTable.TITLE_CONTENT_ID)?.uint64?.toIdString(),
                textContentId = result.getValueReaderOrNull(directAdTable.TEXT_CONTENT_ID)?.uint64?.toIdString(),
                imageContentId = result.getValueReaderOrNull(directAdTable.DIRECT_IMAGE_CONTENT_ID)?.uint64?.toIdString(),
                videoContentId = result.getValueReaderOrNull(directAdTable.DIRECT_VIDEO_CONTENT_ID)?.uint64?.toIdString(),
                html5ContentId = result.getValueReaderOrNull(directAdTable.DIRECT_HTML5_CONTENT_ID)?.uint64?.toIdString(),
            )
            assetsData.add(assetHashes)
        }
        return assetsData
    }

    fun getCountAdsByAdGroupIds(
        uacAdGroupIds: Collection<String>,
    ): Map<String, Int> {
        val countField = AggregateExpression.count().`as`("cnt")
        val directAdIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_DIRECT_AD_GROUP_INDEX) as DirectAdTable

        val queryBuilder = select(
            directAdIndex.DIRECT_AD_GROUP_ID,
            countField,
        )
            .from(directAdIndex)
            .where(directAdIndex.DIRECT_AD_GROUP_ID.`in`(uacAdGroupIds.map { it.toIdLong() })
                .and(directAdIndex.STATUS.neq(DirectAdStatus.DELETED.id)))
            .groupBy(directAdIndex.DIRECT_AD_GROUP_ID)

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeQuery(queryAndParams).getResultSet(0)
        if (result.isEmpty) {
            return mapOf()
        }

        val resultMap = HashMap<String, Int>()
        while (result.next()) {
            val uacAdGroupId = result.getValueReader(directAdIndex.DIRECT_AD_GROUP_ID).uint64.toIdString()
            val adsCount = result.getValueReader(countField).uint64.toInt()

            resultMap[uacAdGroupId] = adsCount
        }
        return resultMap
    }

    /**
     * Подразумевается, что все колонки в запросе не имеют алиаса и совпадают с именованием столбцов таблицы direct_ad
     */
    private fun fetchUacYdbDirectAds(queryBuilder: QueryBuilder): List<UacYdbDirectAd> {
        val queryAndParams = queryBuilder.queryAndParams(path)
        val reader = try {
            ydbClient.executeQuery(queryAndParams).getResultSet(0)
        } catch (e: YdbExecutionQueryException) {
            logger.error("Error while executing query {}, params {}",
                queryAndParams.query, queryAndParams.params.toPb())
            throw e
        }
        val result: MutableList<UacYdbDirectAd> = ArrayList()

        while (reader.next()) {
            result.add(UacYdbDirectAd(
                id = reader.getValueReader(DIRECT_AD.ID).uint64.toIdString(),
                status = DirectAdStatus.fromId(reader.getValueReader(DIRECT_AD.STATUS).uint32.toInt()),
                titleContentId = reader.getValueReaderOrNull(DIRECT_AD.TITLE_CONTENT_ID)?.uint64?.toIdString(),
                textContentId = reader.getValueReaderOrNull(DIRECT_AD.TEXT_CONTENT_ID)?.uint64?.toIdString(),
                directContentId = reader.getValueReaderOrNull(DIRECT_AD.DIRECT_CONTENT_ID)?.uint64?.toIdString(),
                directAdGroupId = reader.getValueReaderOrNull(DIRECT_AD.DIRECT_AD_GROUP_ID)?.uint64?.toIdString(),
                directAdId = reader.getValueReaderOrNull(DIRECT_AD.DIRECT_AD_ID)?.uint64,
                directImageContentId = reader.getValueReaderOrNull(DIRECT_AD.DIRECT_IMAGE_CONTENT_ID)?.uint64?.toIdString(),
                directVideoContentId = reader.getValueReaderOrNull(DIRECT_AD.DIRECT_VIDEO_CONTENT_ID)?.uint64?.toIdString(),
                directHtml5ContentId = reader.getValueReaderOrNull(DIRECT_AD.DIRECT_HTML5_CONTENT_ID)?.uint64?.toIdString()
            ))
        }
        return result
    }

    private fun getAdIdsByDirectAdIds(directAdIds: Collection<Long>): List<Long> {
        val directAdIdIndex = DIRECT_AD.withIndex(DIRECT_AD.DIRECT_AD_DIRECT_AD_ID_INDEX) as DirectAdTable
        val idsQuery = select(directAdIdIndex.ID)
            .from(directAdIdIndex)
            .where(directAdIdIndex.DIRECT_AD_ID.`in`(directAdIds))
            .queryAndParams(path)

        val result = ydbClient.executeOnlineRoQuery(idsQuery, true)
            .getResultSet(0)

        val ids = mutableListOf<Long>()
        while (result.next()) {
            ids.add(result.getValueReader(directAdIdIndex.ID).uint64)
        }
        return ids
    }

    fun updateStatusByDirectAdIds(
        bannerIds: Collection<Long>,
        directAdStatus: DirectAdStatus,
    ) {
        bannerIds.chunked(YDB_MAX_ROWS_COUNT)
            .forEach {
                val ids = getAdIdsByDirectAdIds(it)
                val builder: QueryBuilder = UpdateBuilder
                    .update(DIRECT_AD, UpdateBuilder.set(DIRECT_AD.STATUS, directAdStatus.id))
                    .where(DIRECT_AD.ID.`in`(ids)
                        .and(DIRECT_AD.STATUS.neq(directAdStatus.id)))

                val queryAndParams = builder.queryAndParams(path)
                ydbClient.executeQuery(queryAndParams)
            }
    }

    fun saveDirectAd(uacYdbDirectAd: UacYdbDirectAd) {
        val insertValues = TempTableBuilder.buildTempTable {
            value(DIRECT_AD.ID, uacYdbDirectAd.id.toIdLong())
            value(DIRECT_AD.STATUS, uacYdbDirectAd.status.id)
            value(DIRECT_AD.TITLE_CONTENT_ID, uacYdbDirectAd.titleContentId?.toIdLong())
            value(DIRECT_AD.TEXT_CONTENT_ID, uacYdbDirectAd.textContentId?.toIdLong())
            value(DIRECT_AD.DIRECT_CONTENT_ID, uacYdbDirectAd.directContentId?.toIdLong())
            value(DIRECT_AD.DIRECT_AD_GROUP_ID, uacYdbDirectAd.directAdGroupId?.toIdLong())
            value(DIRECT_AD.DIRECT_AD_ID, uacYdbDirectAd.directAdId)
            value(DIRECT_AD.DIRECT_IMAGE_CONTENT_ID, uacYdbDirectAd.directImageContentId?.toIdLong())
            value(DIRECT_AD.DIRECT_VIDEO_CONTENT_ID, uacYdbDirectAd.directVideoContentId?.toIdLong())
            value(DIRECT_AD.DIRECT_HTML5_CONTENT_ID, uacYdbDirectAd.directHtml5ContentId?.toIdLong())
        }

        val queryAndParams = InsertBuilder.insertInto(DIRECT_AD)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams)
    }

    fun delete(ids: List<String>) {
        val queryAndParams = deleteFrom(DIRECT_AD).where(DIRECT_AD.ID.`in`(ids.toIdsLong())).queryAndParams(path)
        ydbClient.executeQuery(queryAndParams)

    }
}
