package ru.yandex.direct.core.entity.uac.repository.ydb


import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.model.DirectCampaignStatus
import ru.yandex.direct.core.entity.uac.repository.DirectCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.YDB_MAX_ROWS_COUNT
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_CAMPAIGN
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DirectCampaignTable
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.predicate.Predicate
import ru.yandex.direct.ydb.builder.querybuilder.DeleteBuilder.deleteFrom
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder.insertInto
import ru.yandex.direct.ydb.builder.querybuilder.QueryBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.set
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.update
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder.Companion.buildTempTable

@Lazy
@Repository
class UacYdbDirectCampaignRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) : DirectCampaignRepository {

    override fun getDirectCampaignByDirectCampaignId(directCampaignId: Long): UacYdbDirectCampaign? {
        return getDirectCampaign(directCampaignId = directCampaignId)
    }

    fun getDirectCampaignById(id: String): UacYdbDirectCampaign? {
        return getDirectCampaign(id = id.toIdLong())
    }

    fun getCampaignIdsByDirectCampaignIds(directCampaignIds: Collection<Long>): Map<Long, String> {
        val directCampaignIdToCampaignIdMap = mutableMapOf<Long, String>()
        directCampaignIds.chunked(YDB_MAX_ROWS_COUNT)
            .forEach { directCampaignIdToCampaignIdMap.putAll(getCampaignIdsByDirectCampaignIdsChunked(it)) }
        return directCampaignIdToCampaignIdMap.toMap()
    }

    private fun getCampaignIdsByDirectCampaignIdsChunked(directCampaignIds: Collection<Long>): Map<Long, String> {
        val directCampaignTable = DIRECT_CAMPAIGN.withIndex(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID_INDEX) as DirectCampaignTable
        val queryAndParams = select(directCampaignTable.DIRECT_CAMPAIGN_ID, directCampaignTable.ID)
            .from(directCampaignTable)
            .where(directCampaignTable.DIRECT_CAMPAIGN_ID.`in`(directCampaignIds.toSet()))
            .queryAndParams(path)
        val resultSet = ydbClient.executeQuery(queryAndParams).getResultSet(0)
        val directCampaignIdToCampaignIdMap = mutableMapOf<Long, String>()
        while (resultSet.next()) {
            val directCampaignId = resultSet.getValueReader(directCampaignTable.DIRECT_CAMPAIGN_ID).uint64
            val campaignId = resultSet.getValueReader(directCampaignTable.ID).uint64.toIdString()
            directCampaignIdToCampaignIdMap[directCampaignId] = campaignId
        }
        return directCampaignIdToCampaignIdMap

    }

    private fun getDirectCampaign(directCampaignId: Long? = null, id: Long? = null): UacYdbDirectCampaign? {
        require(directCampaignId != null || id != null) { "directCampaignId or id must not be null" }

        val tbl: DirectCampaignTable
        val predicate: Predicate
        if (id != null) {
            tbl = DIRECT_CAMPAIGN
            predicate = tbl.ID.eq(id)
        } else {
            tbl = DIRECT_CAMPAIGN.withIndex(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID_INDEX) as DirectCampaignTable
            predicate = tbl.DIRECT_CAMPAIGN_ID.eq(directCampaignId)
        }

        val queryBuilder = select(
            tbl.ID,
            tbl.DIRECT_CAMPAIGN_ID,
            tbl.STATUS,
            tbl.SYNCED_AT,
            tbl.REJECT_REASONS,
        )
            .from(tbl)
            .where(predicate)

        return fetchDirectCampaign(queryBuilder, tbl)
    }


    private fun fetchDirectCampaign(queryBuilder: QueryBuilder, tbl: DirectCampaignTable): UacYdbDirectCampaign? {
        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        if (!result.next()) {
            return null
        }

        return UacYdbDirectCampaign(
            id = result.getValueReader(tbl.ID).uint64.toIdString(),
            directCampaignId = result.getValueReader(tbl.DIRECT_CAMPAIGN_ID).uint64,
            status = DirectCampaignStatus.fromId(result.getValueReader(tbl.STATUS).uint32.toInt()),
            syncedAt = UacYdbUtils.fromEpochSecond(result.getValueReader(tbl.SYNCED_AT).uint64),
            rejectReasons = result.getValueReaderOrNull(tbl.REJECT_REASONS)?.jsonDocument,
        )
    }

    fun saveDirectCampaign(directCampaign: UacYdbDirectCampaign) {
        val insertValues = buildTempTable {
            value(DIRECT_CAMPAIGN.ID, directCampaign.id.toIdLong())
            value(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID, directCampaign.directCampaignId)
            value(DIRECT_CAMPAIGN.STATUS, directCampaign.status.id)
            value(DIRECT_CAMPAIGN.SYNCED_AT, UacYdbUtils.toEpochSecond(directCampaign.syncedAt))
            value(DIRECT_CAMPAIGN.REJECT_REASONS, directCampaign.rejectReasons)
        }

        val queryAndParams = insertInto(DIRECT_CAMPAIGN)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }

    fun update(ktModelChanges: KtModelChanges<String, UacYdbDirectCampaign>) {//не обновляю directCampaignId, id
        var updateStatement: UpdateBuilder.SetStatement? = null

        updateStatement = updateStatement
            .setFromKtModelChanges(ktModelChanges, DIRECT_CAMPAIGN.STATUS, UacYdbDirectCampaign::status) { it.id }
            .setFromKtModelChanges(
                ktModelChanges,
                DIRECT_CAMPAIGN.SYNCED_AT,
                UacYdbDirectCampaign::syncedAt
            ) { UacYdbUtils.toEpochSecond(it) }
            .setFromKtModelChanges(ktModelChanges, DIRECT_CAMPAIGN.REJECT_REASONS, UacYdbDirectCampaign::rejectReasons)

        if (updateStatement == null) {
            return
        }
        val query = update(DIRECT_CAMPAIGN, updateStatement)
            .where(DIRECT_CAMPAIGN.ID.eq(ktModelChanges.id.toIdLong()))
            .queryAndParams(path)

        ydbClient.executeQuery(query, true)
    }

    fun delete(id: String) {
        val queryAndParams =
            deleteFrom(DIRECT_CAMPAIGN).where(DIRECT_CAMPAIGN.ID.eq(id.toIdLong())).queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }

    fun updateDirectCampaignId(id: String, directCampaignId: Long) {
        val builder = update(
            DIRECT_CAMPAIGN,
            set(DIRECT_CAMPAIGN.DIRECT_CAMPAIGN_ID, directCampaignId)
        ).where(DIRECT_CAMPAIGN.ID.eq(id.toIdLong()))

        val queryAndParams = builder.queryAndParams(path)
        ydbClient.executeQuery(queryAndParams, true)
    }
}
