package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentStatus
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.getValueReaderOrNull
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectContent
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.DIRECT_CONTENT
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder.select
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.table.temptable.TempTableBuilder

@Lazy
@Repository
class UacYdbDirectContentRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {

    fun getDirectContentByIds(uacDirectContentIds: Collection<String>): List<UacYdbDirectContent> {
        val queryBuilder = select(
            DIRECT_CONTENT.ID,
            DIRECT_CONTENT.STATUS,
            DIRECT_CONTENT.TYPE,
            DIRECT_CONTENT.DIRECT_IMAGE_HASH,
            DIRECT_CONTENT.DIRECT_VIDEO_ID,
            DIRECT_CONTENT.DIRECT_HTML5_ID,
        )
            .from(DIRECT_CONTENT)
            .where(DIRECT_CONTENT.ID.`in`(uacDirectContentIds.toIdsLong()))

        val queryAndParams = queryBuilder.queryAndParams(path)

        val reader = ydbClient.executeQuery(queryAndParams).getResultSet(0)
        val result: MutableList<UacYdbDirectContent> = ArrayList<UacYdbDirectContent>()

        while (reader.next()) {
            result.add(UacYdbDirectContent(
                id = reader.getValueReader(DIRECT_CONTENT.ID).uint64.toIdString(),
                status = DirectContentStatus.fromId(reader.getValueReader(DIRECT_CONTENT.STATUS).uint32.toInt()),
                type = DirectContentType.fromId(reader.getValueReader(DIRECT_CONTENT.TYPE).uint32.toInt()),
                directImageHash = reader.getValueReaderOrNull(DIRECT_CONTENT.DIRECT_IMAGE_HASH)?.utf8,
                directVideoId = reader.getValueReaderOrNull(DIRECT_CONTENT.DIRECT_VIDEO_ID)?.uint64,
                directHtml5Id = reader.getValueReaderOrNull(DIRECT_CONTENT.DIRECT_HTML5_ID)?.uint64,
            ))
        }
        return result
    }

    fun addDirectContent(directContents: Collection<UacYdbDirectContent>) {
        val insertValues = TempTableBuilder.buildTempTable {
            directContents.forEach { directContent ->
                value(DIRECT_CONTENT.ID, directContent.id.toIdLong())
                value(DIRECT_CONTENT.STATUS, directContent.status.id)
                value(DIRECT_CONTENT.TYPE, directContent.type.id)
                value(DIRECT_CONTENT.DIRECT_IMAGE_HASH, directContent.directImageHash)
                value(DIRECT_CONTENT.DIRECT_VIDEO_ID, directContent.directVideoId)
                value(DIRECT_CONTENT.DIRECT_HTML5_ID, directContent.directHtml5Id)
                newRecord()
            }
        }

        val queryAndParams = InsertBuilder.insertInto(DIRECT_CONTENT)
            .selectAll()
            .from(insertValues)
            .queryAndParams(path)

        ydbClient.executeQuery(queryAndParams, true)
    }
}
