package ru.yandex.direct.core.entity.uac.repository.ydb

import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbRejectReason
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.REJECT_REASONS
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.RejectReasonsTable
import ru.yandex.direct.i18n.Language
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder
import ru.yandex.direct.ydb.client.YdbClient

@Lazy
@Repository
class UacYdbRejectReasonRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {
    companion object {
        val AVAILABLE_LANGUAGES = setOf(Language.RU, Language.EN, Language.TR)
    }

    fun getRejectReasons(diagIds: Set<Int>, lang: Language): List<UacYdbRejectReason> {
        if (diagIds.isNullOrEmpty()) {
            return listOf()
        }

        val langToFetch = if (!AVAILABLE_LANGUAGES.contains(lang)) Language.EN.langString else lang.langString
        val indexTable = REJECT_REASONS.withIndex(REJECT_REASONS.DIAG_ID_INDEX) as RejectReasonsTable
        val queryBuilder = SelectBuilder.select(
            indexTable.ID,
            indexTable.DIAG_ID,
            indexTable.TITLE,
            indexTable.DESCRIPTION,
            indexTable.LANG,
        )
            .from(indexTable)
            .where(indexTable.DIAG_ID.`in`(diagIds))

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        val rejectReasonsByDiagIdAndLang = mutableMapOf<Int, MutableMap<String, UacYdbRejectReason>>()
        while (result.next()) {
            val diagId = result.getValueReader(indexTable.DIAG_ID).uint32.toInt()
            val language = result.getValueReader(indexTable.LANG).utf8
            val reason = UacYdbRejectReason(
                id = result.getValueReader(indexTable.ID).uint64.toIdString(),
                diagId = diagId,
                title = result.getValueReader(indexTable.TITLE).utf8,
                description = result.getValueReader(indexTable.DESCRIPTION).utf8,
                lang = language,
            )
            val rejectReasonsByLang = rejectReasonsByDiagIdAndLang.getOrPut(diagId, ::mutableMapOf)
            rejectReasonsByLang[language] = reason
        }
        return diagIds
                .mapNotNull { rejectReasonsByDiagIdAndLang[it] }
                .mapNotNull { getReasonByLang(it, langToFetch) }
    }

    private fun getReasonByLang(
        rejectReasonsByLang: Map<String, UacYdbRejectReason>,
        lang: String
    ) = rejectReasonsByLang[lang]
            ?: rejectReasonsByLang[Language.EN.langString]
            ?: rejectReasonsByLang[Language.RU.langString]
}
