package ru.yandex.direct.core.entity.uac.repository.ydb

import java.time.Instant
import java.time.temporal.ChronoUnit
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.configuration.UacYdbConfiguration
import ru.yandex.direct.core.entity.uac.repository.ydb.model.CounterType
import ru.yandex.direct.core.entity.uac.repository.ydb.model.TrackerAppEvent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacTrackerUrlStat
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.TRACKER_URL_STAT
import ru.yandex.direct.core.entity.uac.repository.ydb.schema.TrackerUrlStatTable
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.builder.expression.AggregateExpression
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.column.Column

@Lazy
@Repository
class UacYdbTrackerUrlStatRepository(
    @Qualifier(UacYdbConfiguration.UAC_YDB_CLIENT_BEAN) var ydbClient: YdbClient,
    @Qualifier(UacYdbConfiguration.UAC_YDB_PATH_BEAN) var path: YdbPath,
) {
    companion object {
        val COLUMN_BY_TRACKER_APP_EVENT: Map<TrackerAppEvent, Column<Long>> = mapOf(
            TrackerAppEvent.ACHIEVED_LEVEL to TRACKER_URL_STAT.ACHIEVED_LEVEL,
            TrackerAppEvent.ADDED_PAYMENT_INFO to TRACKER_URL_STAT.ADDED_PAYMENT_INFO,
            TrackerAppEvent.ADDED_TO_CART to TRACKER_URL_STAT.ADDED_TO_CART,
            TrackerAppEvent.ADDED_TO_WISHLIST to TRACKER_URL_STAT.ADDED_TO_WISHLIST,
            TrackerAppEvent.APP_LAUNCHED to TRACKER_URL_STAT.APP_LAUNCHED,
            TrackerAppEvent.COMPLETED_REGISTRATION to TRACKER_URL_STAT.COMPLETED_REGISTRATION,
            TrackerAppEvent.COMPLETED_TUTORIAL to TRACKER_URL_STAT.COMPLETED_TUTORIAL,
            TrackerAppEvent.CUSTOM_EVENT to TRACKER_URL_STAT.CUSTOM_EVENT,
            TrackerAppEvent.EVENT_1 to TRACKER_URL_STAT.EVENT_1,
            TrackerAppEvent.EVENT_2 to TRACKER_URL_STAT.EVENT_2,
            TrackerAppEvent.EVENT_3 to TRACKER_URL_STAT.EVENT_3,
            TrackerAppEvent.INITIATED_CHECKOUT to TRACKER_URL_STAT.INITIATED_CHECKOUT,
            TrackerAppEvent.INSTALLED to TRACKER_URL_STAT.INSTALLS,
            TrackerAppEvent.PURCHASED to TRACKER_URL_STAT.PURCHASED,
            TrackerAppEvent.RATED to TRACKER_URL_STAT.RATED,
            TrackerAppEvent.REMOVED_FROM_CART to TRACKER_URL_STAT.REMOVED_FROM_CART,
            TrackerAppEvent.SEARCHED to TRACKER_URL_STAT.SEARCHED,
            TrackerAppEvent.SHARED to TRACKER_URL_STAT.SHARED,
            TrackerAppEvent.SPENT_CREDITS to TRACKER_URL_STAT.SPENT_CREDITS,
            TrackerAppEvent.SPENT_TIME_IN_APP to TRACKER_URL_STAT.SPENT_TIME_IN_APP,
            TrackerAppEvent.UNLOCKED_ACHIEVEMENT to TRACKER_URL_STAT.UNLOCKED_ACHIEVEMENT,
            TrackerAppEvent.VIEWED_CONTENT to TRACKER_URL_STAT.VIEWED_CONTENT
        )
    }

    fun getTrackerUrlStat(trackerUrl: String, showForDaysNumber: Int): UacTrackerUrlStat? {
        val timestampFrom: Long = Instant.now().minus(showForDaysNumber.toLong(), ChronoUnit.DAYS).epochSecond
        val indexTable = TRACKER_URL_STAT.withIndex(TRACKER_URL_STAT.TRACKER_URL_UPDATE_TIME_INDEX) as TrackerUrlStatTable
        val queryBuilder = SelectBuilder
            .select(
                indexTable.TRACKER_URL,
                AggregateExpression.max(indexTable.COUNTER_TYPE).`as`(indexTable.COUNTER_TYPE.name),
                AggregateExpression.sumLong(indexTable.HIT_COUNT).`as`(indexTable.HIT_COUNT.name),
                *indexTable.TRACKER_EVENT_COLUMNS.map {
                    AggregateExpression.sumLong(it).`as`(it.name)
                }.toTypedArray()
            )
            .from(indexTable)
            .where(indexTable.TRACKER_URL.eq(trackerUrl)
                .and(indexTable.UPDATE_TIME.gt(timestampFrom))
            )
            .groupBy(indexTable.TRACKER_URL)

        val queryAndParams = queryBuilder.queryAndParams(path)
        val result = ydbClient.executeOnlineRoQuery(queryAndParams, true).getResultSet(0)

        if (!result.next()) {
            return null
        }
        return UacTrackerUrlStat(
            trackerUrl = result.getValueReader(indexTable.TRACKER_URL).utf8,
            counterType = CounterType.fromId(result.getValueReader(indexTable.COUNTER_TYPE).uint64.toInt()),
            hitCount = result.getValueReader(indexTable.HIT_COUNT).uint64,
            conversionsByEvent = COLUMN_BY_TRACKER_APP_EVENT.mapValues {result.getValueReader(it.value).uint64}
        )
    }
}
