package ru.yandex.direct.core.entity.uac.repository.ydb

import at.orz.hash.CityHash
import com.yandex.ydb.table.result.ValueReader
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.tracing.util.TraceUtil
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.ydb.builder.expression.NamedExpression
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped
import ru.yandex.direct.ydb.column.Column
import java.math.BigDecimal
import java.math.RoundingMode
import java.time.LocalDateTime
import java.time.ZoneId
import kotlin.reflect.KProperty1

object UacYdbUtils {

    private val MONEY_MULTIPLIER = BigDecimal.valueOf(1000000)

    const val TITLE_MAX_LENGTH = 56
    const val TEXT_MAX_LENGTH = 81
    const val YDB_MAX_ROWS_COUNT = 1_000

    /**
     * Конвертирует epoch second в локальное время. Локальное время используется при чтении timestamp из mysql.
     * Для ydb делаем так-же
     */
    fun toEpochSecond(localDateTime: LocalDateTime): Long =
        localDateTime.atZone(ZoneId.systemDefault()).toEpochSecond()

    fun toEpochSecond(localDateTime: LocalDateTime?): Long? {
        if (localDateTime == null) {
            return null
        }

        return toEpochSecond(localDateTime)
    }

    /**
     * Конвертирует локальное время в epoch second
     */
    fun fromEpochSecond(epochSecond: Long): LocalDateTime =
        DateTimeUtils.fromEpochSeconds(epochSecond)

    fun moneyToDb(money: BigDecimal): Long =
        money.multiply(MONEY_MULTIPLIER).setScale(0, RoundingMode.HALF_UP).toLong()

    fun moneyFromDb(money: Long): BigDecimal =
        BigDecimal.valueOf(money).divide(MONEY_MULTIPLIER, 6, RoundingMode.HALF_UP)

    /**
     * Случайное число, которое может использоваться как id при сохранении объекта в ydb
     */
    fun generateUniqueRandomId(): String {
        // TODO Повторить то, что делают в uac ?
        // https://st.yandex-team.ru/RMP-421
        return TraceUtil.randomId().toIdString()
    }

    fun generateRandomIdLong(): Long {
        return generateUniqueRandomId().toIdLong()
    }

    /**
     * Преобразует строку, содержащую uint64 значение в Long (при переполнении Long возвращает отрицательное число)
     */
    fun String.toIdLong(): Long {
        return this.toULong().toLong()
    }

    fun Collection<String>.toIdsLong(): Collection<Long> {
        return map {
            it.toIdLong()
        }
    }

    fun Long.toIdString(): String {
        return this.toULong().toString()
    }

    inline fun <reified T> ResultSetReaderWrapped.getValueReaderOrNull(column: Column<T?>): ValueReader? {
        val valueReader = this.getValueReader(column)
        return if (valueReader.isOptionalItemPresent) {
            valueReader
        } else null
    }

    inline fun <reified T> ResultSetReaderWrapped.getValueReaderOrNull(expression: NamedExpression<T?>): ValueReader? {
        val valueReader = this.getValueReader(expression)
        return if (valueReader.isOptionalItemPresent) {
            valueReader
        } else null
    }

    fun cityHash64(vararg args: Any): String {
        val bytes = args.joinToString("\t") { it.toString() }
            .encodeToByteArray()
        val hash = CityHash.cityHash64(bytes, 0, bytes.size)
        return hash.toIdString()
    }

    fun MobileAppStoreUrl.toId() =
        cityHash64(
            this.storeContentId,
            this.storeCountry.lowercase(),
            this.storeLanguage.lowercase(),
            Platform.fromOsType(this.osType).id,
            this.store.id,
        ).toIdLong()
}

private fun <T> UpdateBuilder.SetStatement?.set(
    column: Column<T>,
    valueSupplier: () -> T,
    condition: Boolean
): UpdateBuilder.SetStatement? {
    if (!condition) {
        return this
    }
    if (this == null) {
        return UpdateBuilder.set(column, valueSupplier.invoke())
    } else {
        return set(column, valueSupplier.invoke())
    }
}

fun <V, D> UpdateBuilder.SetStatement?.setFromKtModelChanges(
    ktModelChanges: KtModelChanges<*, D>,
    column: Column<V>,
    prop: KProperty1<D, V>
): UpdateBuilder.SetStatement? =
    setFromKtModelChanges(ktModelChanges, column, prop) { it }

fun <V, T, D> UpdateBuilder.SetStatement?.setFromKtModelChanges(
    ktModelChanges: KtModelChanges<*, D>,
    column: Column<T>,
    prop: KProperty1<D, V>,
    mapper: (V) -> T
): UpdateBuilder.SetStatement? =
    set(
        column,
        { mapper.invoke(ktModelChanges.getVal(prop)) },
        ktModelChanges.isPresent(prop)
    )
