package ru.yandex.direct.core.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.audience.client.YaAudienceClient
import ru.yandex.direct.audience.client.model.AudienceSegment
import ru.yandex.direct.audience.client.model.SegmentStatus
import ru.yandex.direct.core.entity.banner.service.BannerSuspendResumeService
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService
import ru.yandex.direct.core.entity.uac.model.CheckAudienceSegmentJobParams
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoalType
import ru.yandex.direct.core.entity.uac.model.UpdateAdsJobParams
import ru.yandex.direct.core.entity.uac.repository.mysql.BannerStatusesRepository
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper

@Service
class AudienceSegmentsService(
    private val yaAudienceClient: YaAudienceClient,
    private val shardHelper: ShardHelper,
    private val dbQueueRepository: DbQueueRepository,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(AudienceSegmentsService::class.java)
        val IS_PROCESSED_STATUSES = listOf(SegmentStatus.IS_PROCESSED, SegmentStatus.IS_UPDATED, SegmentStatus.UPLOADED)
    }

    fun getAudienceSegmentIdsFromRetargetingCondition(retargetingCondition: UacRetargetingCondition?): List<Long> {
        return retargetingCondition?.conditionRules?.flatMap {
            it.goals.filter { goal ->
                goal.type in listOf(
                    UacRetargetingConditionRuleGoalType.LAL_AUDIENCE, UacRetargetingConditionRuleGoalType.AUDIENCE
                )
            }.map { goal -> goal.id }
        }?.filterNotNull() ?: listOf()
    }

    fun getRetargetingSegmentsByIds(clientLogin: String, segmentIds: List<Long>): List<AudienceSegment> {
        return yaAudienceClient.getSegments(clientLogin)
            .filter { it.id in segmentIds }
    }

    fun getRetargetingSegmentsByRetargetingCondition(
        retargetingCondition: UacRetargetingCondition?,
        clientLogin: String
    ): List<AudienceSegment> {
        val auditionSegmentIds = getAudienceSegmentIdsFromRetargetingCondition(retargetingCondition)
        return getRetargetingSegmentsByIds(clientLogin, auditionSegmentIds)
    }

    fun hasFailedSegments(segments: List<AudienceSegment>): Boolean {
        return segments.any { it.status in listOf(SegmentStatus.PROCESSING_FAILED, SegmentStatus.FEW_DATA) }
    }

    fun hasProcessedSegments(segments: List<AudienceSegment>): Boolean {
        return segments.any { it.status in IS_PROCESSED_STATUSES }
    }

    fun hasProcessedSegments(retargetingCondition: UacRetargetingCondition?, clientLogin: String): Boolean {
        val audienceSegments = getRetargetingSegmentsByRetargetingCondition(retargetingCondition, clientLogin)
        return hasProcessedSegments(audienceSegments)
    }

    fun checkAudienceSegmentsDeferred(
        clientId: ClientId,
        operatorUid: Long,
        filledCampaignId: String
    ) {
        val shard = shardHelper.getShardByClientId(clientId)
        val jobParams = CheckAudienceSegmentJobParams(filledCampaignId)
        val jobType = DbQueueJobTypes.UAC_CHECK_AUDITION_SEGMENTS
        val jobId = dbQueueRepository.insertJob(shard, jobType, clientId, operatorUid, jobParams).id
        logger.info("Added job with id $jobId")
    }
}
