package ru.yandex.direct.core.entity.uac.service

import java.time.Duration
import org.slf4j.LoggerFactory
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.UAC_MODERATING_STATUS_FOR_ASSETS_WITHOUT_BANNERS
import ru.yandex.direct.core.entity.banner.model.BannerFlags
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType.BODY
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType.TITLE
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType
import ru.yandex.direct.core.entity.uac.model.direct_ad.BannerModerationSubject
import ru.yandex.direct.core.entity.uac.repository.mysql.BannerStatusesInfo
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.grut.objects.proto.AssetLink.TAssetLink
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinkStatus.EAssetLinkStatus
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_ACTIVE
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_CREATED
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_MODERATING
import ru.yandex.grut.objects.proto.AssetLink.TAssetLinkStatus.EAssetLinkStatus.ALS_REJECTED
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.MediaType.EMediaType
import ru.yandex.grut.objects.proto.client.Schema.TAssetMeta
import kotlin.math.max

data class CampaignUpdateStatusesContainer(
    val contentFlags: Map<String, String>,
    val assetLinkStatusByAssetLinkId: Map<String, EAssetLinkStatus>,
    val assetLinkRejectReasonsByAssetLinkId: Map<String, List<ModerationDiag>>,
    val assetLinkIdToAssetLink: Map<String, TAssetLink>,
)

data class BannerWithAssets(
    val bannerId: Long,
    val banner: Banner.TBannerSpec,
    val assets: List<TAssetMeta>,
    val assetLinkIds: List<String>,
)

abstract class BaseCampaignContentUpdateService(
    private val uacCampaignContentService: UacCampaignContentService,
    val featureService: FeatureService,
    ppcPropertiesSupport: PpcPropertiesSupport
) {
    companion object {
        private val logger = LoggerFactory.getLogger(CampaignContentUpdateService::class.java)
        private var AGE_FLAG = BannerFlags.AGE.key
        private var BANNER_REASONS_WITH_ASSETS_REASONS = setOf(4L, 497L, 498L, 499L)
    }

    private val moderatingInsteadOfRejectedProperty = ppcPropertiesSupport.get(UAC_MODERATING_STATUS_FOR_ASSETS_WITHOUT_BANNERS, Duration.ofSeconds(60))
    fun calculateCampaignContentsAndCampaignFlagsInternal(
        assetLinks: List<TAssetLink>,
        banners: Collection<BannerWithAssets>,
        bannerStatusesInfoList: Collection<BannerStatusesInfo>,
        rejectReasonDiags: Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>>,
        assetsRejectReasonDiags: Map<Long, Map<BannerAssetType, List<ModerationDiag>>>,
    ): CampaignUpdateStatusesContainer {
        val assetLinkIdToAssetLink = assetLinks.associateBy { (if (it.hasId()) it.id else it.assetId).toIdString() }
        val contentFlagsToUpdate = mutableMapOf<String, String>()
        val assetLinkIdToStatuses = mutableMapOf<String, MutableSet<EAssetLinkStatus>>()

        val bannersByBannerId = banners.associateBy { it.bannerId }
        bannerStatusesInfoList.forEach forEachBannerIds@{ bannerStatusesInfo ->

            var hasActiveStatus = true
            val banner = bannersByBannerId[bannerStatusesInfo.bannerId]!!
            val assetByAssetId = banner.assets.associateBy { it.id }
            banner.assetLinkIds.forEach forEachAssetLinkId@ { assetLinkId ->
                val assetLink = assetLinkIdToAssetLink[assetLinkId]
                if (assetLink == null) {
                    logger.error("Asset link with id $assetLinkId not found")
                    return@forEachAssetLinkId
                }
                val assetId = assetLink.assetId
                val asset = assetByAssetId[assetId]
                if (asset == null) {
                    logger.error("Asset with id $assetId not found")
                    return@forEachAssetLinkId
                }
                val status = convertToAssetLinkStatus(bannerStatusesInfo, asset)

                hasActiveStatus = hasActiveStatus && status == ALS_ACTIVE
                assetLinkIdToStatuses.getOrPut(assetLinkId, ::mutableSetOf).add(status)
            }

            // Собираем все флаги для кампании, в одной мапе
            if (hasActiveStatus) {
                if (bannerStatusesInfo.flags == null) {
                    return@forEachBannerIds
                }
                val finalAge = contentFlagsToUpdate.getOrDefault(AGE_FLAG, "0").toInt()
                contentFlagsToUpdate += bannerStatusesInfo.flags.flags

                // Для age флага ставим максимальное значение
                val age = bannerStatusesInfo.flags.flags[AGE_FLAG]
                if (!age.isNullOrBlank()) {
                    contentFlagsToUpdate[AGE_FLAG] = max(finalAge, age.toInt()).toString()
                }
            }
        }

        // Обновляем причины отклонения reject_reasons
        val assetLinkIdToRejectReasons = calculateRejectReasons(
            assetLinkIdToAssetLink,
            bannersByBannerId,
            rejectReasonDiags,
            assetsRejectReasonDiags
        )

        // Список campaignContent, у которых нужно будет обновить статусы
        val assetLinkIdToStatus = assetLinkIdToAssetLink.mapValues {
            getAssetLinkStatus(it.value, assetLinkIdToStatuses[it.key])
        }

        return CampaignUpdateStatusesContainer(
            contentFlags = contentFlagsToUpdate,
            assetLinkRejectReasonsByAssetLinkId = assetLinkIdToRejectReasons,
            assetLinkStatusByAssetLinkId = assetLinkIdToStatus,
            assetLinkIdToAssetLink = assetLinkIdToAssetLink,
        )
    }

    /**
     * Конвертирует статус контента кампании в campaign_content по переданному типу
     */
    private fun convertToAssetLinkStatus(
        bannerStatusesInfo: BannerStatusesInfo,
        asset: TAssetMeta,
    ): EAssetLinkStatus {
        val mediaType = asset.mediaType

        return when (UacCampaignContentService.BANNER_MODERATION_SUBJECT_BY_GRUT_MEDIA_TYPE[mediaType]) {
            BannerModerationSubject.TEXT -> uacCampaignContentService
                .toBannerStatus(bannerStatusesInfo.statusModerate)
            BannerModerationSubject.IMAGE -> uacCampaignContentService
                .toBannerImageStatus(bannerStatusesInfo.imageStatusModerate)
            BannerModerationSubject.CREATIVE -> uacCampaignContentService
                .toBannerCreativeStatus(bannerStatusesInfo.creativeStatusModerate)
            BannerModerationSubject.SITELINKS_SET -> uacCampaignContentService
                .toBannerCreativeStatus(bannerStatusesInfo.siteLinksSetStatusModerate)
            else -> ALS_CREATED
        }
    }

    private fun getAssetLinkStatus(
        assetLink: TAssetLink,
        statuses: Set<EAssetLinkStatus>?
    ): EAssetLinkStatus = when {
        assetLink.hasRemoveTime() -> {
            EAssetLinkStatus.ALS_DELETED
        }
        statuses.isNullOrEmpty() -> {
            if (moderatingInsteadOfRejectedProperty.getOrDefault(false)) ALS_MODERATING
            else ALS_REJECTED
        }
        else -> when {
            statuses.contains(ALS_ACTIVE) -> ALS_ACTIVE
            statuses.contains(ALS_MODERATING) -> ALS_MODERATING
            statuses.contains(ALS_CREATED) -> ALS_CREATED
            else -> ALS_REJECTED
        }
    }

    /**
     * Обновляем причины отклонения у campaignContent
     */
    fun calculateRejectReasons(
        assetLinkIdToAssetLink: Map<String, TAssetLink>,
        bannersByBannerId: Map<Long, BannerWithAssets>,
        bannerIdToModerationDiags: Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>>,
        bannerIdToAssetModerationDiags: Map<Long, Map<BannerAssetType, List<ModerationDiag>>>,
    ): Map<String, List<ModerationDiag>> {
        val assetLinkIdToDiagId = mutableMapOf<String, MutableSet<Long>>()
        val assetLinkIdToTokens = mutableMapOf<String, MutableSet<String>>()
        val assetLinkIdToRejectReasons = mutableMapOf<String, MutableList<ModerationDiag>>()
        bannersByBannerId.values.forEach forEachBannerIds@{ bannerWithAssets ->
            val bannerAssets = bannerWithAssets.assets
            val assetModerationDiagsByType = bannerIdToAssetModerationDiags[bannerWithAssets.bannerId]
            bannerIdToModerationDiags[bannerWithAssets.bannerId]?.forEach { (moderationReasonObjectType, moderationDiags) ->
                val assetById = getAssetsForModerationReasonObjectType(bannerAssets, moderationReasonObjectType).associateBy { it.id }
                val assetIds = assetById.keys
                val assetLinkIds = assetLinkIdToAssetLink
                    .asSequence()
                    .filter { assetIds.contains(it.value.assetId) }
                    .map { it.key }
                    .toList()
                moderationDiags.forEach { moderationDiag ->
                    val diagId = moderationDiag.id
                    val token = moderationDiag.token

                    // если вердикт отклонения баннера может иметь под собой ассетные вердикты
                    // и при этом ассетные вердикты на баннер пришли
                    val useAssetRejectReasons = BANNER_REASONS_WITH_ASSETS_REASONS.contains(diagId)
                        && !assetModerationDiagsByType.isNullOrEmpty()

                    // Собираем причины отклонения на модерации по id контента
                    assetLinkIds.forEach forEachAsset@{ assetLinkId ->
                        val assetId = assetLinkIdToAssetLink[assetLinkId]!!.assetId
                        val asset = assetById[assetId]!!

                        if (useAssetRejectReasons) {
                            val bannerAssetType = toBannerAssetType(asset.mediaType)
                            if (bannerAssetType != null) {
                                val assetModerationDiags = assetModerationDiagsByType!![bannerAssetType]
                                // если есть ассетные вердикты на текущий тип ассета
                                if (!assetModerationDiags.isNullOrEmpty()) {
                                    for (assetDiag in assetModerationDiags) {
                                        if (assetLinkIdToDiagId[assetLinkId]?.contains(assetDiag.id) == true) {
                                            continue
                                        }
                                        assetLinkIdToRejectReasons.computeIfAbsent(assetLinkId) { mutableListOf() }
                                            .add(assetDiag)
                                        assetLinkIdToDiagId.computeIfAbsent(assetLinkId) { mutableSetOf() }
                                            .add(assetDiag.id)
                                    }
                                }
                            }
                            return@forEachAsset
                        }

                        if (assetLinkIdToDiagId[assetLinkId]?.contains(diagId) == true
                            || assetLinkIdToTokens[assetLinkId]?.contains(token) == true
                        ) {
                            return@forEachAsset
                        }

                        assetLinkIdToDiagId.computeIfAbsent(assetLinkId) { mutableSetOf() }
                            .add(diagId)
                        if (token != null) {
                            assetLinkIdToTokens.computeIfAbsent(assetLinkId) { mutableSetOf() }
                                .add(token)
                        }

                        assetLinkIdToRejectReasons.computeIfAbsent(assetLinkId) { mutableListOf() }
                            .add(moderationDiag)
                    }
                }
            }
        }
        return assetLinkIdToAssetLink.keys.associateWith { assetLinkIdToRejectReasons.getOrDefault(it, listOf()) }
    }

    private fun toBannerAssetType(
        mediaType: EMediaType
    ): BannerAssetType? = when (mediaType) {
        EMediaType.MT_TITLE -> TITLE
        EMediaType.MT_TEXT -> BODY
        else -> null
    }

    private fun getAssetsForModerationReasonObjectType(
        assets: Collection<TAssetMeta>,
        moderationReasonObjectType: ModerationReasonObjectType,
    ): List<TAssetMeta> {
        val assetsByType = assets.groupBy { it.mediaType }
        return when (moderationReasonObjectType) {
            ModerationReasonObjectType.BANNER -> {
                //В модерации причины отклоеняни для видео/html5 и прочих креативов могут приходить на баннере (в новом транспорте это так)
                //assetsByType.values.toList().flatten().filter { it.id != null } ?: listOf()

                listOfNotNull(
                    assetsByType[EMediaType.MT_TITLE],
                    assetsByType[EMediaType.MT_TEXT],
                    assetsByType[EMediaType.MT_VIDEO],
                    assetsByType[EMediaType.MT_HTML5]
                )
                    .flatten()
                    .filter { it.id != null } ?: listOf()
            }
            ModerationReasonObjectType.IMAGE -> {
                assetsByType[EMediaType.MT_IMAGE]?.filter { it.id != null } ?: listOf()
            }
            ModerationReasonObjectType.VIDEO_ADDITION -> {
                assetsByType[EMediaType.MT_VIDEO]?.filter { it.id != null } ?: listOf()
            }
            ModerationReasonObjectType.SITELINKS_SET -> {
                assetsByType[EMediaType.MT_SITELINK]?.filter { it.id != null } ?: listOf()
            }
            ModerationReasonObjectType.HTML5_CREATIVE -> {
                assetsByType[EMediaType.MT_HTML5]?.filter { it.id != null } ?: listOf()
            }
            else -> emptyList()
        }
    }
}
