package ru.yandex.direct.core.entity.uac.service

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory
import ru.yandex.direct.core.entity.uac.converter.UacCampaignConverter.toCampaignStatuses
import ru.yandex.direct.core.entity.uac.model.BaseUacTrackingInfo
import ru.yandex.direct.core.entity.uac.model.CampaignStatuses
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.DirectCampaignStatus
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.result.Result
import ru.yandex.grut.objects.proto.client.Schema.TCampaign

data class FetchedContents(
    val mediaContents: List<Content>,
    val texts: List<String>?,
    val titles: List<String>?,
    val sitelinks: List<Sitelink>?,
)

abstract class BaseUacCampaignService(
    private val rmpCampaignService: RmpCampaignService,
    private val campaignService: CampaignService,
    private val campaignSubObjectAccessCheckerFactory: CampaignSubObjectAccessCheckerFactory,
    private val uacClientService: BaseUacClientService,
) : IUacCampaignService {

    companion object {
        private val logger = LoggerFactory.getLogger(BaseUacCampaignService::class.java)
    }

    abstract fun getLogger(): Logger

    fun getOrCreateClient(operator: User, subjectUser: User): String {
        return uacClientService.getOrCreateClient(operator = operator, subjectUser = subjectUser)
    }

    fun checkVisibleCampaign(operatorUid: Long, clientId: ClientId, directCampaignId: Long): Boolean {
        return campaignSubObjectAccessCheckerFactory
            .newCampaignChecker(operatorUid, clientId, listOf(directCampaignId))
            .objectInVisibleCampaign(directCampaignId)
    }

    fun checkWritableCampaign(operatorUid: Long, clientId: ClientId, directCampaignId: Long): Boolean {
        return campaignSubObjectAccessCheckerFactory
            .newCampaignChecker(operatorUid, clientId, listOf(directCampaignId))
            .objectInWritableAndEditableCampaign(directCampaignId)
    }

    fun getCampaignStatuses(
        clientId: ClientId,
        directCampaignId: Long,
        uacYdbCampaign: UacYdbCampaign? = null
    ): CampaignStatuses? {
        return rmpCampaignService.getCampaignStates(clientId, directCampaignId, true)
            ?.toCampaignStatuses(uacYdbCampaign?.isDraft ?: true)
    }

    abstract fun getDirectCampaignIdById(id: String): Long?

    abstract fun getCampaignProtoByDirectCampaignId(directCampaignId: Long): TCampaign?

    abstract fun getCampaignIdByDirectCampaignId(directCampaignId: Long): String?

    fun getCampaignByDirectCampaignId(directCampaignId: Long): UacYdbCampaign? =
        getCampaignIdByDirectCampaignId(directCampaignId)
            ?.let { getCampaignById(it) }

    abstract fun getMinBannerIdForCampaign(id: String): Long?

    abstract fun getDirectCampaignStatus(ydbCampaign: UacYdbCampaign): DirectCampaignStatus?

    abstract fun getAppIdsByAccountId(accountId: String): List<String>

    abstract fun getTextContentsByMediaType(campaingId: Long): Map<MediaType, List<String>>

    abstract fun getMediaCampaignContentsForCampaign(campaignId: Long): List<UacYdbCampaignContent>

    abstract fun getTrackingInfosByAccountIdAndAppId(accountId: String, appId: String): List<BaseUacTrackingInfo>

    fun deleteCampaign(
        campaignId: String,
        directCampaignId: Long,
        operatorUid: Long,
        clientId: ClientId
    ): Result<Void> {
        val result = campaignService
            .deleteCampaigns(listOf(directCampaignId), operatorUid, clientId, Applicability.FULL)
        // проверять isSuccessful или errors нельзя, так как ошибке при удалении кампании будет добавлена в subResult,
        // основной при этом будет successful
        if (result.validationResult?.hasAnyErrors() == true) {
            getLogger().error(
                "Failed to delete campaign ${directCampaignId}, errors: ${
                    result.validationResult?.flattenErrors()
                }"
            )
            return Result.broken(result.validationResult)
        }
        deleteBrief(campaignId)
        return Result.successful(null)
    }

    abstract fun deleteBrief(campaignId: String)

    /**
     * @return `false`, если обновить статус не удалось
     */
    fun updateDirectCampaignStatusShow(
        operator: User,
        clientId: ClientId,
        directCampaignId: Long,
        ydbCampaign: UacYdbCampaign,
        statusShowNew: Boolean,
    ): Boolean {
        var result = updateDirectCampaignStatusShow(
            operator,
            clientId,
            directCampaignId,
            statusShowNew
        )

        // Обновляем statusShow у подкампаний при необходимости
        if (result && ydbCampaign.isEcom == true) {
            val subCampaignIds = campaignService.getSubCampaignIdsWithMasterIds(listOf(directCampaignId), clientId)
            val subCampaigns = campaignService.getCampaigns(clientId, subCampaignIds.keys)
            subCampaigns.forEach {
                val subResult = updateDirectCampaignStatusShow(operator, clientId, it.id, statusShowNew)
                if (!subResult) {
                    logger.error("Unable to change statusShow for ${it.type} subcampaign")
                }
                result = result && subResult
            }
        }

        return result
    }

    private fun updateDirectCampaignStatusShow(
        operator: User,
        clientId: ClientId,
        directCampaignId: Long,
        statusShowNew: Boolean,
    ): Boolean {
        val result = rmpCampaignService.suspendResumeCampaignInDirect(
            directCampaignId, statusShowNew, operator.uid, clientId
        )
        if (result.validationResult?.hasAnyErrors() == true) {
            val vrErrors: String = result.validationResult?.flattenErrors()!!.joinToString("; ")
            getLogger().error("error while updating campaigns: {}", vrErrors)
            return false
        }
        return true
    }

    abstract fun getCampaignsWithRetargetingCondition(
        accountId: String,
        retargetingConditionId: Long
    ): List<UacYdbCampaign>

    fun getCampaignStatusShow(shard: Int, directCampaignId: Long): Boolean? {
        return campaignService.getStatusShowByCampaignId(shard, directCampaignId)
    }
}
