package ru.yandex.direct.core.entity.uac.service

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAccountRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUserRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAccount
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbUser
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.grut.api.ClientGrutApi
import ru.yandex.direct.core.grut.api.ClientGrutModel
import ru.yandex.direct.core.grut.replication.GrutApiService

abstract class BaseUacClientService {
    abstract fun getOrCreateClient(operator: User, subjectUser: User): String
}

@Service
@Lazy
class YdbUacClientService(
    private val ydbAccountRepository: UacYdbAccountRepository,
    private val ydbUserRepository: UacYdbUserRepository
) : BaseUacClientService() {
    override fun getOrCreateClient(operator: User, subjectUser: User): String {
        val user = ydbUserRepository.getUserByUid(operator.uid)
        if (user == null) {
            val newUser = UacYdbUser(uid = operator.uid)
            ydbUserRepository.saveUser(newUser)
        }

        val account = ydbAccountRepository.getAccountByClientId(subjectUser.clientId.asLong())
        if (account != null) {
            return account.id
        }
        val newAccount = UacYdbAccount(uid = subjectUser.uid, directClientId = subjectUser.clientId.asLong())
        ydbAccountRepository.saveAccount(newAccount)
        return newAccount.id
    }
}

@Service
@Lazy
class GrutUacClientService(
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService) : BaseUacClientService() {
    companion object {
        private const val GRUT_CREATE_CLIENT_ATTEMPTS = 5
    }

    override fun getOrCreateClient(operator: User, subjectUser: User): String {
        grutTransactionProvider.runRetryable(GRUT_CREATE_CLIENT_ATTEMPTS) {
            val client = grutApiService.clientGrutDao.getClient(subjectUser.clientId.asLong())

            if (client == null) {
                // здесь создаем неполного клиента, полный будет приезжать по репликации пока
                grutApiService.clientGrutDao.createObjects(listOf(ClientGrutModel(client = Client().withId(subjectUser.clientId.asLong()), ndsHistory = listOf())))
            }
        }
        return subjectUser.clientId.toString()
    }
}
