package ru.yandex.direct.core.entity.uac.service

import ru.yandex.direct.core.entity.uac.UacCommonUtils.CREATIVE_ID_KEY
import ru.yandex.direct.core.entity.uac.UacCommonUtils.CREATIVE_TYPE_KEY
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.result.Result
import java.net.URI

abstract class BaseUacContentService {

    companion object {
        private const val SMART_CENTERS = "smart-centers"
        private val FIELDS = listOf("direct_image_hash", "direct_mds_meta")
        private val COLORWIZ_FIELDS = listOf(
            "ColorWizBack",
            "ColorWizButton",
            "ColorWizButtonText",
            "ColorWizText",
        )
        private val VIDEO_META_FIELDS = listOf(
            "formats",
            "status",
            "thumb",
            "creative_id",
            "creative_type",
            "vast",
        )
        private val HTML5_META_FIELDS = listOf(
            "preview_url",
            "screenshot_url",
            CREATIVE_ID_KEY,
            CREATIVE_TYPE_KEY,
            "url",
        )
        val SIZES = listOf(
            "s16x9",
            "wx1080"
        )
    }

    abstract fun checkVisibleContent(clientId: ClientId, content: UacYdbContent): Boolean
    abstract fun getDbContents(contentIds: Collection<String>): List<UacYdbContent>
    abstract fun getCampaignContents(campaign: UacYdbCampaign): List<UacYdbCampaignContent>
    abstract fun deleteContent(id: String): Result<Void>
    abstract fun insertContent(content: UacYdbContent): Result<Void>
    abstract fun insertContents(contents: List<UacYdbContent>): Result<Void>
    abstract fun getDbContentByHash(hash: String): UacYdbContent?
    abstract fun getDbContentsByDirectCampaignId(
        directCampaignId: Long,
        clientId: ClientId,
        mediaType: MediaType
    ): Collection<UacYdbContent>

    fun getContent(contentId: String): Content? {
        return fillContent(getYdbContent(contentId))
    }

    fun getContentByHash(hash: String): Content? {
        return fillContent(getDbContentByHash(hash))
    }

    fun getYdbContent(contentId: String): UacYdbContent? {
        val contents = getDbContents(listOf(contentId))
        if (contents.isEmpty()) {
            return null
        }
        return contents[0]
    }

    fun getContents(contentIds: Collection<String>): List<Content> {
        if (contentIds.isEmpty()) {
            return emptyList()
        }
        val ydbContents = getDbContents(contentIds)
        return ydbContents.mapNotNull { fillContent(it) }
    }

    fun getContentsByDirectCampaignId(
        directCampaignId: Long,
        clientId: ClientId,
        mediaType: MediaType,
    ): Collection<Content> {
        val ydbContents = getDbContentsByDirectCampaignId(directCampaignId, clientId, mediaType)
        return ydbContents.mapNotNull { fillContent(it) }
    }

    fun fillContent(ydbContent: UacYdbContent?): Content? {
        if (ydbContent == null) {
            return null
        }
        val (tw, th) = extractContentSize(ydbContent)

        val origSize = ydbContent.meta["orig-size"] as Map<*, *>

        val ow = origSize["originalWidth"] as? Int ?: origSize["width"] as Int
        val oh = origSize["originalHeight"] as? Int ?: origSize["height"] as Int

        val (groupId, avatarsKey) = getUrlParams(ydbContent.thumb)
        return Content(
            id = ydbContent.id,
            type = ydbContent.type,
            thumb = ydbContent.thumb,
            thumbId = "$groupId/$avatarsKey",
            sourceUrl = ydbContent.sourceUrl,
            directImageHash = ydbContent.directImageHash,
            mdsUrl = ydbContent.mdsUrl,
            meta = fillContentMeta(ydbContent.meta, ydbContent.type),
            videoDuration = ydbContent.videoDuration,
            filename = ydbContent.filename,
            iw = origSize["width"] as Int,
            ih = origSize["height"] as Int,
            ow = ow,
            oh = oh,
            tw = tw,
            th = th,
        )
    }

    private fun extractContentSize(ydbContent: UacYdbContent): Pair<Int?, Int?> {
        val values = if (ydbContent.type == MediaType.IMAGE) {
            extractImageSize(ydbContent.meta)
        } else {
            extractNotImageSize(ydbContent.meta)
        }
        val tw = values["width"] as Int?
        val th = values["height"] as Int?
        return Pair(tw, th)
    }

    @Suppress("UNCHECKED_CAST")
    protected fun extractImageSize(meta: Map<String, Any?>): Map<String, Any> {
        val sizes = meta["sizes"] ?: return emptyMap()
        val thumbSize = meta.getOrDefault("avatars_thumb_size", "s16x9")
        val value = (sizes as Map<*, *>)[thumbSize] ?: return emptyMap()
        return value as Map<String, Any>
    }

    @Suppress("UNCHECKED_CAST")
    protected fun extractNotImageSize(meta: Map<String, Any?>): Map<String, Any> {
        val thumb = meta["thumb"] ?: return emptyMap()
        return thumb as Map<String, Any>
    }

    protected fun fillContentMeta(dbMeta: Map<String, Any?>, type: MediaType): Map<String, Any?> {
        return when (type) {
            MediaType.IMAGE -> fillImageContentMeta(dbMeta)
            MediaType.VIDEO -> fillVideoContentMeta(dbMeta)
            MediaType.HTML5 -> fillHtml5ContentMeta(dbMeta)
            else -> emptyMap()
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun fillImageContentMeta(dbMeta: Map<String, Any?>): Map<String, Any?> {
        val colorWizValues = COLORWIZ_FIELDS.associateWith { dbMeta[it] }
        val meta: MutableMap<String, Any?> = mutableMapOf("ColorWiz" to colorWizValues)
        for (field in FIELDS) {
            meta[field] = dbMeta[field]
        }
        val sizes = dbMeta["sizes"] as Map<String, Any>
        for ((size, info) in sizes) {
            if (SIZES.contains(size)) {
                if ((info as Map<String, Any>)[SMART_CENTERS] != null) {
                    val copiedInfo = HashMap(info)
                    copiedInfo[SMART_CENTERS] = (info[SMART_CENTERS] as Map<*, *>).values.toList()
                    meta[size] = copiedInfo
                } else {
                    meta[size] = info
                }
            }
        }
        return meta
    }

    @Suppress("UNCHECKED_CAST")
    private fun fillVideoContentMeta(dbMeta: Map<String, Any?>): Map<String, Any> {
        return VIDEO_META_FIELDS.map { it to dbMeta[it] }.filter { it.second != null }.toMap() as Map<String, Any>
    }

    @Suppress("UNCHECKED_CAST")
    private fun fillHtml5ContentMeta(dbMeta: Map<String, Any?>): Map<String, Any> {
        return HTML5_META_FIELDS.map { it to dbMeta[it] }.filter { it.second != null }.toMap() as Map<String, Any>
    }

    protected fun getUrlParams(url: String): MdsUrlPart {
        val uri = URI(url)
        val path = uri.path
        return try {
            val splitted = path.split("/")
            val groupId = splitted[2]
            val avatarsKey = splitted[3]
            MdsUrlPart(groupId, avatarsKey)
        } catch (e: IndexOutOfBoundsException) {
            MdsUrlPart(null, null)
        }
    }

    data class MdsUrlPart(
        val groupId: String?,
        val avatarsKey: String?
    )
}
