package ru.yandex.direct.core.entity.uac.service

import java.time.ZoneId
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.moderationdiag.convertList
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toGrutMediaType
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toCampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.mysql.BannerStatusesInfo
import ru.yandex.direct.core.entity.uac.repository.mysql.BannerStatusesRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAdGroup
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.grut.objects.proto.AssetLink
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.client.Schema

@Lazy
@Service
class CampaignContentUpdateService(
    uacCampaignContentService: UacCampaignContentService,
    featureService: FeatureService,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val uacYdbCampaignContentRepository: UacYdbCampaignContentRepository,
    private val uacBannerService: UacBannerService,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
    private val bannerRelationsRepository: BannerRelationsRepository,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val bannerStatusesRepository: BannerStatusesRepository,
    private val shardHelper: ShardHelper,
) : BaseCampaignContentUpdateService(uacCampaignContentService, featureService, ppcPropertiesSupport) {

    companion object {
        private val logger = LoggerFactory.getLogger(CampaignContentUpdateService::class.java)
    }

    /**
     * Аналог python функции update_campaign_content из tasks.py
     * Обновляем у campaignContent статусы, причины отклонения и флаги кампании
     * https://a.yandex-team.ru/arc/trunk/arcadia/yabs/rmp/backend/src/uac/campaign/updaters.py?rev=r8172954#L129
     */
    fun updateCampaignContentsAndCampaignFlags(
        uacCampaignId: String,
        clientId: ClientId,
        uacYdbDirectAdGroups: List<UacYdbDirectAdGroup>,
        uacYdbCampaignContents: Collection<UacYdbCampaignContent>,
    ) {
        val adGroupIdToUacAdGroupId = uacYdbDirectAdGroups
            .associateBy({ it.directAdGroupId }, { it.id })

        val uacYdbDirectAds = uacBannerService.getDirectAdsByContentIds(adGroupIdToUacAdGroupId.values)
        val bannersStatusesInfoList = uacBannerService
            .getBannerStatusesInfoByAdGroupIds(clientId, adGroupIdToUacAdGroupId.keys)
        logger.info("Get ${uacYdbDirectAds.size} banners from ydb and ${bannersStatusesInfoList.size} from mysql")

        // Получаем причины отклонения баннеров
        val bannerIdToModerationDiags = uacBannerService
            .getModerationDiagsByBannerIds(clientId, bannersStatusesInfoList.map { it.bannerId })
        val bannerIdToAssetsModerationDiags = uacBannerService
            .getAssetsModerationDiagsByBannerIds(clientId, bannersStatusesInfoList.map { it.bannerId })

        updateCampaignContentsAndCampaignFlagsInternal(uacCampaignId, bannersStatusesInfoList,
            bannerIdToModerationDiags, bannerIdToAssetsModerationDiags, uacYdbCampaignContents, uacYdbDirectAds
        )
    }

    /**
     *  Обновляет у campaignContent статусы, причины отклонения, а у campaign флаги контента
     *
     *  1) Получает по переданным campaignsIds все баннеры
     *  2) Для баннеров получает причины отклонения
     *  3) Для всех контентов кампании пересчитывает status, rejected_resaons из баннеров
     *  4) Для заказа пересчитывает флаги
     *
     */
    fun updateCampaignContentsAndCampaignFlags(
        shard: Int,
        directCampaignIds: Collection<Long>,
    ) {
        if (directCampaignIds.isEmpty()) return
        val directCampaignIdToCampaignId =
            uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(directCampaignIds)
        updateCampaignContentsAndCampaignFlags(shard, directCampaignIds, directCampaignIdToCampaignId)
    }

    fun updateCampaignContentsAndCampaignFlags(
        shard: Int,
        directCampaignIds: Collection<Long>,
        directCampaignIdToCampaignId: Map<Long, String>
    ) {
        logger.info("Going to handle campaigns $directCampaignIds")
        val bannerIdsToRecalc = bannerRelationsRepository.getBannerIdsByCampaignIds(shard, directCampaignIds)
        val uacYdbDirectAdsByDirectId = uacYdbDirectAdRepository.getByDirectAdId(bannerIdsToRecalc)
            .associateBy { it.directAdId!! }

        val bannerStatusesInfo = bannerStatusesRepository.getBannerStatusesInfo(shard, uacYdbDirectAdsByDirectId.keys)
        // если по одной кампании будет медленно работать, то надо добавить индекс campaign_id,id, и делать массовый запрос с пагинацией
        val campaignIdToCampaignContents = directCampaignIdToCampaignId.values.associateWith {
            uacYdbCampaignContentRepository.getCampaignContents(it)
        }

        logger.info("Get ${uacYdbDirectAdsByDirectId.size} banners from ydb and ${bannerStatusesInfo.size} from mysql")

        val bannerIds = bannerStatusesInfo.map { it.bannerId }
        // Получаем причины отклонения баннеров
        val bannerIdToModerationDiags = uacBannerService.getModerationDiagsByBannerIds(shard, bannerIds)
        val bannerIdToAssetsModerationDiags = uacBannerService.getAssetsModerationDiagsByBannerIds(shard, bannerIds)

        val bannerStatusesInfoByDirectCampaignId = bannerStatusesInfo
            .groupBy { it.campaignId }
        val clientIdByDirectCampaignId = shardHelper.getClientIdsByCampaignIds(
            bannerStatusesInfoByDirectCampaignId.keys
        ).mapValues { ClientId.fromLong(it.value) }

        if (bannerStatusesInfoByDirectCampaignId.size != directCampaignIdToCampaignId.size) {
            logger.error(
                "Some campaigns without banners: ${
                    directCampaignIdToCampaignId.keys.filterNot {
                        bannerStatusesInfoByDirectCampaignId.containsKey(
                            it
                        )
                    }
                }"
            )
        }
        for ((directCampaignId, bannersStatusesInfo) in bannerStatusesInfoByDirectCampaignId) {
            val clientId = clientIdByDirectCampaignId[directCampaignId]
            if (clientId == null) {
                logger.error("No clientId for direct campaign $directCampaignId")
                continue
            }
            val campaignId = directCampaignIdToCampaignId[directCampaignId]
            if (campaignId == null) {
                logger.error("No ydb campaign for direct campaign $directCampaignId")
                continue
            }
            val contentList = campaignIdToCampaignContents[campaignId]
            if (contentList == null || contentList.isEmpty()) {
                logger.error("No content for uac campaign $campaignId")
                continue
            }

            val directAds = bannersStatusesInfo.map { uacYdbDirectAdsByDirectId[it.bannerId]!! }

            val rejectReasonsByDirectBannerId = bannersStatusesInfo
                .associate { it.bannerId to bannerIdToModerationDiags.getOrDefault(it.bannerId, mapOf()) }
            val assetsRejectReasonsByDirectBannerId = bannersStatusesInfo
                .associate { it.bannerId to bannerIdToAssetsModerationDiags.getOrDefault(it.bannerId, mapOf()) }

            updateCampaignContentsAndCampaignFlagsInternal(campaignId, bannersStatusesInfo,
                rejectReasonsByDirectBannerId, assetsRejectReasonsByDirectBannerId, contentList, directAds
            )
        }
    }

    private fun updateCampaignContentsAndCampaignFlagsInternal(
        uacCampaignId: String,
        bannerStatusesInfoList: Collection<BannerStatusesInfo>,
        rejectReasonDiags: Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>>,
        assetsRejectReasonDiags: Map<Long, Map<BannerAssetType, List<ModerationDiag>>>,
        uacYdbCampaignContents: Collection<UacYdbCampaignContent>,
        uacYdbDirectAds: List<UacYdbDirectAd>,
    ) {

        val assetLinksList = uacYdbCampaignContents.map {
            AssetLink.TAssetLink.newBuilder().apply {
                assetId = it.id.toIdLong()
                createTime = it.createdAt.atZone(ZoneId.systemDefault()).toEpochSecond().toInt()
                if (it.removedAt != null)
                    removeTime = it.removedAt.atZone(ZoneId.systemDefault()).toEpochSecond().toInt()
                order = it.order
            }.build()
        }

        val assetsMetaListByAssetId = uacYdbCampaignContents.associate {
            it.id to Schema.TAssetMeta.newBuilder().apply {
                id = it.id.toIdLong()
                mediaType = it.type.toGrutMediaType()
            }.build()
        }

        // сайтлинки не хранятся как контент баннера, они хранятся на уровне заявки и у каждого баннера одинаковые
        // но рассчитывать статус для них надо, поэтому добавляем сайтлинки к контентам всех баннеров кампании
        val siteLinksSetContentIds: List<String> =
            uacYdbCampaignContents.filter { it.type == MediaType.SITELINK }.map { it.id }
        val bannerSpecList = uacYdbDirectAds.associate {
            val assetIds = listOfNotNull(
                it.directContentId,
                it.titleContentId,
                it.textContentId,
                it.directImageContentId,
                it.directVideoContentId,
                it.directHtml5ContentId
            ) + siteLinksSetContentIds
            it.directAdId to Banner.TBannerSpec.newBuilder().apply {
                addAllAssetIds(assetIds.map { it.toIdLong() })
            }.build()
        }
        val bannersWithAssets = bannerSpecList.map {
            val assets = it.value.assetIdsList.mapNotNull { assetId -> assetsMetaListByAssetId[assetId.toIdString()] }
            BannerWithAssets(bannerId = it.key!!, banner = it.value, assets = assets,
                assetLinkIds = assets.map { asset -> asset.id.toIdString() })
        }
        val campaignUpdateStatusesContainer = calculateCampaignContentsAndCampaignFlagsInternal(
            assetLinks = assetLinksList,
            banners = bannersWithAssets,
            bannerStatusesInfoList = bannerStatusesInfoList,
            rejectReasonDiags = rejectReasonDiags.mapKeys { it.key },
            assetsRejectReasonDiags = assetsRejectReasonDiags.mapKeys { it.key })

        updateRejectReasonsAndStatus(
            uacYdbCampaignContents,
            campaignUpdateStatusesContainer.assetLinkRejectReasonsByAssetLinkId,
            campaignUpdateStatusesContainer.assetLinkStatusByAssetLinkId
        )
        updateCampaignFlags(uacCampaignId, campaignUpdateStatusesContainer.contentFlags)
    }

    /**
     * Обновляем причины отклонения и статус у campaignContent
     *
     * @param uacYdbCampaignContents список связок кампания-контент для кампании
     * @param contentIdToRejectReasons мапа: id контента в список причин отклонения баннера
     * @param contentIdToStatus мапа: id  id контента в статус связки кампания-контент
     */
    private fun updateRejectReasonsAndStatus(
        uacYdbCampaignContents: Collection<UacYdbCampaignContent>,
        contentIdToRejectReasons: Map<String, List<ModerationDiag>>,
        contentIdToStatus: Map<String, AssetLink.TAssetLinkStatus.EAssetLinkStatus>
    ) {
        val campaignContentsByCampaignContentId = uacYdbCampaignContents.associateBy { it.id }
        val campaignContentIdToConvertedRejectReasons = contentIdToRejectReasons.mapValues { convertList(it.value) }
            .filter {
                it.value.sortedBy { rejReason -> rejReason.diagId } !=
                    campaignContentsByCampaignContentId[it.key]!!.rejectReasons?.sortedBy { rejReason -> rejReason.diagId }
            }
            .mapKeys { campaignContentsByCampaignContentId[it.key]!!.id }

        val campaignContentIdToStatus = contentIdToStatus.mapValues { it.value.toCampaignContentStatus()!! }
            .filter { it.value != campaignContentsByCampaignContentId[it.key]!!.status }
            .mapKeys { campaignContentsByCampaignContentId[it.key]!!.id }
        updateRejectReasons(campaignContentIdToConvertedRejectReasons)
        updateStatuses(campaignContentIdToStatus)
    }

    fun updateStatuses(campaignContentIdToStatus: Map<String, CampaignContentStatus>) {
        val statusToCampaignContentIds =
            campaignContentIdToStatus.map { it.value to it.key }.groupBy({ it.first }, { it.second })
        statusToCampaignContentIds.keys.forEach { status ->
            val campaignContentIds = statusToCampaignContentIds[status]
            uacYdbCampaignContentRepository.updateStatusForCampaignContents(campaignContentIds!!, status)
        }
    }

    /**
     * Обновляем причины отклонения у campaignContent
     */
    private fun updateRejectReasons(
        campaignContentIdToRejectReasons: Map<String, List<ModerationDiagData>>,
    ) {
        logger.info("Try to update reject_reasons for ${campaignContentIdToRejectReasons.size} campaignContents")

        campaignContentIdToRejectReasons.keys.forEach { campaignContentId ->
            val modelChanges: KtModelChanges<String, UacYdbCampaignContent> = KtModelChanges(campaignContentId)
            modelChanges.process(
                UacYdbCampaignContent::rejectReasons,
                campaignContentIdToRejectReasons[campaignContentId]
            )

            uacYdbCampaignContentRepository.updateRejectReasons(modelChanges)
        }
    }

    /**
     * Обновляем content_flags кампании
     */
    private fun updateCampaignFlags(
        uacCampaignId: String,
        contentFlags: Map<String, String>,
    ) {
        val modelChanges: KtModelChanges<String, UacYdbCampaign> = KtModelChanges(uacCampaignId)
        modelChanges.process(UacYdbCampaign::contentFlags, contentFlags)
        uacYdbCampaignRepository.updateContentFlags(modelChanges)
    }
}
