package ru.yandex.direct.core.entity.uac.service

import java.util.Optional
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup
import ru.yandex.direct.core.entity.adgroup.model.CriterionType.USER_PROFILE
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateOperationFactory
import ru.yandex.direct.core.entity.adgroup.service.complex.cpm.ComplexCpmAdGroupAddOperation
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsAddOperation
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsUpdateOperation
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentType
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.regions.GeoTree
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.result.Result

@Service
class CpmBannerCampaignService(
    val campaignOperationService: CampaignOperationService,
    val complexAdGroupAddOperationFactory: ComplexAdGroupAddOperationFactory,
    val complexAdGroupUpdateOperationFactory: ComplexAdGroupUpdateOperationFactory,
    val uacCampaignsCoreService: UacCampaignsCoreService,
    val shardHelper: ShardHelper,
    val campaignTypedRepository: CampaignTypedRepository,
    val clientGeoService: ClientGeoService,
) {

    fun getCpmBannerCampaign(clientId: ClientId, campaignId: Long): CpmBannerCampaign? {
        return uacCampaignsCoreService.getCampaign(clientId, campaignId, CpmBannerCampaign::class)
    }

    fun updateCpmBannerCampaign(
        operator: User,
        uidAndClientId: UidAndClientId,
        cpmBannerCampaign: CpmBannerCampaign,
        modelChanges: ModelChanges<CpmBannerCampaign>
    ): Result<CpmBannerCampaign> {
        val options = CampaignOptions()
        val updateOperation: RestrictedCampaignsUpdateOperation =
            campaignOperationService.createRestrictedCampaignUpdateOperation(
                listOf(modelChanges),
                operator.uid,
                uidAndClientId,
                options
            )

        val campaignResult = updateOperation.apply()

        if (campaignResult.validationResult.hasAnyErrors()) {
            return Result.broken(campaignResult.validationResult)
        }

        return Result.successful(cpmBannerCampaign)
    }

    fun addCpmBannerCampaign(
        operator: User,
        uidAndClientId: UidAndClientId,
        cpmBannerCampaign: CpmBannerCampaign
    ): Result<CpmBannerCampaign> {
        val options = CampaignOptions()

        val addOperation: RestrictedCampaignsAddOperation =
            campaignOperationService.createRestrictedCampaignAddOperation(
                listOf(cpmBannerCampaign),
                operator.uid,
                uidAndClientId,
                options
            )

        val result: Optional<MassResult<Long>> = addOperation.prepare()
        if (result.isPresent) {
            val groupResult = result.get()

            if (groupResult.result.size > 0) {
                return Result.broken(groupResult.get(0).validationResult)
            }

            return Result.broken(groupResult.validationResult)
        }

        val campaignResult = addOperation.apply()

        if (!campaignResult.isSuccessful) {
            return Result.broken(campaignResult.validationResult)
        }

        return Result.successful(cpmBannerCampaign)
    }

    fun createCpmBannerGroup(
        complexCpmAdGroup: ComplexCpmAdGroup?,
        geoTree: GeoTree,
        autoPriceParams: ShowConditionAutoPriceParams?,
        operatorUid: Long,
        uidAndClientId: UidAndClientId
    ): MassResult<Long> {
        val addGroupOperation: ComplexCpmAdGroupAddOperation =
            complexAdGroupAddOperationFactory.createCpmAdGroupAddOperation(
                false, listOf(complexCpmAdGroup), geoTree, true, autoPriceParams,
                operatorUid, uidAndClientId.clientId, uidAndClientId.uid, true
            )
        return addGroupOperation.prepareAndApply()
    }

    private fun adGroupByType(
        id: Long?,
        campaign: CpmBannerCampaign?,
        brief: UacAdGroupBrief,
        type: AdGroupType,
        geo: List<Long>?,
        cpmDirectContentType: DirectContentType?,
    ): AdGroup {
        val group: AdGroup

        when (type) {
            AdGroupType.CPM_VIDEO -> {
                group = CpmVideoAdGroup()
                group.withIsNonSkippable(brief.videosAreNonSkippable)
                cpmDirectContentType.let {
                    group.withIsNonSkippable(cpmDirectContentType == DirectContentType.NON_SKIPPABLE_VIDEO)
                }
                group.withCriterionType(USER_PROFILE)
            }
            AdGroupType.CPM_BANNER -> {
                group = CpmBannerAdGroup()
                group.withCriterionType(USER_PROFILE)
            }
            else -> {
                throw IllegalStateException("Incorrect group type")
            }
        }

        with(group) {
            withId(id)
            campaign?.let {
                withCampaignId(campaign.id);
                withName(campaign.name)
            }
            withType(type)
            withGeo(geo)
            withHyperGeoId(brief.hyperGeoId)
        }

        return group
    }

    fun complexGroupFromUac(
        id: Long?,
        campaign: CpmBannerCampaign?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        banners: List<BannerWithSystemFields>?,
        brief: UacAdGroupBrief,
        type: AdGroupType,
        geo: List<Long>?,
        cpmDirectContentType: DirectContentType? = null
    ): ComplexCpmAdGroup {
        val group = this.adGroupByType(id, campaign, brief, type, geo, cpmDirectContentType)

        val complexGroup = ComplexCpmAdGroup()
        with(complexGroup) {
            withAdGroup(group)
            withComplexBidModifier(complexBidModifier)
            retargetingCondition?.let {
                withTargetInterests(listOf(it.targetInterest));
                withRetargetingConditions(listOf(it))
            }

            banners?.let { withBanners(banners) }
        }

        return complexGroup
    }

    fun updateCpmBannerGroups(
        complexCpmAdGroups: List<ComplexCpmAdGroup>,
        autoPriceParams: ShowConditionAutoPriceParams?,
        operatorUid: Long,
        client: Client,
    ): MassResult<Long> {
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(client.clientId)
        val updateOperation =
            complexAdGroupUpdateOperationFactory.createCpmAdGroupUpdateOperation(
                complexCpmAdGroups,
                geoTree,
                false,
                autoPriceParams,
                operatorUid,
                ClientId.fromLong(client.clientId),
                client.chiefUid,
                false
            )
        return updateOperation.prepareAndApply()
    }
}
