package ru.yandex.direct.core.entity.uac.service

import java.net.IDN
import java.time.Duration
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.db.PpcPropertyNames.MARKETPLACE_DOMAINS
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.gemini.GeminiClient
import ru.yandex.direct.utils.CommonUtils
import ru.yandex.direct.utils.model.UrlParts

private const val DEFAULT_MIN_OFFERS_AMOUNT_ON_SITE_ALLOWED = 10L
private const val DEFAULT_MAX_OFFERS_AMOUNT_ON_SITE_ALLOWED = 100_000L

@Service
class EcomDomainsService @Autowired constructor(
    private val ecomDomainsRepository: EcomDomainsRepository,
    private val geminiClient: GeminiClient,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) {
    /**
     * Получение ecom доменов, с учетом главного зеркала, по урлам сайтов (с указанием схемы)
     * Если схема не указана - поиск происходит напрямую по домену
     */
    fun getEcomDomainsByUrls(urls: Collection<String>): Map<String, EcomDomain> {
        val urlsWithSchema = urls.filter { containsSchema(it) }
        val mainMirrors = toMainMirrors(urlsWithSchema)

        val domainsByOriginalUrl = urls
            .associateWith { mainMirrors[it] ?: it }

        val ecomDomains = ecomDomainsRepository.getByDomains(domainsByOriginalUrl.values.toSet())
            .associateBy { it.domain }

        return domainsByOriginalUrl
            .filterValues { ecomDomains.containsKey(it) }
            .mapValues { ecomDomains[it.value]!! }
    }

    fun toMainMirrors(urls: Collection<String>): Map<String, String> {
        return geminiClient.getMainMirrors(urls)
            .mapValues { UrlParts.fromUrl(it.value).domain }
    }

    fun normalizeDomain(url: String): String {
        val domain = if (containsSchema(url)) { UrlParts.fromUrl(url).domain } else url

        return IDN.toASCII(domain).lowercase()
    }

    private fun containsSchema(url: String): Boolean {
        return url.split("://").size == 2
    }

    fun getEcomDomainsByDomains(domains: Collection<String>): List<EcomDomain> {
        val normalized = domains.map { normalizeDomain(it) }

        return ecomDomainsRepository.getByDomainsAndMinMaxOffers(normalized,
            getMinOffersForEcomAllowed(), getMaxOffersForEcomAllowed())
    }

    fun getEcomDomainsWithOutdatedPreview(expirationHours: Long): List<EcomDomain> {
        return ecomDomainsRepository.getEcomDomainsWithOutdatedPreviews(expirationHours)
    }

    fun getMinOffersForEcomAllowed(): Long =
        ppcPropertiesSupport.get(PpcPropertyNames.MIN_OFFERS_AMOUNT_ON_SITE_ALLOWED, Duration.ofMinutes(15))
            .getOrDefault(DEFAULT_MIN_OFFERS_AMOUNT_ON_SITE_ALLOWED)

    fun getMaxOffersForEcomAllowed(): Long =
        ppcPropertiesSupport.get(PpcPropertyNames.MAX_OFFERS_AMOUNT_ON_SITE_ALLOWED, Duration.ofMinutes(15))
            .getOrDefault(DEFAULT_MAX_OFFERS_AMOUNT_ON_SITE_ALLOWED)

    fun isDomainSuitableEcomCheckNeeded(): Boolean {
        return CommonUtils.nvl(
            ppcPropertiesSupport.get(PpcPropertyNames.DO_VALIDATE_FEED_SITES_BY_ECOM_DOMAINS, Duration.ofMinutes(15))
                .get(),
            false
        )
    }

    fun getMarketplacesByUrls(urls: Collection<String>): Map<String, String> {
        val urlsWithSchema = urls.filter { containsSchema(it) }
        val mainMirrors = toMainMirrors(urlsWithSchema)

        val domainsByOriginalUrl = urls
            .associateWith { mainMirrors[it] ?: it }

        val marketplaceDomains = ppcPropertiesSupport.get(MARKETPLACE_DOMAINS).getOrDefault(emptySet())

        return domainsByOriginalUrl
            .filterValues { marketplaceDomains.contains(it) }
    }

    fun getMarketplacesByDomains(domains: Collection<String>): Collection<String> {
        val marketplaceDomains = ppcPropertiesSupport.get(MARKETPLACE_DOMAINS).getOrDefault(emptySet())

        return domains.asSequence()
            .map { normalizeDomain(it) }
            .filter { marketplaceDomains.contains(it) }
            .toList()
    }

}
