package ru.yandex.direct.core.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.model.EcomOfferCatalog
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomOfferCatalogsRepository
import ru.yandex.direct.gemini.GeminiClient
import ru.yandex.direct.utils.model.UrlParts

data class CatalogHrefWithBreadcrumbs(
    val catalogId: Long,
    val href: String,
    val title: String?,
    val imageUrl: String?,
    val breadcrumbs: List<String>,
    val visitsCount: Long
)

@Service
class EcomOfferCatalogsService(
    private val geminiClient: GeminiClient,
    private val ecomOfferCatalogsRepository: EcomOfferCatalogsRepository
) {
    /**
     * Поиск каталогов товаров с учетом главного зеркала по хостам (с указанием схемы),
     * отсортированными по количеству переходов
     */
    fun getCatalogsByHosts(hosts: Collection<String>): Map<String, List<CatalogHrefWithBreadcrumbs>> {
        val validHosts = hosts
            .filter { isValidHost(it) }

        val mainMirrors = geminiClient.getMainMirrors(validHosts)

        val catalogsByHosts = ecomOfferCatalogsRepository.getByHosts(mainMirrors.values)
            .groupBy { it.host }

        return mainMirrors
            .mapValues { catalogsByHosts[it.value] }
            .filterValues { it != null }
            .mapValues { sortAndFilterCatalogs(it.value!!) }
            .mapValues { convertCatalogs(it.value) }
    }

    private fun sortAndFilterCatalogs(catalogs: List<EcomOfferCatalog>): List<EcomOfferCatalog> {
        return catalogs
            .filter { it.title != null || it.catalogPath != null }
            .sortedByDescending { it.visitsCount }
    }

    private fun convertCatalogs(catalogs: List<EcomOfferCatalog>): List<CatalogHrefWithBreadcrumbs> {
        return catalogs.map { catalog ->
            val breadcrumbs = catalog.catalogPath?.split("\\") ?: emptyList()
            CatalogHrefWithBreadcrumbs(
                catalogId = catalog.id,
                href = catalog.url,
                title = catalog.title,
                imageUrl = catalog.imageUrl,
                breadcrumbs = breadcrumbs,
                visitsCount = catalog.visitsCount
            )
        }
    }

    private fun isValidHost(url: String): Boolean {
        val urlParts = UrlParts.fromUrl(url)
        return urlParts.protocol.isNotBlank() && (urlParts.path.isBlank() || urlParts.path == "/")
    }
}
