package ru.yandex.direct.core.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.bannerstorage.client.BannerStorageClient
import ru.yandex.direct.bannerstorage.client.model.Creative
import ru.yandex.direct.bannerstorage.client.model.CreativeGroup
import ru.yandex.direct.bannerstorage.client.model.Parameter
import ru.yandex.direct.bannerstorage.client.model.TemplateInclude
import ru.yandex.direct.bannerstorage.client.model.TnsArticle
import ru.yandex.direct.bannerstorage.client.model.TnsBrand
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType
import ru.yandex.direct.core.entity.creative.model.CreativeType
import ru.yandex.direct.core.entity.creative.model.StatusModerate
import ru.yandex.direct.core.entity.creative.service.CreativeService
import ru.yandex.direct.core.entity.feed.model.BusinessType
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesBusinessType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.utils.model.UrlParts

@Service
class EcomUcBannerService(
        private val bannerStorageClient: BannerStorageClient,
        private val creativeService: CreativeService,
        private val feedService: FeedService) {

    companion object {
        private val logger = LoggerFactory.getLogger(EcomUcBannerService::class.java)

        private const val DEFAULT_THEME_ID = 910
        private val FEED_TYPE_TO_THEME = mapOf(
                BusinessType.RETAIL to 740,
                BusinessType.HOTELS to 741,
                BusinessType.REALTY to 778,
                BusinessType.AUTO to 779,
                BusinessType.FLIGHTS to 783,
                BusinessType.OTHER to 910
        )

        // TNS категории и бренды для смартов всегда одни и те же
        private val TNS_ARTICLES = listOf(TnsArticle(-3))
        private val TNS_BRANDS = listOf(TnsBrand(1))
    }

    fun fillEcomPerformanceBanners(clientId: ClientId, brief: UacAdGroupBrief): List<BannerWithSystemFields> {
        val feeds = feedService.getFeeds(clientId, listOf(brief.feedId))
        if (feeds.isEmpty()) {
            throw IllegalStateException("Feed doesn't exist: ${brief.feedId}")
        }
        val themeId = FEED_TYPE_TO_THEME[feeds[0].businessType] ?: DEFAULT_THEME_ID
        val template = bannerStorageClient.getTemplate(themeId, TemplateInclude.LAYOUT_CODES)
        val result = bannerStorageClient.createSmartCreativeGroup(
                CreativeGroup(
                        null,
                        "Creatives for ecommerce campaign",
                        template.layoutCodes!!.map {
                            Creative()
                                    .withTemplateId(themeId)
                                    .withLayoutCodeId(it.id)
                                    .withParameters(getCreativesParameters(brief.domain()))
                                    .withTnsArticles(TNS_ARTICLES)
                                    .withTnsBrands(TNS_BRANDS)
                        }
                )
        )

        val creativesToAdd = result.creatives.map { creative ->
            ru.yandex.direct.core.entity.creative.model.Creative()
                    .withId(creative.id.toLong())
                    .withClientId(clientId.asLong())
                    .withName(creative.name)
                    .withType(CreativeType.PERFORMANCE)
                    .withStatusModerate(StatusModerate.NEW)
                    .withPreviewUrl(creative.screenshotUrl ?: creative.thumbnailUrl)
                    .withLivePreviewUrl(creative.preview.url)
                    .withWidth(creative.width.toLong())
                    .withHeight(creative.height.toLong())
                    .withStockCreativeId(creative.id.toLong())
                    .withLayoutId(creative.layoutCode.layoutId.toLong())
                    .withTemplateId(creative.templateId.toLong())
                    .withVersion(creative.version.toLong())
                    .withThemeId(creative.layoutCode.themeId.toLong())
                    .withBusinessType(CreativeBusinessType.fromSource(PerfCreativesBusinessType.valueOf(creative.businessType.directId)))
                    .withGroupName(result.name)
                    .withCreativeGroupId(result.id!!.toLong())
                    .withIsBannerstoragePredeployed(creative.isPredeployed)
                    .withIsAdaptive(false)
                    .withIsBrandLift(false)
                    .withHasPackshot(false)
                    .withModerateTryCount(0)
        }

        val massResult = creativeService.createOrUpdate(creativesToAdd, clientId)

        if (!massResult.isSuccessful) {
            logger.error("Can't create creatives for ecom smart banners, first error {}",
                    massResult.validationResult.flattenErrors()[0].toString())
            return listOf()
        }

        return massResult.result.map {
            PerformanceBanner()
                    .withCreativeId(it.result)
        }
    }

    private fun getCreativesParameters(domain: String) = listOf(
                Parameter()
                        .withParamName("LOGO")
                        .withValues(listOf()),
                Parameter()
                        .withParamName("DOMAIN_LIST")
                        .withValues(listOf(domain)),
                Parameter()
                        .withParamName("BUY_BUTTON_TEXT")
                        .withValues(listOf("Купить")),
                Parameter()
                        .withParamName("BUTTON_TEXT_COLOR")
                        .withValues(listOf("#000000")),
                Parameter()
                        .withParamName("BUTTON_COLOR")
                        .withValues(listOf("#FFDB4D")),
        )

    private fun UacAdGroupBrief.domain() = UrlParts.fromUrl(this.url).domain
}
