package ru.yandex.direct.core.entity.uac.service

import java.time.Duration
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag
import ru.yandex.direct.core.entity.moderationdiag.service.ModerationDiagService
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.contentFlagsMapToGrutContentFlags
import ru.yandex.direct.core.entity.uac.converter.getProtoEnumValueName
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName.UAC_MULTIPLE_AD_GROUPS_ENABLED
import ru.yandex.grut.objects.proto.AssetLink
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.Campaign.TCampaignBrief
import ru.yandex.grut.objects.proto.Campaign.TCampaignSpec
import ru.yandex.grut.objects.proto.MediaType
import ru.yandex.grut.objects.proto.RejectReasons
import ru.yandex.grut.objects.proto.client.Schema.TCampaign
import ru.yandex.grut.objects.proto.client.Schema.TCampaignMeta


@Lazy
@Service
class GrutCampaignContentUpdateService(
    uacCampaignContentService: UacCampaignContentService,
    featureService: FeatureService,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val uacBannerService: UacBannerService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutContext: GrutContext,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val grutUacContentService: GrutUacContentService,
    private val grutApiService: GrutApiService
) : BaseCampaignContentUpdateService(uacCampaignContentService, featureService, ppcPropertiesSupport) {

    private val fetchDataFromGrutProp = ppcPropertiesSupport.get(PpcPropertyNames.FETCH_DATA_FROM_GRUT_FOR_ASSET_LINK_STATUSES,
        Duration.ofMinutes(5))

    companion object {
        private val logger = LoggerFactory.getLogger(CampaignContentUpdateService::class.java)
    }

    /**
     *  Обновляет у campaignContent статусы, причины отклонения, а у campaign флаги контента
     *
     *  1) Получает по переданным campaignsIds все баннеры
     *  2) Для баннеров получает причины отклонения
     *  3) Для всех контентов кампании пересчитывает status, rejected_resaons из баннеров
     *  4) Для заказа пересчитывает флаги
     *
     */
    fun updateCampaignContentsAndCampaignFlags(
        shard: Int,
        directCampaignIds: Collection<Long>,
    ) {
        logger.info("Going to handle campaigns $directCampaignIds")
        val grutCampaigns = grutUacCampaignService.getCampaigns(directCampaignIds.map { it.toString() })

        if (grutCampaigns.isEmpty()) {
            logger.info("No campaigns in GrUT")
            return
        }

        val cidsWithAdGroupBriefEnabled = getCampaignIdsWithAdGroupBriefEnabled(grutCampaigns)

        // Получение групповых заявок по id кампании
        val grupAdGroupBriefsByCampaignId: Map<Long, List<AssetLink.TAssetLink>> =
            if (cidsWithAdGroupBriefEnabled.isEmpty()) mapOf()
            else grutApiService.adGroupBriefGrutApi.selectTAdGroupBriefsByCampaignIds(
                cidsWithAdGroupBriefEnabled,
                attributeSelector = listOf("/meta/campaign_id", "/spec/brief_asset_links"),
            )
                .groupBy { it.meta.campaignId }
                .mapValues { (_, grutAdGroupBriefs) ->
                    grutAdGroupBriefs
                        .map { grutAdGroupBrief -> grutAdGroupBrief.spec.briefAssetLinks.linksList }
                        .flatten()
                }

        // Собираем ассеты по кампаниям
        val grutAssetLinksByCampaignId: Map<Long, List<AssetLink.TAssetLink>> = grutCampaigns
            .associateBy({ it.meta.id }) { grutCampaign ->
                if (grupAdGroupBriefsByCampaignId[grutCampaign.meta.id].isNullOrEmpty()) {
                    grutCampaign.spec.briefAssetLinks.linksList
                } else {
                    grupAdGroupBriefsByCampaignId[grutCampaign.meta.id]!!
                }
            }

        val fetchDataFromGrut = fetchDataFromGrutProp.getOrDefault(false)
        val campaignIdToBanners = selectGrutBannersByCampaignIds(grutCampaigns, grutAssetLinksByCampaignId)
        val bannerIds = campaignIdToBanners.values.flatten().map { it.bannerId }

        // получаем информацию о статусах модерации баннера, флагах и причинах отклонения
        val bannerModerationInfos = uacBannerService.getBannerModerationData(shard, bannerIds, fetchDataFromGrut)

        val campaignUpdateStatusesContainers = grutCampaigns.associate { grutCampaign ->
            val campaignId = grutCampaign.meta.id.toIdString()

            val banners = campaignIdToBanners[campaignId] ?: listOf()
            val bannerStatuses = banners.mapNotNull { bannerModerationInfos[it.bannerId]?.bannerStatuses }
            val rejectReasonsByDirectBannerId = banners.associate {
                it.bannerId to (bannerModerationInfos[it.bannerId]?.moderationReasons ?: emptyMap())
            }
            val assetsRejectReasonsByDirectBannerId = banners.associate {
                it.bannerId to (bannerModerationInfos[it.bannerId]?.assetModerationReasons ?: emptyMap())
            }

            grutCampaign.meta.id.toIdString() to calculateCampaignContentsAndCampaignFlagsInternal(
                grutAssetLinksByCampaignId[grutCampaign.meta.id]!!,
                banners,
                bannerStatuses,
                rejectReasonsByDirectBannerId,
                assetsRejectReasonsByDirectBannerId)
        }
        grutTransactionProvider
            .runInTransaction {
                updateStatusesAndFlagsInTransaction(campaignUpdateStatusesContainers)
            }
    }

    private fun updateStatusesAndFlagsInTransaction(campaignUpdateStatusesContainers: Map<String, CampaignUpdateStatusesContainer>) {
        val campaignIds = campaignUpdateStatusesContainers.keys
        val specsByCampaignId = grutUacCampaignService.getCampaigns(campaignIds).associate { it.meta.id.toIdString() to it.spec }

        val briefsToUpdate = campaignUpdateStatusesContainers
            .filter {
                specsByCampaignId.containsKey(it.key)
            }
            .map { campaignUpdateStatusesContainer ->
                TCampaign.newBuilder()
                    .apply {
                        meta = TCampaignMeta.newBuilder().setId(campaignUpdateStatusesContainer.key.toIdLong()).build()
                        val rejectReasonsByAssetLinkId = campaignUpdateStatusesContainer.value.assetLinkRejectReasonsByAssetLinkId
                            .mapValues { it.value.map { rejectReason -> convertModerationDiag(rejectReason) } }
                        val newAssetLinkStatuses = campaignUpdateStatusesContainer.value.assetLinkStatusByAssetLinkId
                            .map { (assetLinkId, status) ->
                                AssetLink.TAssetLinkStatus.newBuilder().apply {
                                    this.assetLinkId = assetLinkId.toIdLong()
                                    assetId = campaignUpdateStatusesContainer.value.assetLinkIdToAssetLink[assetLinkId]!!.assetId
                                    rejectReasons = rejectReasonsBuilder.addAllRejectReasons(rejectReasonsByAssetLinkId.getOrDefault(assetLinkId, listOf())).build()
                                    this.status = status
                                }.build()
                            }
                        val oldSpec = specsByCampaignId[campaignUpdateStatusesContainer.key]
                        val oldCampaignBrief = oldSpec!!.campaignBrief
                        val newBrief = TCampaignBrief.newBuilder(oldCampaignBrief)
                            .setContentFlags(contentFlagsMapToGrutContentFlags(campaignUpdateStatusesContainer.value.contentFlags)).build()
                        val newSpec = TCampaignSpec.newBuilder(oldSpec)
                            .setBriefAssetLinksStatuses(AssetLink.TAssetLinksStatuses.newBuilder().addAllLinkStatuses(newAssetLinkStatuses).build())
                            .setCampaignBrief(newBrief)
                            .build()
                        spec = newSpec
                    }.build()
            }
        grutApiService.briefGrutApi.updateBriefsFull(briefsToUpdate)
    }

    private fun convertModerationDiag(diag: ModerationDiag) =
        RejectReasons.TRejectReason.newBuilder().apply {
            showDetailsUrl = ModerationDiagService.needShowDetailsUrl(diag)
            diagId = diag.id.toString()
            badReason = diag.strongReason
            unbanIsProhibited = diag.unbanIsProhibited
            diagText = diag.diagText
            if (diag.token != null) {
                token = diag.token
            }
            shortText = diag.shortText
            allowFirstAid = diag.allowFirstAid
        }.build()

    private fun selectGrutBannersByCampaignIds(
        campaigns: Collection<TCampaign>,
        tAssetsByCampaignId: Map<Long, List<AssetLink.TAssetLink>>,
    ): Map<String, List<BannerWithAssets>> {
        val campaignIds = campaigns.map { it.meta.id }
        val banners = grutApiService.briefBannerGrutApi.selectBanners(
            filter = "[/meta/campaign_id] IN (${campaignIds.distinct().joinToString { "$it" }})" +
                " AND [/spec/status] != \"${getProtoEnumValueName(Banner.TBannerSpec.EBannerStatus.BSS_DELETED)}\"",
            index = "banners_by_campaign"
        )
        val assetIds = tAssetsByCampaignId.values
            .flatten()
            .map { it.assetId.toIdString() }
            .toSet()
        val assetsByAssetId = grutUacContentService.getAssets(assetIds)
            .associateBy { it.meta.id.toIdString() }

        // сайтлинки не хранятся как контент баннера, они хранятся на уровне заявки и у каждого баннера одинаковые
        // но рассчитывать статус для них надо, поэтому добавляем сайтлинки к контентам всех баннеров кампании
        val campaignSiteLinks = campaigns.associate {
            val siteLinks = (tAssetsByCampaignId[it.meta.id] ?: emptyList())
                .mapNotNull { assetLink -> assetsByAssetId[assetLink.assetId.toIdString()] }
                .filter { asset -> asset.meta.mediaType == MediaType.EMediaType.MT_SITELINK }
                .map { asset -> asset.meta }
            it.meta.id to siteLinks
        }
        val campaignSiteLinksAssetLinkIds = campaigns.associate {
            val siteLinksAssetLinkIds = (tAssetsByCampaignId[it.meta.id] ?: emptyList())
                .filter { assetLink -> assetsByAssetId[assetLink.assetId.toIdString()]?.meta?.mediaType == MediaType.EMediaType.MT_SITELINK }
                .map { assetLink -> if (assetLink.hasId()) assetLink.id else assetLink.assetId }
            it.meta.id to siteLinksAssetLinkIds
        }

        return banners.map {
            val bannerAssets = it.spec.assetIdsList.mapNotNull { assetId ->
                val asset = assetsByAssetId[assetId.toIdString()]
                if (asset == null) {
                    logger.error("Absent asset $assetId for banner ${it.meta.id}")
                    null
                } else {
                    asset.meta
                }
            }
            val siteLinksAssets = campaignSiteLinks[it.meta.campaignId] ?: listOf()
            val assets = bannerAssets + siteLinksAssets
            val bannerAssetLinkIds = if (it.spec.assetLinkIdsCount == 0 && it.spec.assetIdsCount != 0) {
                it.spec.assetIdsList
            } else {
                it.spec.assetLinkIdsList
            }
            val siteLinksAssetLinkIds = campaignSiteLinksAssetLinkIds[it.meta.campaignId] ?: listOf()
            val assetLinkIds = bannerAssetLinkIds + siteLinksAssetLinkIds
            it.meta.campaignId to BannerWithAssets(it.meta.id, it.spec, assets,
                assetLinkIds.map { assetLinkId -> assetLinkId.toIdString() })
        }.groupBy({ it.first.toIdString() }, { it.second })

    }


    /**
     * Возвращает id кампаний, для которых включены групповые заявки
     */
    private fun getCampaignIdsWithAdGroupBriefEnabled(grutCampaigns: List<TCampaign>): Set<Long> {
        val clientIds = grutCampaigns
            .map { ClientId.fromLong(it.meta.clientId) }
            .toSet()
        val clientIdsWithAdGroupBriefEnabled =
            featureService.isEnabledForClientIds(clientIds, UAC_MULTIPLE_AD_GROUPS_ENABLED)
                .filterValues { it == true }
                .mapKeys { it.key.asLong() }
                .keys
        return grutCampaigns
            .filter { clientIdsWithAdGroupBriefEnabled.contains(it.meta.clientId) }
            .map { it.meta.id }
            .toSet()
    }
}
