package ru.yandex.direct.core.entity.uac.service

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toSitelink
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.readAssetLinks
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toCampaignSpec
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacYdbCampaign
import ru.yandex.direct.core.entity.uac.converter.getProtoEnumValueName
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.BaseUacTrackingInfo
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.moneyToDb
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.grut.api.GrutApiBase
import ru.yandex.direct.core.grut.api.UpdatedObject
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.client.Schema.TCampaignMeta
import java.math.BigDecimal

@Service
@Lazy
class GrutUacCampaignService(
    private val uacContentService: GrutUacContentService,
    rmpCampaignService: RmpCampaignService,
    campaignService: CampaignService,
    campaignSubObjectAccessCheckerFactory: CampaignSubObjectAccessCheckerFactory,
    uacClientService: GrutUacClientService,
    private val grutContext: GrutContext,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService,
) : BaseUacCampaignService(
    rmpCampaignService,
    campaignService,
    campaignSubObjectAccessCheckerFactory,
    uacClientService,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutUacCampaignService::class.java)
    }

    override fun getLogger(): Logger = logger

    override fun getCampaignById(id: String): UacYdbCampaign? {
        return grutApiService.briefGrutApi.getBrief(id.toIdLong())?.toUacYdbCampaign()
    }

    fun getCampaigns(ids: Collection<String>): List<Schema.TCampaign> {
        return grutApiService.briefGrutApi.getBriefs(ids.toIdsLong())
    }

    fun updateCampaign(campaign: UacYdbCampaign) {
        val briefUpdated = Schema.TCampaign.newBuilder().apply {
            meta = TCampaignMeta.newBuilder().setId(campaign.id.toIdLong()).build()
            spec = toCampaignSpec(campaign)
        }.build()
        grutApiService.briefGrutApi.updateBriefFull(briefUpdated)
    }

    override fun deleteBrief(campaignId: String) {
        grutApiService.briefGrutApi.deleteObjects(listOf(campaignId.toIdLong()))
    }

    override fun getCampaignsWithRetargetingCondition(
        accountId: String,
        retargetingConditionId: Long
    ): List<UacYdbCampaign> {
        val selectResult = grutApiService.briefGrutApi.selectBriefs(
            filter = "[/meta/client_id] = $accountId" +
                " AND [/spec/campaign_brief/retargeting_condition/id] = $retargetingConditionId"
        )

        return selectResult
            .map { it.toUacYdbCampaign() }
    }

    fun getCampaignNameByIds(ids: Collection<Long>) =
        grutApiService.briefGrutApi.getBriefs(
            ids = ids,
            attributeSelector = listOf("/meta/id", "/spec/campaign_data/name")
        ).associateBy({ it.meta.id }, { it.spec.campaignData.name })

    fun getCampaignIdsByClientId(clientId: String) =
        grutApiService.briefGrutApi.selectBriefs(
            filter = "[/meta/client_id] = $clientId",
            attributeSelector = listOf("/meta/id")
        ).map { it.meta.id }

    override fun getDirectCampaignIdById(id: String) = id.toIdLong()

    override fun getMinBannerIdForCampaign(id: String): Long? {
        val banners = grutApiService.briefBannerGrutApi.selectBanners(
            filter = "[/meta/campaign_id] = $id AND [/spec/status] != \"${
                getProtoEnumValueName(Banner.TBannerSpec.EBannerStatus.BSS_DELETED)
            }\"",
            index = "banners_by_campaign",
            attributeSelector = listOf("/meta/id")
        )
        return banners
            .map { it.meta.id }
            .minOrNull()
    }

    override fun getCampaignIdByDirectCampaignId(directCampaignId: Long) = directCampaignId.toString()

    fun fetchContents(assetLinks: List<UacYdbCampaignContent>): FetchedContents {
        val assetContainer = uacContentService.getAssetContainer(assetLinks)
        return FetchedContents(
            mediaContents = assetContainer.mediaAssets
                .filter { !it.isRemoved }
                .sortedBy { it.order }
                .map { it.asset }
                .let { uacContentService.fillMediaContentsFromAssets(it) },
            texts = assetContainer.texts.filter { !it.isRemoved }.sortedBy { it.order }.map { it.asset.text },
            titles = assetContainer.titles.filter { !it.isRemoved }.sortedBy { it.order }.map { it.asset.title },
            sitelinks = assetContainer.sitelinks.filter { !it.isRemoved }.sortedBy { it.order }
                .map { toSitelink(it.asset) },
        )
    }

    fun fetchContents(campaign: Schema.TCampaign): FetchedContents {
        val assetLinks = readAssetLinks(
            campaign.spec.briefAssetLinks,
            campaign.spec.briefAssetLinksStatuses,
            campaign.meta.id.toIdString()
        )
        return fetchContents(assetLinks)
    }

    override fun getDirectCampaignStatus(ydbCampaign: UacYdbCampaign) = ydbCampaign.directCampaignStatus

    override fun getAppIdsByAccountId(accountId: String): List<String> {
        val selectResult = grutApiService.briefGrutApi.selectBriefs(
            filter = "[/meta/client_id] = $accountId AND NOT is_null([/spec/campaign_brief/app_id])",
            attributeSelector = listOf("/spec/campaign_brief/app_id")
        )

        return selectResult
            .map { it.spec.campaignBrief.appId }
            .distinct()
    }

    override fun getTextContentsByMediaType(campaingId: Long): Map<MediaType, List<String>> {
        val uacCampaign = getCampaignByDirectCampaignId(campaingId) ?: return emptyMap()
        return uacContentService.getCampaignContents(uacCampaign)
            .asSequence()
            .filter { it.removedAt == null }
            .filter { it.type != null && it.type.isTextType() }
            .groupBy({ it.type!! }, { it.text!! })
    }

    override fun getTrackingInfosByAccountIdAndAppId(accountId: String, appId: String): List<BaseUacTrackingInfo> {
        val selectResult = grutApiService.briefGrutApi.selectBriefs(
            filter = "[/meta/client_id] = $accountId AND [/spec/campaign_brief/app_id] = \"$appId\""
        )

        return selectResult
            .map {
                BaseUacTrackingInfo(
                    trackingUrl = it.spec.campaignBrief.targetHref.trackingUrl,
                    impressionUrl = it.spec.campaignBrief.targetHref.impressionUrl,
                    id = it.meta.id,
                    name = it.spec.campaignData.name,
                    createdTime = DateTimeUtils.fromEpochSeconds(it.meta.creationTime / 1_000_000)
                )
            }
    }

    override fun getMediaCampaignContentsForCampaign(campaignId: Long): List<UacYdbCampaignContent> {
        val uacCampaign = getCampaignByDirectCampaignId(campaignId) ?: return listOf()
        return uacContentService.getCampaignContents(uacCampaign)
            .asSequence()
            .filter { it.removedAt == null }
            .filter { it.contentId != null }
            .toList()
    }

    /**
     * Функция проверяет какие ассеты из assetIds, добавлены в кампании заданного клиента и
     * возвращает список идентификаторов найденных ассетов
     *  @param clientId - идентификатор клиента
     *  @param assetIds - список идентификаторов ассетов, которые нужно проверить на вхождение в кампании клиента
     */
    fun getClientCampaignAssets(
        clientId: String,
        assetIds: Set<String>,
        limit: Long = 2000L
    ): Set<String> {
        val iterationLimit = 500L
        var continuationToken: String? = null
        var iteration = 0L
        val clientAssetIds = mutableSetOf<String>()

        while (true) {
            val result = grutTransactionProvider.runRetryable(3) {
                grutContext.client.selectObjects(
                    ObjectApiServiceOuterClass.TReqSelectObjects.newBuilder().apply {
                        filter = "[/meta/client_id] = $clientId"
                        objectType = Schema.EObjectType.OT_CAMPAIGN
                        addAllAttributeSelector(listOf("/spec/brief_asset_links"))
                        this.limit = limit
                        continuationToken?.let { this.continuationToken = it }
                    }.build()
                )
            }
            val currentAssets = result.payloadsList
                .asSequence()
                .mapNotNull { Schema.TCampaign.parseFrom(it.protobuf) }
                .mapNotNull { campaign -> campaign.spec.briefAssetLinks.linksList }
                .flatten()
                .mapNotNull { it.assetId.toIdString() }
                .filter { assetIds.contains(it) }
                .toSet()
            clientAssetIds.addAll(currentAssets)

            continuationToken = result.continuationToken
            if (iteration++ == iterationLimit) {
                throw IllegalStateException("Too many campaigns for $clientId. Aborting reading")
            }
            if (result.payloadsCount < limit) {
                break
            }
        }
        return clientAssetIds
    }

    fun updateCampaignGoals(campaignIds: List<Long>, goalIds: Set<Long>) {
        val campaigns = getCampaigns(campaignIds.map { it.toString() })
        if (campaigns.isEmpty()) {
            return
        }
        val campaignsToUpdate = campaigns.map {
            createRequestToUpdateCampaignGoals(
                campaignId = it.meta.id,
                strategy = it.spec.campaignBrief.strategy,
                goalIds = goalIds
            )
        }
        grutApiService.briefGrutApi.updateBriefs(campaignsToUpdate, listOf("/spec/campaign_brief/strategy/goals"))
    }

    private fun createRequestToUpdateCampaignGoals(
        campaignId: Long,
        strategy: Campaign.TBriefStrategy,
        goalIds: Set<Long>
    ): Schema.TCampaign {
        val newStrategyBuilder = strategy.toBuilder()
        if (goalIds.isNotEmpty()) newStrategyBuilder.clearGoals()
        goalIds.forEach { newStrategyBuilder.addGoalsBuilder().goalId = it }

        return Schema.TCampaign.newBuilder().apply {
            meta = TCampaignMeta.newBuilder().setId(campaignId).build()
            spec = Campaign.TCampaignSpec.newBuilder().apply {
                campaignBrief = Campaign.TCampaignBrief.newBuilder()
                    .setStrategy(newStrategyBuilder.build())
                    .build()
            }.build()
        }.build()
    }

    fun updateCampaignAvgCpm(campaignIds: List<Long>, avgCpm: BigDecimal) {
        val campaigns = getCampaigns(campaignIds.map { it.toString() })

        if (campaigns.isEmpty()) {
            return
        }

        val campaignsToUpdate = campaigns.map {
            createRequestToUpdateAvgCpm(
                campaignId = it.meta.id,
                strategy = it.spec.campaignBrief.strategy,
                avgCpm = moneyToDb(avgCpm)
            )
        }
        grutApiService.briefGrutApi.updateBriefs(
            campaignsToUpdate,
            listOf("/spec/campaign_brief/strategy/strategy_data/avg_cpm")
        )
    }

    private fun createRequestToUpdateAvgCpm(
        campaignId: Long,
        strategy: Campaign.TBriefStrategy,
        avgCpm: Long
    ): Schema.TCampaign {
        val newStrategy = strategy.toBuilder().setStrategyData(
            strategy.strategyData.toBuilder().setAvgCpm(avgCpm)
        )
            .build()

        return Schema.TCampaign.newBuilder().apply {
            meta = TCampaignMeta.newBuilder().setId(campaignId).build()
            spec = Campaign.TCampaignSpec.newBuilder()
                .apply {
                    campaignBrief = Campaign.TCampaignBrief.newBuilder()
                        .setStrategy(newStrategy).build()
                }
                .build()
        }.build()
    }

    fun deleteCampaignKeywordsAndAppendMinusKeywords(
        keywordsToDeleteByCampaignId: Map<Long, List<String>>,
        minusKeywordsToAppendByCampaignId: Map<Long, List<String>>
    ) {
        val allPossibleCampaignIds = keywordsToDeleteByCampaignId.keys
            .plus(minusKeywordsToAppendByCampaignId.keys)
            .map { it.toString() }

        val campaigns = getCampaigns(allPossibleCampaignIds)
        if (campaigns.isEmpty()) {
            return
        }

        val briefsToUpdate = campaigns.map {
            val keywordsToDelete = keywordsToDeleteByCampaignId[it.meta.id] ?: emptyList()

            createRequestToDeleteKeywordsAndAppendMinusKeywords(
                campaignId = it.meta.id,
                currentKeywords = it.spec.campaignBrief.keywordsList,
                keywordsToDelete = keywordsToDelete,
                currentMinusKeywords = it.spec.campaignBrief.minusKeywordsList,
                minusKeywordsToAppend = minusKeywordsToAppendByCampaignId[it.meta.id]!!
            )
        }
        grutApiService.briefGrutApi.updateBriefs(briefsToUpdate)
    }

    private fun createRequestToDeleteKeywordsAndAppendMinusKeywords(
        campaignId: Long,
        currentKeywords: List<String>,
        keywordsToDelete: List<String>,
        currentMinusKeywords: List<String>,
        minusKeywordsToAppend: List<String>
    ): UpdatedObject {
        val newKeywords = currentKeywords.filterNot { keyword -> keywordsToDelete.contains(keyword) }
        val newMinusKeywords = currentMinusKeywords.union(minusKeywordsToAppend)
        val setPaths = mutableListOf<String>()
        val removePaths = mutableListOf<String>()
        if (newKeywords.isEmpty()) {
            removePaths.add("/spec/campaign_brief/keywords")
        } else {
            setPaths.add("/spec/campaign_brief/keywords")
        }

        if (newMinusKeywords.isEmpty()) {
            removePaths.add("/spec/campaign_brief/minus_keywords")
        } else {
            setPaths.add("/spec/campaign_brief/minus_keywords")
        }
        return UpdatedObject(
            meta = TCampaignMeta.newBuilder().setId(campaignId).build().toByteString(),
            spec = Campaign.TCampaignSpec.newBuilder().apply {
                campaignBrief = Campaign.TCampaignBrief.newBuilder().apply {
                    if (newKeywords.isNotEmpty()) {
                        addAllKeywords(newKeywords)
                    }
                    if (newMinusKeywords.isNotEmpty()) {
                        addAllMinusKeywords(newMinusKeywords)
                    }
                }.build()
            }.build().toByteString(),
            setPaths = setPaths,
            removePaths = removePaths,
        )
    }

    fun updateWeeklyBudget(campaignIds: List<Long>, newWeeklyBudget: Long) {
        val campaigns = getCampaigns(campaignIds.map { it.toString() })
        if (campaigns.isEmpty()) {
            return
        }

        val briefsToUpdate = campaigns.map {
            getBriefToUpdateToUpdateWeeklyBudget(
                campaignId = it.meta.id,
                strategy = it.spec.campaignBrief.strategy,
                weeklyBudget = newWeeklyBudget
            )
        }

        grutApiService.briefGrutApi
            .updateBriefs(briefsToUpdate, setPaths = listOf("/spec/campaign_brief/strategy/week_limit"))
    }

    private fun getBriefToUpdateToUpdateWeeklyBudget(
        campaignId: Long,
        strategy: Campaign.TBriefStrategy,
        weeklyBudget: Long
    ): Schema.TCampaign {
        val newStrategy = strategy.toBuilder().setWeekLimit(weeklyBudget).build()

        return Schema.TCampaign.newBuilder().apply {
            meta = TCampaignMeta.newBuilder().setId(campaignId).build()
            spec = Campaign.TCampaignSpec.newBuilder().apply {
                campaignBrief = Campaign.TCampaignBrief.newBuilder()
                    .setStrategy(newStrategy)
                    .build()
            }.build()
        }.build()
    }

    fun updateAudienceSegmentsSyncedFields(campaign: UacYdbCampaign) {
        val briefUpdated = Schema.TCampaign.newBuilder().apply {
            meta = TCampaignMeta.newBuilder().setId(campaign.id.toIdLong()).build()
            spec = toCampaignSpec(campaign)
        }.build()

        grutApiService.briefGrutApi.updateBrief(
            briefUpdated,
            setPaths = listOf("/spec/campaign_brief/audience_segments_synchronized")
        )
    }

    override fun getCampaignProtoByDirectCampaignId(directCampaignId: Long): Schema.TCampaign? {
        return grutApiService.briefGrutApi.getBrief(directCampaignId)
    }

    fun getCampaignsByFeedIds(
        accountId: String,
        feedIds: Collection<Long>
    ): Map<Long, List<UacYdbCampaign>> {
        if (feedIds.isEmpty()) {
            return emptyMap()
        }
        val selectResult = grutApiService.briefGrutApi.selectBriefs(
            filter = "[/meta/client_id] = $accountId AND int64([/spec/campaign_brief/ecom/feed_id]) IN (${feedIds.distinct().joinToString { "$it" }})"
        )
        val uacYdbCampaigns = selectResult.map { it.toUacYdbCampaign() }.filter { it.feedId != null }
        return uacYdbCampaigns.groupBy { it.feedId!! }
    }

    fun updateCampaignFeeds(
        accountId: String,
        feedIdsToNewFeedIds: Map<Long, Long>
    ) {
        grutTransactionProvider.runInRetryableTransaction(GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val oldFeedIds = feedIdsToNewFeedIds.keys
            val oldCampaigns = getCampaignsByFeedIds(accountId, oldFeedIds).values.flatten()

            val updatedCampaigns = oldCampaigns.map {
                it.copy(feedId = feedIdsToNewFeedIds[it.feedId])
            }

            val campaignsToUpdate = updatedCampaigns.map { campaign ->
                Schema.TCampaign.newBuilder().apply {
                    meta = TCampaignMeta.newBuilder().setId(campaign.id.toIdLong()).build()
                    spec = toCampaignSpec(campaign)
                }.build()
            }

            grutApiService.briefGrutApi.updateBriefs(
                campaignsToUpdate,
                listOf("/spec/campaign_brief/ecom/feed_id")
            )
        }
    }
}
