package ru.yandex.direct.core.entity.uac.service

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.bs.export.model.AssetHashes
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toAssetGrut
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toGrutMediaType
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toHtml5Asset
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toImageAsset
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toMediaType
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toSitelinkAsset
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toUacYdbContent
import ru.yandex.direct.core.entity.uac.converter.UacGrutAssetsConverter.toVideoAsset
import ru.yandex.direct.core.entity.uac.converter.getProtoEnumValueName
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.AssetContainer
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.LinkedAsset
import ru.yandex.direct.core.entity.uac.model.MediaAsset
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.model.TextAsset
import ru.yandex.direct.core.entity.uac.model.TitleAsset
import ru.yandex.direct.core.entity.uac.model.UacGrutPage
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdsLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.grut.api.AdGroupBriefGrutModel
import ru.yandex.direct.core.grut.api.UpdatedObject
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName.UAC_MULTIPLE_AD_GROUPS_ENABLED
import ru.yandex.direct.result.Result
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Asset.TAssetSpec
import ru.yandex.grut.objects.proto.AssetLink
import ru.yandex.grut.objects.proto.Banner.TBannerSpec.EBannerStatus.BSS_DELETED
import ru.yandex.grut.objects.proto.MediaType.EMediaType
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.client.Schema.TAsset
import ru.yandex.grut.objects.proto.client.Schema.TAssetMeta

data class AssetPatch(
    val id: String,
    val spec: TAssetSpec,
    val setPaths: List<String>,
)

@Service
@Lazy
class GrutUacContentService(
    val grutContext: GrutContext,
    val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService,
    val shardHelper: ShardHelper,
    val featureService: FeatureService,
) : BaseUacContentService() {

    companion object {
        private const val GRUT_READ_ATTEMPTS = 3
        private const val GRUT_CHUNK_SIZE = 2000L
    }

    override fun checkVisibleContent(clientId: ClientId, content: UacYdbContent): Boolean {
        return clientId.toString() == content.accountId
    }

    override fun getDbContents(contentIds: Collection<String>): List<UacYdbContent> {
        val contentIdToOrder = contentIds.mapIndexed { order, id ->
            id to order
        }.toMap()

        val assetContainer = getAssetContainer(contentIds.toSet(), contentIdToOrder)

        return assetContainer.mediaAssets.sortedBy { it.order }
            .map { it.asset }
            .map { toUacYdbContent(it) }
    }

    override fun getCampaignContents(campaign: UacYdbCampaign): List<UacYdbCampaignContent> {
        if (campaign.assetLinks.isNullOrEmpty()) return listOf()
        val contentIds = campaign.assetLinks
            .map { it.contentId!! }
            .toSet()
        val assetsById = getAssets(contentIds)
            .associateBy { it.meta.id.toIdString() }

        return campaign.assetLinks
            .map { fillUacYdbCampaignContent(it, assetsById[it.contentId]!!) }
    }

    fun getAssetsByAdGroups(uacAdGroupBriefs: List<AdGroupBriefGrutModel>?): Map<Long, List<UacYdbCampaignContent>> {
        if (uacAdGroupBriefs.isNullOrEmpty()) return mapOf()
        val assetIdsByBriefId = uacAdGroupBriefs
            .filter { it.id != null && it.assetLinks != null }
            .associateBy({ it.id!! }) { it.assetLinks!! }
        val contentIds = assetIdsByBriefId.values
            .flatten()
            .mapNotNull { it.contentId }
        if (contentIds.isEmpty()) return mapOf()

        val assetsById = getAssets(contentIds)
            .associateBy { it.meta.id.toIdString() }

        return assetIdsByBriefId
            .mapValues { (_, uacContents) ->
                uacContents
                    .map { uacContent ->
                        fillUacYdbCampaignContent(uacContent, assetsById[uacContent.contentId]!!)
                    }
            }
    }

    private fun fillUacYdbCampaignContent(
        uacCampaignContent: UacYdbCampaignContent,
        grutAsset: TAsset,
    ): UacYdbCampaignContent {
        val text = when (grutAsset.meta.mediaType) {
            EMediaType.MT_TEXT -> {
                grutAsset.spec.text
            }
            EMediaType.MT_TITLE -> {
                grutAsset.spec.title
            }
            else -> {
                null
            }
        }
        val grutSitelink = if (grutAsset.spec.hasSitelink()) grutAsset.spec.sitelink else null
        var uacSitelink: Sitelink? = null
        grutSitelink?.let { sl ->
            val description = if (sl.hasDescription()) sl.description else null
            uacSitelink = Sitelink(title = sl.title, href = sl.href, description = description)
        }

        return uacCampaignContent.copy(
            type = grutAsset.meta.mediaType.toMediaType(),
            text = text,
            sitelink = uacSitelink,
            asset = grutAsset
        )
    }

    override fun deleteContent(id: String): Result<Void> {
        grutTransactionProvider.runRetryable(3) {
            grutApiService.assetGrutApi.deleteObject(id.toIdLong())
        }
        return Result.successful(null)
    }

    fun getAssetContainer(campaign: UacYdbCampaign): AssetContainer {
        return getAssetContainer(campaign.assetLinks ?: emptyList())
    }

    fun getAssetContainer(assetLinks: List<UacYdbCampaignContent>): AssetContainer {
        val assetLinkIdToAssetId = assetLinks.associate { it.id to it.contentId!! }
        val orderByAssetLinkId = assetLinks.associateBy({ it.id }, { it.order })
        val removedAssetLinkIds = assetLinks.asSequence()
            .filter { it.removedAt != null }
            .map { it.id }
            .toSet()

        return getAssetContainer(assetLinkIdToAssetId, orderByAssetLinkId, removedAssetLinkIds)
    }

    fun getAssetContainer(
        assetIds: Set<String>,
        assetOrderById: Map<String, Int>
    ): AssetContainer {
        return getAssetContainer(assetIds, assetOrderById, emptySet())
    }

    fun getAssetContainer(
        assetIds: Set<String>,
        assetOrderById: Map<String, Int>,
        removedAssetIds: Set<String>,
    ): AssetContainer {
        return getAssetContainer(assetIds.associateBy { it }, assetOrderById, removedAssetIds)
    }

    fun getAssetContainer(
        assetLinkIdToAssetId: Map<String, String>,
        orderByAssetLinkId: Map<String, Int>,
        removedAssetLinkIds: Set<String>,
    ): AssetContainer {
        val assetByAssetId = getAssets(assetLinkIdToAssetId.values).associateBy { it.meta.id }
        val assetLinkIdToAsset = assetLinkIdToAssetId.asSequence()
            .map { it.key to assetByAssetId[it.value.toIdLong()] }
            .filter { it.second != null }
            .map { it.first to it.second!! }
            .toMap()
        val assetTypeToAssetLinkIds = assetLinkIdToAsset.map {
            it.value.meta.mediaType.toMediaType()!! to it.key
        }.groupBy({ it.first }, { it.second })

        return AssetContainer(
            texts = assetTypeToAssetLinkIds[MediaType.TEXT]?.map {
                val asset = assetLinkIdToAsset[it]!!
                LinkedAsset(orderByAssetLinkId[it]!!, removedAssetLinkIds.contains(it),
                    TextAsset(asset.meta.id.toIdString(), asset.meta.clientId.toIdString(), asset.spec.text))
            } ?: emptyList(),
            titles = assetTypeToAssetLinkIds[MediaType.TITLE]?.map {
                val asset = assetLinkIdToAsset[it]!!
                LinkedAsset(orderByAssetLinkId[it]!!, removedAssetLinkIds.contains(it),
                    TitleAsset(asset.meta.id.toIdString(), asset.meta.clientId.toIdString(), asset.spec.title))
            } ?: emptyList(),
            sitelinks = assetTypeToAssetLinkIds[MediaType.SITELINK]?.map {
                val asset = assetLinkIdToAsset[it]!!
                LinkedAsset(orderByAssetLinkId[it]!!, removedAssetLinkIds.contains(it),
                    asset.toSitelinkAsset())
            } ?: emptyList(),
            images = assetTypeToAssetLinkIds[MediaType.IMAGE]?.map {
                val asset = assetLinkIdToAsset[it]!!
                LinkedAsset(orderByAssetLinkId[it]!!, removedAssetLinkIds.contains(it),
                    asset.toImageAsset())
            } ?: emptyList(),
            videos = assetTypeToAssetLinkIds[MediaType.VIDEO]?.map {
                val asset = assetLinkIdToAsset[it]!!
                LinkedAsset(orderByAssetLinkId[it]!!, removedAssetLinkIds.contains(it),
                    asset.toVideoAsset())
            } ?: emptyList(),
            html5s = assetTypeToAssetLinkIds[MediaType.HTML5]?.map {
                val asset = assetLinkIdToAsset[it]!!
                LinkedAsset(orderByAssetLinkId[it]!!, removedAssetLinkIds.contains(it),
                    asset.toHtml5Asset())
            } ?: emptyList(),
        )
    }

    fun getAssets(assetIds: Collection<String>): List<TAsset> {
        if (assetIds.isEmpty()) {
            return emptyList()
        }
        return grutApiService.assetGrutApi.getAssets(assetIds.toIdsLong())
    }

    fun getAssetsByType(clientId: Long, assetIds: Collection<String>, mediaType: EMediaType): List<TAsset> {
        return grutApiService.assetGrutApi.selectAssets(
            filter = "[/meta/client_id] = $clientId " +
                "AND [/meta/id] IN (${assetIds.joinToString { "$it" }}) " +
                "AND [/meta/media_type] = ${mediaType.number}"
        )
    }

    fun updateAssets(assets: List<AssetPatch>) {
        if (assets.isEmpty()) {
            return
        }

        grutApiService.assetGrutApi.updateAssets(
            assets.map {
                UpdatedObject(
                    meta = TAssetMeta.newBuilder().setId(it.id.toIdLong()).build().toByteString(),
                    spec = it.spec.toByteString(),
                    setPaths = it.setPaths
                )
            }
        )
    }

    fun fillMediaContentsFromAssets(assets: List<MediaAsset>): List<Content> {
        return assets.map {
            fillContent(
                toUacYdbContent(it)
            )!!
        }
    }

    override fun insertContent(content: UacYdbContent): Result<Void> {
        grutApiService.assetGrutApi.createObject(toAssetGrut(content))
        return Result.successful(null)
    }

    override fun insertContents(contents: List<UacYdbContent>): Result<Void> {
        grutApiService.assetGrutApi.createObjects(contents.map { toAssetGrut(it) })
        return Result.successful(null)
    }

    override fun getDbContentByHash(hash: String): UacYdbContent? {
        val assets = grutApiService.assetGrutApi.selectAssets(
            filter = "[/spec/image/direct_image_hash] = \"$hash\"",
            index = "assets_by_direct_image_hash"
        )

        if (assets.isEmpty()) {
            return null
        }
        return toUacYdbContent(assets[0].toImageAsset())
    }

    override fun getDbContentsByDirectCampaignId(
        directCampaignId: Long,
        clientId: ClientId,
        mediaType: MediaType,
    ): Collection<UacYdbContent> {
        var assetLinks: List<AssetLink.TAssetLink>? = null
        if (featureService.isEnabledForClientId(clientId, UAC_MULTIPLE_AD_GROUPS_ENABLED)) {
            val adGroupBriefs = grutApiService.adGroupBriefGrutApi.selectTAdGroupBriefsByCampaignIds(
                campaignIds = setOf(directCampaignId),
                attributeSelector = listOf("/spec/brief_asset_links")
            )
            if (adGroupBriefs.isNotEmpty()) {
                assetLinks = adGroupBriefs.flatMap { it.spec.briefAssetLinks.linksList }
            }
        }
        if (assetLinks == null) {
            val campaignResult = grutApiService.briefGrutApi.getBrief(directCampaignId)

            check(campaignResult != null) {
                "Campaign $directCampaignId doesn't exists"
            }
            assetLinks = campaignResult.spec.briefAssetLinks.linksList
        }
        val assetIds = assetLinks!!.map { it.assetId.toIdString() }
        if (assetIds.isEmpty()) return listOf()
        return getAssetsByType(
            clientId.asLong(),
            assetIds,
            mediaType.toGrutMediaType()
        ).mapNotNull { toUacYdbContent(it) }
    }

    fun getAssetHashesByCampaignIds(campaignIds: Set<Long>): List<AssetHashes> {
        if (campaignIds.isEmpty()) {
            return listOf()
        }

        val banners = grutTransactionProvider.runRetryable(GRUT_READ_ATTEMPTS) {
            grutApiService.briefBannerGrutApi.selectBanners(
                filter = "[/meta/campaign_id] IN (${campaignIds.distinct().joinToString { "$it" }})" +
                    " AND [/spec/status] != \"${getProtoEnumValueName(BSS_DELETED)}\"",
                index = "banners_by_campaign",
                attributeSelector = listOf(
                    "/meta/id",
                    "/meta/campaign_id",
                    "/spec/asset_ids",
                    "/spec/asset_link_ids",
                )
            )
        }

        val allAssetIds = banners.flatMap { it.spec.assetIdsList.map { it.toIdString() } }.toSet()

        val campaignIdsWithAdGroupBriefEnabled = getCampaignIdsWithAdGroupBriefEnabled(campaignIds)
        val campaignIdsForCampaignBrief = (campaignIds - campaignIdsWithAdGroupBriefEnabled).toMutableSet()

        val campaignIdToAssetLinkIdToAssetLinkId: MutableMap<Long, Map<Long, Long>> = mutableMapOf()
        // Ассеты на групповых заявках
        if (campaignIdsWithAdGroupBriefEnabled.isNotEmpty()) {
            campaignIdToAssetLinkIdToAssetLinkId += getCampaignIdToAssetLinkIdToAssetLinkIdInAdGroupBrief(
                campaignIdsWithAdGroupBriefEnabled,
                allAssetIds,
            )
        }
        // Добавляем к списку id кампаний те у которых не нашли групповых заявок
        if (campaignIdToAssetLinkIdToAssetLinkId.size < campaignIdsWithAdGroupBriefEnabled.size) {
            campaignIdsForCampaignBrief += campaignIdsWithAdGroupBriefEnabled - campaignIdToAssetLinkIdToAssetLinkId.keys
        }
        // Ассеты на заявках кампаний
        if (campaignIdsForCampaignBrief.isNotEmpty()) {
            campaignIdToAssetLinkIdToAssetLinkId += getCampaignIdToAssetLinkIdToAssetLinkIdInCampaignBrief(
                campaignIdsForCampaignBrief,
                allAssetIds,
            )
        }

        val assetsById = grutTransactionProvider.runRetryable(GRUT_READ_ATTEMPTS) {
            getAssets(allAssetIds).associateBy { it.meta.id.toIdString() }
        }

        return banners.map { banner ->
            var titleId: String? = null
            var textId: String? = null
            var imageId: String? = null
            var videoId: String? = null
            var html5Id: String? = null
            val assetLinkIdToAssetId = campaignIdToAssetLinkIdToAssetLinkId[banner.meta.campaignId]!!
            val assetLinkIds = if (banner.spec.assetLinkIdsCount == 0 && banner.spec.assetIdsCount != 0) {
                banner.spec.assetIdsList
            } else {
                banner.spec.assetLinkIdsList
            }
            assetLinkIds.map { assetLinkId -> assetLinkId to assetLinkIdToAssetId[assetLinkId] }
                .forEach { (assetLinkId, assetId) ->
                    val asset = assetId?.let { assetsById[assetId.toIdString()] } ?: return@forEach
                    when {
                        asset.spec.hasTitle() -> {
                            titleId = assetLinkId.toIdString()
                        }
                        asset.spec.hasText() -> {
                            textId = assetLinkId.toIdString()
                        }
                        asset.spec.hasImage() -> {
                            imageId = assetLinkId.toIdString()
                        }
                        asset.spec.hasVideo() -> {
                            videoId = assetLinkId.toIdString()
                        }
                        asset.spec.hasHtml5() -> {
                            html5Id = assetLinkId.toIdString()
                        }
                    }
                }
            AssetHashes(
                bannerId = banner.meta.id,
                campaignId = banner.meta.campaignId,
                titleContentId = titleId,
                textContentId = textId,
                imageContentId = imageId,
                videoContentId = videoId,
                html5ContentId = html5Id,
            )
        }
    }

    /**
     * Возвращает мапу <campaign id, <asset id, asset asset id>> с поиском ассетов в заявках кампаний
     */
    private fun getCampaignIdToAssetLinkIdToAssetLinkIdInCampaignBrief(
        campaignIds: Set<Long>,
        allAssetIds: Set<String>,
    ): Map<Long, Map<Long, Long>> =
        grutTransactionProvider.runRetryable(GRUT_READ_ATTEMPTS) {
            grutApiService.briefGrutApi.getBriefs(
                campaignIds,
                attributeSelector = listOf("/meta/id", "/spec/brief_asset_links")
            ).asSequence()
                .map { campaign ->
                    campaign.meta.id to campaign.spec.briefAssetLinks.linksList.asSequence()
                        .filter { allAssetIds.contains(it.assetId.toIdString()) }
                        .map { assetLink ->
                            (if (assetLink.hasId()) assetLink.id else assetLink.assetId) to assetLink.assetId
                        }.toMap()
                }
                .toMap()
        }

    /**
     * Возвращает мапу <campaign id, <asset id, asset asset id>> с поиском ассетов в групповых заявках
     */
    private fun getCampaignIdToAssetLinkIdToAssetLinkIdInAdGroupBrief(
        campaignIds: Set<Long>,
        allAssetIds: Set<String>,
    ): Map<Long, Map<Long, Long>> = grutTransactionProvider.runRetryable(GRUT_READ_ATTEMPTS) {
        grutApiService.adGroupBriefGrutApi.selectTAdGroupBriefsByCampaignIds(
            campaignIds,
            attributeSelector = listOf("/meta/campaign_id", "/spec/brief_asset_links"),
        )
            .groupBy { it.meta.campaignId }
            .mapValues { (_, adGroupBriefs) ->
                adGroupBriefs
                    .map { adGroupBrief ->
                        adGroupBrief.spec.briefAssetLinks.linksList.asSequence()
                            .filter { allAssetIds.contains(it.assetId.toIdString()) }
                            .map { assetLink ->
                                (if (assetLink.hasId()) assetLink.id else assetLink.assetId) to assetLink.assetId
                            }
                            .toMap()
                    }
                    .flatMap { it.asSequence() }
                    .associateBy({ it.key }) { it.value }
            }
            .toMap()
    }

    /**
     * Возвращает id кампаний, для которых включены групповые заявки
     */
    private fun getCampaignIdsWithAdGroupBriefEnabled(campaignIds: Set<Long>): Set<Long> {
        val clientIdByDirectCampaignId = shardHelper.getClientIdsByCampaignIds(campaignIds)
        val allClients = clientIdByDirectCampaignId.values
            .map { ClientId.fromLong(it) }
            .toSet()

        val clientIdsWithAdGroupBriefEnabled =
            featureService.isEnabledForClientIds(allClients, UAC_MULTIPLE_AD_GROUPS_ENABLED)
                .filterValues { it == true }
                .mapKeys { it.key.asLong() }
                .keys
        return campaignIds
            .filter { clientIdsWithAdGroupBriefEnabled.contains(clientIdByDirectCampaignId[it]) }
            .toSet()
    }

    /**
     * Функция ищет ассеты по заданному типу в пределах [minLookupTime, maxLookupTime),
     * поддерживает пагинацию
     * @param minLookupTime - минимальное время создания ассета в микросекундах(включительно),
     * @param maxLookupTime - максимальное время создания ассета в микросекундах(не включительно)
     * @param mediaType - тип ассета
     * @param continuationToken - токен для получения следующего чанка
     * @param limit - размер чанка
     * @param index - индекс, по которому осуществляется сортировка
     */
    fun getAssetsUpperHalfCreationTimeInterval(
        minLookupTimeInMicros: Long,
        maxLookupTimeInMicros: Long,
        mediaType: EMediaType,
        continuationToken: String? = null,
        limit: Long? = null,
        index: String? = "assets_by_creation_time"
    ): UacGrutPage<List<UacYdbContent>> {
        val response = grutTransactionProvider.runRetryable(3) {
            grutContext.client.selectObjects(
                ObjectApiServiceOuterClass.TReqSelectObjects.newBuilder().apply {
                    filter = "[/meta/creation_time] >= $minLookupTimeInMicros AND " +
                        "[/meta/creation_time] < $maxLookupTimeInMicros AND " +
                        "[/meta/media_type] = ${mediaType.number}"
                    objectType = Schema.EObjectType.OT_ASSET
                    addAllAttributeSelector(listOf("/meta", "/spec"))
                    limit?.let { this.limit = limit }
                    index?.let { this.index = index }
                    continuationToken?.let { this.continuationToken = it }
                    // Index has hash_expression column for balancing load across shards.
                    // Using inequality operators in filter does not allow calculate
                    // hash_expression. Therefore, request marked as full scan.
                    allowFullScan = index != null
                }.build()
            )
        }

        val newToken = response.continuationToken
        val content = response.payloadsList
            .mapNotNull { TAsset.parseFrom(it.protobuf) }
            .map { toUacYdbContent(it)!! }

        return UacGrutPage(content, newToken)
    }
}
