package ru.yandex.direct.core.entity.uac.service

import java.net.URI
import java.net.URISyntaxException
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.avatars.client.exception.AvatarsClientCommonException

data class AssetIdToThumb(
    val id: String,
    val thumb: String,
)

@Lazy
@Service
class GrutUacDeleteImageContentService(
    private val grutUacContentService: GrutUacContentService,
    private val uacAvatarsService: UacAvatarsService,
) {

    private val logger = LoggerFactory.getLogger(GrutUacDeleteImageContentService::class.java)

    fun deleteUnusedAsset(
        assetId: String,
        avatarId: String,
        idleTime: Long = 1000L
    ): Boolean {
        var retryCount = 3
        while (true) {
            val mdsDeleteResult = try {
                uacAvatarsService.deleteImage(avatarId)
            } catch (e: AvatarsClientCommonException) {
                logger.error("Error occurred in MDS service on delete image", e)
                false
            }
            if (mdsDeleteResult) {
                val result = grutUacContentService.deleteContent(assetId)
                if (result.isSuccessful) {
                    logger.info("Asset $assetId fully deleted")
                } else {
                    logger.warn("Avatar $avatarId was removed, but asset $assetId will be removed later")
                }
                return true
            }

            if (--retryCount <= 0) {
                break
            } else {
                logger.info("Unable delete asset with id: $assetId. Next try in $idleTime milliseconds")
                Thread.sleep(idleTime)
                continue
            }
        }

        logger.error("Fatal error! Failed to delete $assetId with avatarId $avatarId")
        return false
    }

    fun uacThumbToAvatarId(thumb: String, delimiter: String): String? {
        return try {
            val uri = URI(thumb)
            val splitPath = uri.path.split(delimiter)
            return if (splitPath.size < 4) null
            else splitPath[1] + delimiter + splitPath[2] + delimiter + splitPath[3]
        } catch (ex: URISyntaxException) {
            null
        }
    }
}
