package ru.yandex.direct.core.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.campaign.service.RestrictedCampaignsAddOperation
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppAddOperation
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser
import ru.yandex.direct.core.entity.uac.converter.UacCampaignConverter.toCampaignsPlatform
import ru.yandex.direct.core.entity.uac.model.TargetType
import ru.yandex.direct.core.entity.uac.model.UacStrategyName
import ru.yandex.direct.core.entity.uac.model.UpdateUacCampaignRequest
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.moneyFromDb
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.result.Result
import java.util.Optional

@Service
class RmpCampaignService(
    private val shardHelper: ShardHelper,
    private val mobileAppService: MobileAppService,
    private val campaignOperationService: CampaignOperationService,
    private val aggregatedStatusesViewService: AggregatedStatusesViewService,
    private val campaignService: CampaignService,
    private val uacCampaignsCoreService: UacCampaignsCoreService,
    private val featureService: FeatureService,
    private val uacDisabledDomainsService: UacDisabledDomainsService,
) {

    /**
     * Метод добавляем РМП кампанию, предварительно добавив приложение.
     * Если такое приложение у пользователя уже есть, то используется оно, вместо добавления нового.
     * @return Результат с добавленной кампанией при успехе, иначе ошибки валидации кампании либо приложения
     */
    fun addRmpCampaign(
        operator: User,
        uidAndClientId: UidAndClientId,
        campaign: MobileContentCampaign,
        mobileApp: MobileApp
    ): Result<MobileContentCampaign> {
        val existingMobileAppId = campaign.mobileAppId ?: findExistingMobileAppId(uidAndClientId.clientId, mobileApp)

        val campaignOperation: Result<RestrictedCampaignsAddOperation>
        if (existingMobileAppId == null) {
            val mobileAppAddOperation = prepareMobileApp(operator, uidAndClientId.clientId, mobileApp)
            if (!mobileAppAddOperation.isSuccessful) {
                return Result.broken(mobileAppAddOperation.validationResult)
            }

            campaignOperation = prepareCampaign(operator, uidAndClientId, campaign)
            if (!campaignOperation.isSuccessful) {
                return Result.broken(campaignOperation.validationResult)
            }

            val mobileAppResult = mobileAppAddOperation.result!!.apply()
            if (!mobileAppResult.isSuccessful) {
                return Result.broken(mobileAppResult.validationResult)
            }

            campaign
                .withMobileAppId(mobileAppResult[0].result)
        } else {
            campaign
                .withMobileAppId(existingMobileAppId)
            campaignOperation = prepareCampaign(operator, uidAndClientId, campaign)
            if (!campaignOperation.isSuccessful) {
                return Result.broken(campaignOperation.validationResult)
            }
        }

        val campaignResult = campaignOperation.result!!.apply()
        if (!campaignResult.isSuccessful) {
            return Result.broken(campaignResult.validationResult)
        }

        return Result.successful(campaign)
    }

    fun getMobileContentCampaign(clientId: ClientId, campaignId: Long): MobileContentCampaign? {
        return uacCampaignsCoreService.getCampaign(clientId, campaignId, MobileContentCampaign::class)
    }

    fun getCampaignStates(clientId: ClientId, campaign: MobileContentCampaign): UacCampaignStates {
        val shard = shardHelper.getShardByClientId(clientId)
        return getCampaignStates(shard, campaign, false)
    }

    private fun getMobileAppByUrl(clientId: ClientId, storeHref: String): MobileApp? {
        return getMobileAppsByUrl(clientId, storeHref).firstOrNull()
    }

    fun getMobileAppsByUrl(clientId: ClientId, storeHref: String): List<MobileApp> {
        val mobileAppStoreUrl = MobileAppStoreUrlParser.parseStrict(storeHref)
        return mobileAppService.getMobileApps(clientId)
            .filter { app -> MobileAppStoreUrlParser.parseStrict(app.storeHref) == mobileAppStoreUrl }
    }

    fun getMobileAppsByAppId(clientId: ClientId, mobileAppId: Long): List<MobileApp> {
        val mobileApp = mobileAppService.getMobileApp(clientId, mobileAppId)
        if (mobileApp.isEmpty) {
            return listOf()
        }
        return getMobileAppsByUrl(clientId, mobileApp.get().storeHref)
    }

    //параметр bid - временный костыль, нужный для корректного получения stateReasons в том числе и через баннер
    // (для не рмп кампаний)
    fun getCampaignStates(
        clientId: ClientId,
        campaignId: Long,
        withAdGroupsStateReasons: Boolean,
        bid: Long? = null
    ): UacCampaignStates? {
        val shard = shardHelper.getShardByClientId(clientId)
        val campaign = uacCampaignsCoreService.getCampaign(shard, campaignId, CommonCampaign::class) ?: return null
        return getCampaignStates(shard, campaign, withAdGroupsStateReasons, bid)
    }

    // параметр withAdGroupsStateReasons нужен на время, пока статусы вычисляются python бекэндом
    // и в intapi ручке возвращаются отдельно stateResasons для кампании и отдельно на группе,
    // а уже в python объединяются
    // https://a.yandex-team.ru/arc/trunk/arcadia/yabs/rmp/backend/src/uac/campaign/updaters.py?rev=r8272719#L142
    // когда останется только kotlin бекэнд можно всегда вычислять stateReasons по группу и по кампании
    private fun getCampaignStates(
        shard: Int, campaign: CommonCampaign,
        withAdGroupsStateReasons: Boolean,
        bid: Long? = null
    ): UacCampaignStates {
        val aggrStatuses = aggregatedStatusesViewService.getCampaignStatusesByIds(shard, setOf(campaign.id))

        val status = aggrStatuses[campaign.id]
        val campaignState = status?.let { toCampaignState(it.statusUnsafe) }
        val moderationState = toModerationState(campaign.statusModerate)

        val resultStateReasons = mutableSetOf<GdSelfStatusReason>()
        status?.reasonsUnsafe?.let { resultStateReasons.addAll(it) }
        if (withAdGroupsStateReasons) {
            val aggrAdGroupStates = aggregatedStatusesViewService
                .getAdGroupStatusesByCampaignId(shard, ClientId.fromLong(campaign.clientId), campaign.id)
                .values
                .flatMap { it.reasonsUnsafe }
            resultStateReasons.addAll(aggrAdGroupStates)
        }
        //получение статусов по баннеру
        if (bid != null) {
            val aggrAdStates = aggregatedStatusesViewService
                .getAdStatusesByIds(shard, setOf(bid))
                .values
                .flatMap { it.reasonsUnsafe }
            resultStateReasons.addAll(aggrAdStates)
        }

        return UacCampaignStates(
            moderationState,
            campaignState,
            resultStateReasons.toList(),
            campaign.statusShow,
            status != null && status.isObsolete
        )
    }

    private fun findExistingMobileAppId(clientId: ClientId, mobileApp: MobileApp): Long? {
        if (mobileApp.id != null) {
            return mobileApp.id
        }
        return getMobileAppByUrl(clientId, mobileApp.storeHref)?.id
    }

    private fun prepareCampaign(
        operator: User,
        uidAndClientId: UidAndClientId,
        campaign: MobileContentCampaign,
    ): Result<RestrictedCampaignsAddOperation> {
        val options = CampaignOptions.Builder()
            .withSkipValidateMobileApp(true)
            .build()
        val addOperation = campaignOperationService.createRestrictedCampaignAddOperation(
            listOf(campaign),
            operator.uid,
            uidAndClientId,
            options
        )
        val result: Optional<MassResult<Long>> = addOperation.prepare()
        if (result.isPresent) {
            return Result.broken(result.get().validationResult)
        }
        return Result.successful(addOperation)
    }

    private fun prepareMobileApp(user: User, clientId: ClientId, mobileApp: MobileApp): Result<MobileAppAddOperation> {
        val addOperation = mobileAppService.createAddFullOperation(user, clientId, listOf(mobileApp))
        val result: Optional<MassResult<Long>> = addOperation.prepare()
        if (result.isPresent) {
            return Result.broken(result.get()[0].validationResult)
        }
        return Result.successful(addOperation)
    }

    fun updateUacCampaignInDirect(
        client: Client,
        operator: User,
        campaignChanges: List<ModelChanges<out BaseCampaign>>,
    ): Result<*> {
        val options = CampaignOptions()
        val campaignResult = campaignOperationService.createRestrictedCampaignUpdateOperation(
            campaignChanges,
            operator.uid,
            UidAndClientId.of(client.chiefUid, ClientId.fromLong(client.clientId)),
            options
        ).apply()

        return if (campaignResult.errorCount != 0) {
            campaignResult[0]
        } else {
            campaignResult
        }
    }

    fun suspendResumeCampaignInDirect(
        campaignId: Long,
        resume: Boolean,
        operatorUid: Long,
        clientId: ClientId
    ): Result<*> {
        return if (resume) {
            campaignService.resumeCampaigns(listOf(campaignId), operatorUid, clientId)
        } else {
            campaignService.suspendCampaigns(listOf(campaignId), operatorUid, clientId)
        }
    }

    fun updateRmpCampaign(
        client: Client,
        operator: User,
        campaign: MobileContentCampaign,
        request: UpdateUacCampaignRequest,
    ): Result<*> {
        return updateUacCampaignInDirect(
            client,
            operator,
            createMobileContentCampaignChanges(campaign, request, ClientId.fromLong(client.clientId)),
        )
    }

    private fun createMobileContentCampaignChanges(
        campaign: MobileContentCampaign,
        request: UpdateUacCampaignRequest,
        clientId: ClientId,
    ): List<ModelChanges<MobileContentCampaign>> {

        val disabledDomains = uacDisabledDomainsService.getDisabledDomains(
            request.uacDisabledPlaces?.disabledPlaces.orEmpty()
        )
        val changes = ModelChanges(campaign.id, MobileContentCampaign::class.java)
            .processNotNull(request.name, MobileContentCampaign.NAME)
            .processNotNull(request.statusShow, MobileContentCampaign.STATUS_SHOW)
            .processNotNull(request.adultContentEnabled, MobileContentCampaign.IS_ALLOWED_ON_ADULT_CONTENT)
            .processNotNull(disabledDomains, MobileContentCampaign.DISABLED_DOMAINS)
            .processNotNull(request.alternativeAppStores, MobileContentCampaign.ALTERNATIVE_APP_STORES)
        val strategy = campaign.strategy

        if (request.weekLimit != null) {
            strategy.strategyData
                .withSum(moneyFromDb(request.weekLimit))
        }

        if (request.strategyPlatform != null) {
            strategy.withPlatform(toCampaignsPlatform(request.strategyPlatform))
        }

        if (request.strategy != null) {
            strategy.strategyName = request.strategy.uacStrategyName.toDbStrategyName()

            strategy.strategyData
                .withName(StrategyName.toSource(strategy.strategyName)!!.literal)
                .withPayForConversion(request.strategy.uacStrategyData.payForConversion)

            if (request.targetId != null && request.targetId != TargetType.CPC) {
                strategy.strategyData
                    .withGoalId(request.targetId.goalId)
            }

            strategy.strategyData
                .withAvgBid(null)
                .withAvgCpa(null)
                .withAvgCpi(null)
                .withCrr(null)

            when (request.strategy.uacStrategyName) {
                UacStrategyName.AUTOBUDGET_CRR -> {
                    strategy.strategyData
                        .withCrr(request.strategy.uacStrategyData.crr)
                }
                UacStrategyName.AUTOBUDGET_AVG_CPA -> {
                    strategy.strategyData
                        .withAvgCpa(request.strategy.uacStrategyData.avgCpa)
                }
                UacStrategyName.AUTOBUDGET_AVG_CPI -> {
                    strategy.strategyData
                        .withAvgCpi(request.strategy.uacStrategyData.avgCpi)
                }

                UacStrategyName.AUTOBUDGET_AVG_CLICK -> {
                    strategy.strategyData
                        .withAvgBid(request.strategy.uacStrategyData.avgBid)
                }
                else -> throw IllegalStateException(
                    "Received illegal strategy name: ${request.strategy.uacStrategyName}"
                )
            }

            return listOf(changes.process(strategy, MobileContentCampaign.STRATEGY))
        }
        // Everything below should be removed after migration RMP-2996
        val cost = if (request.cpa != null) {
            moneyFromDb(request.cpa)
        } else if (strategy.strategyData.avgCpa != null) {
            strategy.strategyData.avgCpa
        } else {
            strategy.strategyData.avgCpi
        }

        if (request.targetId != null) {
            val strategyName = if (request.targetId == TargetType.CPC || campaign.isSkadNetworkEnabled) {
                StrategyName.AUTOBUDGET_AVG_CLICK
            } else if (featureService.isEnabledForClientId(clientId, FeatureName.UAC_FIX_CPA_STRATEGY_ENABLED)) {
                StrategyName.AUTOBUDGET_AVG_CPA
            } else {
                StrategyName.AUTOBUDGET_AVG_CPI
            }

            strategy.strategyName = strategyName
            if (request.targetId != TargetType.CPC) {
                strategy.strategyData.withGoalId(request.targetId.goalId)
            }

            when (strategy.strategyName) {
                StrategyName.AUTOBUDGET_AVG_CPA -> strategy.strategyData
                    .withPayForConversion(true)
                    .withGoalId(TargetType.INSTALL.goalId)
                StrategyName.AUTOBUDGET_AVG_CPI -> strategy.strategyData
                    .withPayForConversion(request.targetId == TargetType.INSTALL)
                StrategyName.AUTOBUDGET_AVG_CLICK -> strategy.strategyData
                    .withPayForConversion(false)
                else -> {}
            }

            strategy.strategyData.withName(StrategyName.toSource(strategy.strategyName)!!.literal)
        }

        if (request.cpa != null) {
            when (strategy.strategyName) {
                StrategyName.AUTOBUDGET_AVG_CPA -> strategy.strategyData
                    .withAvgCpi(null)
                    .withAvgCpa(cost)
                    .withAvgBid(null)
                StrategyName.AUTOBUDGET_AVG_CPI -> strategy.strategyData
                    .withAvgCpi(cost)
                    .withAvgCpa(null)
                    .withAvgBid(null)
                StrategyName.AUTOBUDGET_AVG_CLICK -> strategy.strategyData
                    .withAvgCpi(null)
                    .withAvgCpa(null)
                    .withAvgBid(cost)
                else -> {}
            }
        }

        return listOf(changes.process(strategy, MobileContentCampaign.STRATEGY))
    }

    fun deleteCampaign(
        clientId: ClientId,
        operatorUid: Long,
        campaignId: Long,
    ): MassResult<Long> = campaignService.deleteCampaigns(listOf(campaignId), operatorUid, clientId, Applicability.FULL)
}
