package ru.yandex.direct.core.entity.uac.service

import com.google.common.collect.ImmutableList
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup
import ru.yandex.direct.core.entity.adgroup.container.ComplexDynamicAdGroup
import ru.yandex.direct.core.entity.adgroup.container.ComplexMobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.container.ComplexPerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService
import ru.yandex.direct.core.entity.adgroup.service.ModerationMode
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateOperationFactory
import ru.yandex.direct.core.entity.adgroup.service.complex.cpm.ComplexCpmAdGroupAddOperation
import ru.yandex.direct.core.entity.adgroup.service.complex.mobilecontent.ComplexMobileContentAdGroupAddOperation
import ru.yandex.direct.core.entity.adgroup.service.complex.text.ComplexTextAdGroupAddOperation
import ru.yandex.direct.core.entity.banner.container.ComplexBanner
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.dynamictextadtarget.container.DynamicTextAdTargetSelectionCriteria
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilter
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterCondition
import ru.yandex.direct.core.entity.keyword.model.Keyword
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository
import ru.yandex.direct.core.entity.keyword.service.KeywordUtils
import ru.yandex.direct.core.entity.keyword.service.validation.ClientKeywordCommonValidator
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService
import ru.yandex.direct.core.entity.offerretargeting.model.OfferRetargeting
import ru.yandex.direct.core.entity.offerretargeting.repository.OfferRetargetingRepository
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchRepository
import ru.yandex.direct.core.entity.retargeting.Constants
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService
import ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils
import ru.yandex.direct.core.entity.uac.UacCommonUtils
import ru.yandex.direct.core.entity.uac.converter.GroupRetargetingConditionConverter.toGroupRetargetingCondition
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.defaultAvailableDynamicFeedRule
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.defaultAvailableFeedFilterCondition
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.defaultAvailablePerformanceCondition
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.defaultDynamicFilter
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.defaultFeedFilter
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.defaultSmartFilter
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toDynamicFilters
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toFeedFilter
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toSmartFilters
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toUacFeedFiltersConditions
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.UacComplexBidModifier
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentType
import ru.yandex.direct.core.entity.uac.model.relevance_match.toRelevanceMatchCategory
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAppInfoRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.UacGeoService.getGeoForUacGroups
import ru.yandex.direct.core.grut.GrutUtils.Companion.getDatabaseMode
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType.interests
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.regions.GeoTree
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.CommonUtils
import java.time.Duration
import java.util.Objects

@Service
@Lazy
class UacAdGroupService(
    private val clientGeoService: ClientGeoService,
    private val adGroupService: AdGroupService,
    private val complexAdGroupAddOperationFactory: ComplexAdGroupAddOperationFactory,
    private val complexAdGroupUpdateOperationFactory: ComplexAdGroupUpdateOperationFactory,
    private val bannersAddOperationFactory: BannersAddOperationFactory,
    private val mobileAppService: MobileAppService,
    private val cpmBannerCampaignService: CpmBannerCampaignService,
    private val feedService: FeedService,
    private val performanceFilterService: PerformanceFilterService,
    private val dynamicTextAdTargetService: DynamicTextAdTargetService,
    private val retargetingConditionRepository: RetargetingConditionRepository,
    private val targetingCategoriesCache: TargetingCategoriesCache,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    private val shardHelper: ShardHelper,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val performanceFilterStorage: PerformanceFilterStorage,
    private val uacYdbAppInfoRepository: UacYdbAppInfoRepository,
    private val uacAppInfoService: UacAppInfoService,
    private val featureService: FeatureService,
    private val retargetingService: RetargetingService,
    val keywordRepository: KeywordRepository,
    val relevanceMatchRepository: RelevanceMatchRepository,
    val offerRetargetingRepository: OfferRetargetingRepository,
    val retargetingConditionService: RetargetingConditionService,
) {
    private val targetInterestsEnabled: PpcProperty<Boolean> = ppcPropertiesSupport
        .get(PpcPropertyNames.UAC_TARGET_INTERESTS_ENABLED, Duration.ofMinutes(5))

    companion object {
        private const val DEFAULT_FILTER_NAME = "filter"

        private val logger = LoggerFactory.getLogger(UacAdGroupService::class.java)
    }

    fun getAdGroupTypesByIds(clientId: ClientId, adGroupIds: Collection<Long>) =
        adGroupService.getAdGroupTypes(clientId, adGroupIds)

    fun updateMobileContentAdGroups(
        client: Client,
        operatorUid: Long,
        adGroupIds: List<Long>,
        regionIds: List<Long>?,
        minusRegionIds: List<Long>?,
        keywords: List<Keyword>?,
        minusKeywords: List<String>?,
        retargetingCondition: RetargetingCondition?,
        campaign: CommonCampaign?,
        appInfo: AppInfo?,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)

        val complexAdGroups: List<ComplexMobileContentAdGroup> = fetchComplexAdGroups(clientId, adGroupIds)
        val targetInterestsIds = if (!targetInterestsEnabled.getOrDefault(false)) setOf()
        else appInfo?.interests?.map { targetingCategoriesCache.getCategoryByImportId(it.toBigInteger()) }?.toSet()
            ?: setOf()
        val geo = getGeoForUacGroups(regionIds, minusRegionIds)

        complexAdGroups.forEach { complexAdGroup ->
            val existedTargetInterestIds = complexAdGroup.targetInterests
                ?.mapNotNull { it.interestId }
                ?.toSet()
                ?: setOf()
            val targetInterests: MutableList<TargetInterest> = mutableListOf()
            if (retargetingCondition != null) {
                targetInterests.add(
                    TargetInterest()
                        .withRetargetingConditionId(retargetingCondition.id)
                        .withCampaignId(campaign?.id)
                )
            }
            if (isLalOrNullRetargeting(retargetingCondition)) {
                val existedTargetInterests = targetInterestsIds
                    .filter { !existedTargetInterestIds.contains(it) }
                    .map {
                        TargetInterest()
                            .withInterestId(it)
                            .withCampaignId(campaign?.id)
                    }
                targetInterests.addAll(
                    complexAdGroup.targetInterests
                        ?.filter { targetInterestsIds.contains(it.interestId) }
                        ?.plus(existedTargetInterests)
                        ?: listOf()
                )
            }
            val relevanceMatch = if (isLalOrNullRetargeting(retargetingCondition) && complexAdGroup.relevanceMatches != null) {
                complexAdGroup.relevanceMatches
            } else if (isLalOrNullRetargeting(retargetingCondition)) {
                listOf(
                    RelevanceMatch()
                        .withAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY)
                )
            } else {
                listOf()
            }
            geo.apply { complexAdGroup.adGroup.geo = geo }
            if (keywords != null) {
                complexAdGroup.keywords = KeywordUtils.mergeNewKeywordsWithExisting(keywords, complexAdGroup.keywords)
            } else {
                complexAdGroup.keywords = null
            }
            complexAdGroup.adGroup.minusKeywords = minusKeywords
            targetInterests.apply { complexAdGroup.targetInterests = targetInterests }
            relevanceMatch.apply { complexAdGroup.relevanceMatches = relevanceMatch }
        }
        val geoTree = clientGeoService.getClientTranslocalGeoTree(clientId)

        val operation = complexAdGroupUpdateOperationFactory.createMobileContentAdGroupUpdateOperation(
            complexAdGroups,
            geoTree,
            false,
            null,
            operatorUid,
            clientId,
            client.chiefUid,
            false,
        )

        return operation.prepareAndApply()
    }

    private fun fetchComplexAdGroups(
        clientId: ClientId, adGroupIds: List<Long>
    ): List<ComplexMobileContentAdGroup> {
        val shard = shardHelper.getShardByClientId(clientId)

        val adGroups: List<AdGroup> = adGroupService.getAdGroups(clientId, adGroupIds)

        val keywordsByAdGroupId: Map<Long, List<Keyword>> =
            keywordRepository.getKeywordsByAdGroupIds(shard, clientId, adGroupIds)

        val relevanceMatchesByAdGroupId: Map<Long, RelevanceMatch> =
            relevanceMatchRepository.getRelevanceMatchesByAdGroupIds(shard, clientId, adGroupIds)

        val targetInterests: Map<Long, List<TargetInterest>> =
            retargetingService.getTargetInterestsWithInterestByAdGroupIds(
                adGroupIds,
                clientId,
                shard
            ).groupBy { it.adGroupId }

        return adGroups.map { adGroup ->
            ComplexMobileContentAdGroup()
                .withAdGroup(adGroup as MobileContentAdGroup)
                .withKeywords(keywordsByAdGroupId[adGroup.id])
                .withRelevanceMatches(
                    relevanceMatchesByAdGroupId[adGroup.id]?.let { listOf(it) }
                )
                .withTargetInterests(targetInterests[adGroup.id])
        }
    }

    fun updateAdGroups(
        adGroupIds: List<Long>,
        groupTypeById: Map<Long, AdGroupType>,
        operatorUid: Long,
        client: Client,
        brief: UacAdGroupBrief,
        keywords: List<Keyword>?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        campaign: CommonCampaign,
    ): MassResult<Long> {
        return updateAdGroupsInternal(
            adGroupIds,
            groupTypeById,
            operatorUid,
            client,
            brief,
            keywords,
            complexBidModifier,
            retargetingCondition,
            campaign,
        )
    }

    private fun updateAdGroupsInternal(
        adGroupIds: List<Long>,
        groupTypeById: Map<Long, AdGroupType>,
        operatorUid: Long,
        client: Client,
        brief: UacAdGroupBrief,
        keywords: List<Keyword>?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        campaign: CommonCampaign,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)
        when (campaign.type) {
            CampaignType.MOBILE_CONTENT -> {
                val ydbAppInfo = uacYdbAppInfoRepository.getAppInfoById(brief.appId!!)
                val appInfo = uacAppInfoService.getAppInfo(ydbAppInfo!!)
                return updateMobileContentAdGroups(
                    client, operatorUid,
                    adGroupIds, brief.regions, brief.minusRegions,
                    keywords, brief.minusKeywords, retargetingCondition, campaign, appInfo
                )
            }
            CampaignType.TEXT -> {
                val ecomUcNewBackend = brief.isEcom == true && featureService.isEnabledForClientId(clientId,
                    FeatureName.ECOM_UC_NEW_BACKEND_ENABLED)

                val relevanceMatch = RelevanceMatch()
                    .withAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY)
                val relevanceMatchCategories = getRelevanceMatchCategory(brief)
                val filter = if (ecomUcNewBackend) {
                    createFeedFilter(client, brief)
                } else {
                    null
                }

                val complexTextAdGroupTemplateProvider = {
                    ComplexTextAdGroup()
                        .withKeywords(keywords)
                        .withComplexBidModifier(complexBidModifier)
                        .withTargetInterests(retargetingCondition?.let { listOf(it.targetInterest) })
                        .withRetargetingCondition(retargetingCondition)
                        .withRelevanceMatches(listOf(relevanceMatch))
                        .withOfferRetargetings(listOf(OfferRetargeting()).takeIf { ecomUcNewBackend })
                        .withAdGroup(
                            TextAdGroup()
                                .withName(brief.name)
                                .withType(AdGroupType.BASE)
                                .withFeedFilter(filter)
                                .withFeedId(brief.feedId.takeIf { ecomUcNewBackend })
                                .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
                                .withHyperGeoId(brief.hyperGeoId)
                        )
                }
                return updateTextAdGroups(
                    clientId, client.chiefUid, operatorUid,
                    adGroupIds, relevanceMatchCategories,
                    complexTextAdGroupTemplateProvider,
                    retargetingCondition,
                )
            }
            CampaignType.CPM_BANNER -> {
                return updateCpmBannerGroup(
                    client,
                    operatorUid,
                    adGroupIds,
                    groupTypeById,
                    retargetingCondition,
                    campaign as CpmBannerCampaign,
                    brief
                )
            }
            CampaignType.DYNAMIC -> {
                val complexGroupTemplate = ComplexDynamicAdGroup()
                    .withComplexBidModifier(complexBidModifier)
                    .withAdGroup(
                        AdGroup()
                            .withName(brief.name)
                            .withType(AdGroupType.DYNAMIC)
                            .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
                            .withHyperGeoId(brief.hyperGeoId)
                            .withTrackingParams(brief.trackingParams)
                    )
                return updateDynamicAdGroups(client, operatorUid, adGroupIds, complexGroupTemplate, brief)
            }
            CampaignType.PERFORMANCE -> {
                val complexGroupTemplate = ComplexPerformanceAdGroup()
                    .withComplexBidModifier(complexBidModifier)
                    .withAdGroup(
                        AdGroup()
                            .withName(brief.name)
                            .withType(AdGroupType.PERFORMANCE)
                            .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
                            .withHyperGeoId(brief.hyperGeoId)
                            .withTrackingParams(brief.trackingParams)
                    )
                return updatePerformanceAdGroups(client, operatorUid, adGroupIds, complexGroupTemplate, brief)
            }
            else -> {
                throw IllegalStateException("Unsupported campaign type: ${campaign.type}")
            }
        }
    }

    private fun updateCpmBannerGroup(
        client: Client,
        operatorUid: Long,
        adGroupIds: Collection<Long>,
        groupTypeById: Map<Long, AdGroupType>,
        retargetingCondition: RetargetingCondition?,
        campaign: CpmBannerCampaign,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val shard = this.shardHelper.getShardByClientId(ClientId.fromLong(client.clientId))
        val retargetingConditions =
            retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, adGroupIds)
        val clientId = ClientId.fromLong(client.clientId)
        val targetInterests: Map<Long, List<TargetInterest>> =
            retargetingService.getTargetInterestsWithInterestByAdGroupIds(
                adGroupIds,
                clientId,
                shard
            ).groupBy { it.retargetingConditionId }

        val uacComplexBidModifier = UacComplexBidModifier(
            advType = brief.advType,
            socdem = brief.socdem,
            deviceTypes = brief.deviceTypes,
            inventoryTypes = null,
            isSmartTVEnabled = featureService.isEnabledForClientId(clientId, FeatureName.SMARTTV_BID_MODIFIER_ENABLED),
            isTabletModifierEnabled = featureService.isEnabledForClientId(
                clientId, FeatureName.TABLET_BIDMODIFER_ENABLED
            ),
        )

        // Для каждой группы создаем новые корректировки, иначе при добавлении корректировок в базу,
        // они будут добавляться к одной из групп, а не к каждой отдельно
        val groups = adGroupIds.map { groupid ->
            cpmBannerCampaignService.complexGroupFromUac(
                groupid, campaign, UacCommonUtils.getComplexBidModifier(uacComplexBidModifier, campaign.id),
                toGroupRetargetingCondition(groupid, retargetingCondition, retargetingConditions, targetInterests),
                null, brief, groupTypeById[groupid] ?: AdGroupType.CPM_BANNER,
                getGeoForUacGroups(brief.regions, brief.minusRegions)
            )
        }

        return cpmBannerCampaignService.updateCpmBannerGroups(groups, null, operatorUid, client)
    }

    private fun updateTextAdGroups(
        clientId: ClientId,
        clientUid: Long,
        operatorUid: Long,
        adGroupIds: List<Long>,
        relevanceMatchCategories: Set<RelevanceMatchCategory>,
        complexAdGroupTemplateProvider: () -> ComplexTextAdGroup,
        retargetingCondition: RetargetingCondition?,
    ): MassResult<Long> {
        val shard = shardHelper.getShardByClientId(clientId)
        val keywordsByAdGroupId = keywordRepository.getKeywordsByAdGroupIds(shard, adGroupIds)
        val existingRelMatchesByAdGroupId: Map<Long, RelevanceMatch> =
            relevanceMatchRepository.getRelevanceMatchesByAdGroupIds(shard, clientId, adGroupIds)
        val enableRelevanceMatchCategories =
            featureService.isEnabledForClientId(clientId, FeatureName.RELEVANCE_MATCH_CATEGORIES_ALLOWED_IN_UC)
        val existingOfferRetargetingsByAdGroupId: Map<Long, OfferRetargeting> =
            offerRetargetingRepository.getOfferRetargetingsByAdGroupIds(shard, clientId, adGroupIds)
        val ucCustomAudienceEnabled = featureService.isEnabledForClientId(clientId, FeatureName.UC_CUSTOM_AUDIENCE_ENABLED)

        val interestsByAdGroupId = getInterestsByAdGroupId(shard, clientId, adGroupIds, ucCustomAudienceEnabled)
        val interestsConditionIds = getInterestConditionIds(clientId, adGroupIds, ucCustomAudienceEnabled)

        // У ТГО может быть только одна группа
        val complexAdGroups = adGroupIds.map { groupId ->
            if (ucCustomAudienceEnabled && retargetingCondition != null) {
                val targetInterest = RetargetingUtils.getTargetInterestWithInterestsCondition(
                    interestsConditionIds,
                    groupId,
                    interestsByAdGroupId[groupId],
                )

                retargetingCondition.id = targetInterest.retargetingConditionId
                retargetingCondition.targetInterest = targetInterest
            }

            copyComplexTextAdGroup(
                groupId, complexAdGroupTemplateProvider, keywordsByAdGroupId, existingRelMatchesByAdGroupId,
                existingOfferRetargetingsByAdGroupId, relevanceMatchCategories, enableRelevanceMatchCategories,
                retargetingCondition,
            )
        }
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)

        val operation = complexAdGroupUpdateOperationFactory.createTextAdGroupUpdateOperation(
            complexAdGroups, geoTree, false, null,
            ucCustomAudienceEnabled, operatorUid, clientId, clientUid, false
        )

        return operation.prepareAndApply()
    }

    private fun getInterestConditionIds(
        clientId: ClientId,
        adGroupIds: List<Long>,
        ucCustomAudienceEnabled: Boolean,
    ): Set<Long> =
        if (!ucCustomAudienceEnabled) emptySet()
        else retargetingConditionService.getRetargetingConditions(
            clientId, null, adGroupIds, null, setOf(interests), LimitOffset.limited(adGroupIds.size))
            .map { it.id }
            .toSet()

    private fun getInterestsByAdGroupId(
        shard: Int,
        clientId: ClientId,
        adGroupIds: List<Long>,
        ucCustomAudienceEnabled: Boolean,
    ): Map<Long, List<TargetInterest>> =
        if (!ucCustomAudienceEnabled) emptyMap()
        else retargetingService.getTargetInterestsWithInterestByAdGroupIds(adGroupIds, clientId, shard)
            .groupBy { it.adGroupId }

    fun updateDynamicAdGroups(
        client: Client,
        operatorUid: Long,
        adGroupIds: List<Long>,
        complexGroupTemplate: ComplexDynamicAdGroup,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)
        val complexGroups = adGroupIds
            .map { copyComplexDynamicAdGroup(it, complexGroupTemplate) }
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        recreateDynamicFilters(client, operatorUid, adGroupIds, brief)
        val operation = complexAdGroupUpdateOperationFactory
            .createDynamicAdGroupUpdateOperation(complexGroups, geoTree, operatorUid, clientId, false)
        return operation.prepareAndApply()
    }

    fun updatePerformanceAdGroups(
        client: Client,
        operatorUid: Long,
        adGroupIds: List<Long>,
        complexGroupTemplate: ComplexPerformanceAdGroup,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)
        val complexGroups = adGroupIds
            .map { copyComplexPerformanceAdGroup(it, complexGroupTemplate) }
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        recreatePerformanceFilters(client, operatorUid, adGroupIds, brief)
        val operation = complexAdGroupUpdateOperationFactory
            .createComplexPerformanceAdGroupUpdateOperation(
                Applicability.FULL, complexGroups, geoTree, true, operatorUid, clientId, ModerationMode.FORCE_MODERATE
            )
        return operation.prepareAndApply()
    }

    private fun copyComplexDynamicAdGroup(
        adGroupId: Long,
        complexAdGroupTemplate: ComplexDynamicAdGroup
    ): ComplexDynamicAdGroup {
        val adGroup = complexAdGroupTemplate.adGroup
        return ComplexDynamicAdGroup()
            .withComplexBidModifier(complexAdGroupTemplate.complexBidModifier)
            .withAdGroup(
                DynamicFeedAdGroup()
                    .withId(adGroupId)
                    .withType(adGroup.type)
                    .withName(adGroup.name).withGeo(adGroup.geo)
                    .withHyperGeoId(adGroup.hyperGeoId)
                    .withTrackingParams(adGroup.trackingParams)
            )
    }

    private fun copyComplexPerformanceAdGroup(
        adGroupId: Long,
        complexAdGroupTemplate: ComplexPerformanceAdGroup
    ): ComplexPerformanceAdGroup {
        val adGroup = complexAdGroupTemplate.adGroup
        return ComplexPerformanceAdGroup()
            .withComplexBidModifier(complexAdGroupTemplate.complexBidModifier)
            .withAdGroup(
                PerformanceAdGroup()
                    .withId(adGroupId)
                    .withType(adGroup.type)
                    .withName(adGroup.name).withGeo(adGroup.geo)
                    .withHyperGeoId(adGroup.hyperGeoId)
                    .withTrackingParams(adGroup.trackingParams)
            )
    }

    private fun copyComplexTextAdGroup(
        adGroupId: Long,
        complexAdGroupTemplateProvider: () -> ComplexTextAdGroup,
        existingKeywordsByAdGroupId: Map<Long, List<Keyword>>,
        existingRelMatchesByAdGroupId: Map<Long, RelevanceMatch>,
        existingOfferRetargetingsByAdGroupId: Map<Long, OfferRetargeting>,
        relevanceMatchCategories: Set<RelevanceMatchCategory>,
        enableRelevanceMatchCategories: Boolean,
        retargetingCondition: RetargetingCondition?,
    ): ComplexTextAdGroup {
        val complexAdGroupTemplate = complexAdGroupTemplateProvider.invoke()
        val existingKeywords = existingKeywordsByAdGroupId.getOrDefault(adGroupId, listOf())
        val adGroup = complexAdGroupTemplate.adGroup as TextAdGroup
        val relevanceMatchesForUpdate = existingRelMatchesByAdGroupId[adGroupId]?.let { listOf(it) }
            ?: complexAdGroupTemplate.relevanceMatches
        val offerRetargetingsForUpdate = existingOfferRetargetingsByAdGroupId[adGroupId]?.let { listOf(it) }
            ?: complexAdGroupTemplate.offerRetargetings
        if (enableRelevanceMatchCategories) {
            relevanceMatchesForUpdate.forEach { it.withRelevanceMatchCategories(relevanceMatchCategories) }
        }

        return ComplexTextAdGroup()
            .withKeywords(complexAdGroupTemplate.keywords?.let {
                KeywordUtils.mergeNewKeywordsWithExisting(it, existingKeywords)
            })
            .withComplexBidModifier(complexAdGroupTemplate.complexBidModifier)
            .withTargetInterests(retargetingCondition?.let { listOf(it.targetInterest) })
            .withRetargetingCondition(retargetingCondition)
            .withRelevanceMatches(relevanceMatchesForUpdate)
            .withOfferRetargetings(offerRetargetingsForUpdate)
            .withAdGroup(
                TextAdGroup()
                    .withId(adGroupId)
                    .withType(adGroup.type)
                    .withName(adGroup.name)
                    .withGeo(adGroup.geo)
                    .withHyperGeoId(adGroup.hyperGeoId)
                    .withFeedId(adGroup.feedId)
                    .withFeedFilter(adGroup.feedFilter)
            )
    }

    fun createMissingOrNonexistentGroups(
        operatorUid: Long,
        client: Client,
        brief: UacAdGroupBrief,
        campaign: CommonCampaign,
        subCampaigns: Map<CampaignType, CommonCampaign>,
        cpmDirectContentType: DirectContentType?,
        appInfo: AppInfo?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        banners: List<BannerWithSystemFields>,
        keywords: List<Keyword>?,
        adGroupType: AdGroupType,
    ): MassResult<Long> {
        return createMissingOrNonexistentGroupsInternal(
            operatorUid,
            client,
            brief,
            campaign,
            subCampaigns,
            cpmDirectContentType,
            appInfo,
            complexBidModifier,
            retargetingCondition,
            banners,
            keywords,
            adGroupType,
        )
    }

    private fun createMissingOrNonexistentGroupsInternal(
        operatorUid: Long,
        client: Client,
        brief: UacAdGroupBrief,
        campaign: CommonCampaign,
        subCampaigns: Map<CampaignType, CommonCampaign>,
        cpmDirectContentType: DirectContentType?,
        appInfo: AppInfo?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        banners: List<BannerWithSystemFields>,
        keywords: List<Keyword>?,
        adGroupType: AdGroupType,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)
        when (adGroupType) {
            AdGroupType.MOBILE_CONTENT -> {
                val targetInterests: MutableList<TargetInterest> = mutableListOf()
                if (retargetingCondition != null) {
                    targetInterests.add(
                        TargetInterest()
                            .withRetargetingConditionId(retargetingCondition.id)
                            .withCampaignId(campaign.id)
                    )
                }
                if (targetInterestsEnabled.getOrDefault(false) && isLalOrNullRetargeting(retargetingCondition)) {
                    targetInterests.addAll(
                        appInfo?.interests?.map {
                            TargetInterest()
                                .withInterestId(targetingCategoriesCache.getCategoryByImportId(it.toBigInteger()))
                                .withCampaignId(campaign.id)
                        } ?: listOf()
                    )
                }
                return createMobileContentAdGroupWithBanner(
                    operatorUid,
                    client,
                    campaign as MobileContentCampaign,
                    keywords,
                    brief.minusKeywords,
                    banners,
                    brief.regions,
                    brief.minusRegions,
                    targetInterests,
                    retargetingCondition,
                )
            }
            AdGroupType.BASE -> {
                val textCampaign = campaign as TextCampaign
                return createTextAdGroupWithBanner(
                    operatorUid, client, textCampaign, keywords, complexBidModifier, retargetingCondition,
                    banners, brief, brief.feedId
                )
            }
            AdGroupType.CPM_BANNER -> {
                val cpmBannerCampaign = campaign as CpmBannerCampaign

                when (cpmDirectContentType){
                    DirectContentType.VIDEO, DirectContentType.NON_SKIPPABLE_VIDEO -> {
                        return createCpmVideoGroupWithBanner(
                            operatorUid, client, cpmBannerCampaign, complexBidModifier, retargetingCondition,
                            banners, brief, cpmDirectContentType
                        )
                    }
                    DirectContentType.HTML5 -> {
                        return createCpmBannerGroupWithBanner(
                            operatorUid, client, cpmBannerCampaign, complexBidModifier, retargetingCondition,
                            banners, brief
                        )
                    }
                    else -> {
                        throw java.lang.IllegalStateException("No suitable creatives")
                    }
                }
            }
            AdGroupType.DYNAMIC -> {
                return if (featureService.isEnabledForClientId(clientId, FeatureName.ECOM_UC_NEW_BACKEND_ENABLED)) {
                    createTextGroupWithBannerAndDynamicFilters(
                        operatorUid, client, campaign as TextCampaign, complexBidModifier, banners, brief
                    )
                } else {
                    val dynamicCampaign = subCampaigns[CampaignType.DYNAMIC]!! as DynamicCampaign
                    createDynamicGroupWithBanner(
                        operatorUid, client, dynamicCampaign, complexBidModifier, banners, brief
                    )
                }
            }
            else -> {
                throw IllegalStateException("Unsupported ad type: $adGroupType")
            }
        }
    }

    private fun createTextAdGroupWithBanner(
        operatorUid: Long,
        client: Client,
        campaign: TextCampaign,
        keywords: List<Keyword>?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        banners: List<BannerWithSystemFields>,
        brief: UacAdGroupBrief,
        feedId: Long? = null
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)
        val ecomUcNewBackend = brief.isEcom == true &&
            featureService.isEnabledForClientId(clientId, FeatureName.ECOM_UC_NEW_BACKEND_ENABLED)
        val enableRelevanceMatchCategories =
            featureService.isEnabledForClientId(clientId, FeatureName.RELEVANCE_MATCH_CATEGORIES_ALLOWED_IN_UC)

        val complexBanners = banners.map { banner -> ComplexBanner().withBanner(banner) }
        val filter = if (ecomUcNewBackend) {
            createFeedFilter(client, brief)
        } else {
            null
        }
        val group = TextAdGroup()
            .withCampaignId(campaign.id)
            .withName(campaign.name)
            .withType(AdGroupType.BASE)
            .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
            .withFeedId(feedId.takeIf { ecomUcNewBackend })
            .withFeedFilter(filter)
            .withHyperGeoId(brief.hyperGeoId)
        val relevanceMatch = RelevanceMatch()
            .withAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY)
        if (enableRelevanceMatchCategories) {
            relevanceMatch.withRelevanceMatchCategories(getRelevanceMatchCategory(brief))
        }
        val complexGroup = ComplexTextAdGroup()
            .withKeywords(keywords)
            .withAdGroup(group)
            .withComplexBidModifier(complexBidModifier)
            .withRetargetingCondition(retargetingCondition)
            .withTargetInterests(retargetingCondition?.let { listOf(it.targetInterest) })
            .withRelevanceMatches(listOf(relevanceMatch))
            .withOfferRetargetings(listOf(OfferRetargeting()).takeIf { ecomUcNewBackend })
            .withComplexBanners(complexBanners)
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        val operation: ComplexTextAdGroupAddOperation = complexAdGroupAddOperationFactory
            .createTextAdGroupAddOperation(
                false,
                listOf(complexGroup),
                geoTree,
                false,
                null,
                operatorUid,
                clientId,
                client.chiefUid,
                featureService.getDatabaseMode(clientId)
            )
        return operation.prepareAndApply()
    }

    private fun createMobileContentAdGroupWithBanner(
        operatorUid: Long,
        client: Client,
        campaign: MobileContentCampaign,
        keywords: List<Keyword>?,
        minusKeywords: List<String>?,
        banners: List<BannerWithSystemFields>,
        regions: List<Long>?,
        minusRegions: List<Long>?,
        targetInterests: List<TargetInterest>?,
        retargetingCondition: RetargetingCondition?,
    ): MassResult<Long> {
        val complexBanners = banners.map { banner -> ComplexBanner().withBanner(banner) }
        val mobileApp = mobileAppService.getMobileApp(ClientId.fromLong(client.clientId), campaign.mobileAppId).get()
        val group = MobileContentAdGroup()
            .withCampaignId(campaign.id)
            .withName(campaign.name)
            .withType(AdGroupType.MOBILE_CONTENT)
            .withMobileContentId(mobileApp.mobileContentId)
            .withStoreUrl(mobileApp.storeHref)
            .withDeviceTypeTargeting(
                setOf(
                    MobileContentAdGroupDeviceTypeTargeting.PHONE,
                    MobileContentAdGroupDeviceTypeTargeting.TABLET
                )
            )
            .withNetworkTargeting(
                setOf(
                    MobileContentAdGroupNetworkTargeting.CELLULAR,
                    MobileContentAdGroupNetworkTargeting.WI_FI
                )
            )
            .withMinimalOperatingSystemVersion(CommonUtils.nvl(mobileApp.minimalOperatingSystemVersion, "1.0"))
            .withGeo(getGeoForUacGroups(regions, minusRegions))
            .withMinusKeywords(minusKeywords)
        val relevanceMatch = if (isLalOrNullRetargeting(retargetingCondition)) listOf(
            RelevanceMatch()
                .withAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY)
        )
        else listOf()
        val complexGroup = ComplexMobileContentAdGroup()
            .withKeywords(keywords)
            .withAdGroup(group)
            .withRelevanceMatches(relevanceMatch)
            .withComplexBanners(complexBanners)
            .withTargetInterests(targetInterests)
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(ClientId.fromLong(client.clientId))
        val operation: ComplexMobileContentAdGroupAddOperation = complexAdGroupAddOperationFactory
            .createMobileContentAdGroupAddOperation(
                false,
                listOf(complexGroup),
                geoTree,
                false,
                null,
                operatorUid,
                ClientId.fromLong(client.clientId),
                client.chiefUid
            )
        return operation.prepareAndApply()
    }

    //используется только в интапи, фразы в ydb не синкаем
    //TODO: удалить метод
    fun createMobileContentAdGroupWithBanner(
        userId: Long,
        client: Client,
        campaign: MobileContentCampaign,
        banner: BannerWithSystemFields,
        regions: List<Long>?,
        minusRegions: List<Long>?,
        keywords: List<Keyword>?,
        minusKeywords: List<String>?,
        targetInterests: List<TargetInterest>?,
    ): MassResult<Long> {
        return createMobileContentAdGroupWithBanner(
            userId,
            client,
            campaign,
            keywords,
            minusKeywords,
            listOf(banner),
            regions,
            minusRegions,
            targetInterests,
            null
        )
    }

    // производит синк фраз из mysql в ydb/grut,
    // поскольку в mysql фразы могли записаться в изменённом по сравнению с пользовательским виде
    // обновляем только если фразы на кампании в хранилище не поменялись
    fun syncMysqlPhrasesToUacStorage(
        directAdGroupId: Long,
        adGroupType: AdGroupType,
        oldKeywordsInStorage: List<String>?,
        oldMinusKeywordsInStorage: List<String>?,
    ) {
        if (!ClientKeywordCommonValidator.adGroupCanHaveKeywords(adGroupType)) {
            return
        }
        val adGroup = adGroupService.getAdGroup(directAdGroupId)
        if (adGroup == null) {
            logger.error("can not find adgroup " + directAdGroupId)
            return
        }
        val directCampaignId = adGroup.campaignId!!
        val mysqlKeywords: List<String>? = keywordRepository
            .getKeywordsByAdGroupIds(
                shardHelper.getShardByCampaignId(directCampaignId),
                listOf(directAdGroupId)
            )
            .get(directAdGroupId)
            ?.map { it.phrase }
        val mysqlMinusKeywords = adGroup.minusKeywords?.ifEmpty { null }
        uacCampaignServiceHolder.performUpdateActionOnCampaignUsingDirectId(
            directCampaignId
        ) { baseUacCampaignUpdateService, campaign ->
            val modelChanges = KtModelChanges<String, UacYdbCampaign>(campaign.id)
            if (Objects.equals(oldKeywordsInStorage, campaign.keywords)) {
                logger.info("updating keywords in storage")
                modelChanges.process(UacYdbCampaign::keywords, mysqlKeywords)
            }
            if (Objects.equals(oldMinusKeywordsInStorage, campaign.minusKeywords)) {
                logger.info("updating minus keywords in storage")
                modelChanges.process(UacYdbCampaign::minusKeywords, mysqlMinusKeywords)
            }
            if (modelChanges.isPresent(UacYdbCampaign::minusKeywords) ||
                modelChanges.isPresent(UacYdbCampaign::keywords)
            ) {
                baseUacCampaignUpdateService.updateCampaignFromModelChanges(campaign, modelChanges)
            } else {
                logger.info("distinct objects in storages, skip updating")
            }
        }
    }

    private fun createCpmBannerGroupWithBanner(
        userId: Long,
        client: Client,
        campaign: CpmBannerCampaign,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        banners: List<BannerWithSystemFields>,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val complexGroup: ComplexCpmAdGroup = cpmBannerCampaignService.complexGroupFromUac(
            null, campaign, complexBidModifier, retargetingCondition, banners, brief, AdGroupType.CPM_BANNER,
            getGeoForUacGroups(brief.regions, brief.minusRegions)
        )
        return createCpmGroup(complexGroup, userId, client)
    }

    private fun createCpmGroup(complexGroup: ComplexCpmAdGroup, userId: Long, client: Client): MassResult<Long> {
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(ClientId.fromLong(client.clientId))

        val operation: ComplexCpmAdGroupAddOperation = complexAdGroupAddOperationFactory
            .createCpmAdGroupAddOperation(
                false,
                listOf(complexGroup),
                geoTree,
                false,
                null,
                userId,
                ClientId.fromLong(client.clientId),
                client.chiefUid,
                true
            )

        return operation.prepareAndApply()
    }

    private fun createCpmVideoGroupWithBanner(
        userId: Long,
        client: Client,
        campaign: CpmBannerCampaign,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        banners: List<BannerWithSystemFields>,
        brief: UacAdGroupBrief,
        cpmDirectContentType: DirectContentType?,
    ): MassResult<Long> {
        val complexGroup = cpmBannerCampaignService.complexGroupFromUac(
            null, campaign, complexBidModifier, retargetingCondition, banners, brief, AdGroupType.CPM_VIDEO,
            getGeoForUacGroups(brief.regions, brief.minusRegions), cpmDirectContentType
        )
        return createCpmGroup(complexGroup, userId, client)
    }

    private fun createDynamicGroupWithBanner(
        operatorUid: Long,
        client: Client,
        campaign: DynamicCampaign,
        complexBidModifier: ComplexBidModifier?,
        banners: List<BannerWithSystemFields>,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.id)
        val group = DynamicFeedAdGroup()
            .withCampaignId(campaign.id)
            .withName(campaign.name)
            .withType(AdGroupType.DYNAMIC)
            .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
            .withHyperGeoId(brief.hyperGeoId)
            .withTrackingParams(brief.trackingParams)
            .withFeedId(brief.feedId)
            .withBanners(banners)
        val complexGroup = ComplexDynamicAdGroup()
            .withAdGroup(group)
            .withBanners(banners)
            .withComplexBidModifier(complexBidModifier)
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        val operation = complexAdGroupAddOperationFactory.createDynamicAdGroupAddOperation(
            false,
            listOf(complexGroup),
            geoTree,
            operatorUid,
            clientId
        )
        val result = operation.prepareAndApply()
        if (result.validationResult.hasAnyErrors()) {
            return result
        }
        val groupId = result[0].result
        createDynamicFilter(client, operatorUid, groupId, brief)
        return result
    }

    fun createTextGroupWithBannerAndDynamicFilters(
        operatorUid: Long,
        client: Client,
        campaign: TextCampaign,
        complexBidModifier: ComplexBidModifier?,
        banners: List<BannerWithSystemFields>,
        brief: UacAdGroupBrief
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.clientId)
        val result = createTextAdGroupWithBanner(
            operatorUid, client,
            campaign, null, complexBidModifier, null, banners, brief, brief.feedId)
        if (result.isSuccessful && featureService.isEnabledForClientId(clientId,
                FeatureName.ENABLED_DYNAMIC_FEED_AD_TARGET_IN_TEXT_AD_GROUP)) {
            result.result.forEach { createDynamicFilter(client, operatorUid, it.result, brief) }
        }
        return result
    }

    /**
     * Создаёт смарт-группу с новым родительским баннером в ней
     */
    fun createSmartGroupWithMainBanner(
        operatorUid: Long,
        client: Client,
        campaign: CommonCampaign,
        complexBidModifier: ComplexBidModifier?,
        banner: PerformanceBannerMain,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.id)
        val group = PerformanceAdGroup()
            .withCampaignId(campaign.id)
            .withName(campaign.name)
            .withType(AdGroupType.PERFORMANCE)
            .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
            .withHyperGeoId(brief.hyperGeoId)
            .withTrackingParams(brief.trackingParams)
            .withFeedId(brief.feedId)
            .withBanners(listOf(banner))
        val complexGroup = ComplexPerformanceAdGroup()
            .withAdGroup(group)
            .withBanners(listOf(banner))
            .withComplexBidModifier(complexBidModifier)
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        val operation = complexAdGroupAddOperationFactory.createPerformanceAdGroupAddOperation(
            Applicability.FULL,
            false,
            listOf(complexGroup),
            geoTree,
            operatorUid,
            clientId
        )
        val groupResult = operation.prepareAndApply()
        if (groupResult.validationResult.hasAnyErrors()) {
            return groupResult
        }
        val groupId = groupResult[0].result
        createSmartFilter(client, operatorUid, groupId, brief)
        return groupResult
    }

    /**
     * Создаёт смарт группу и баннер в ней
     *
     * В отличие от других методов здесь, этот создаёт сущности последовательно — сначала группу,
     * потом баннер в ней. Комплексно делать это мы пока не умеем. Причина этого в том, что для
     * валидации создания баннера надо уже иметь готовую группу.
     *
     * @see BannerWithCreativeValidationContainerFactory#disallowPerformanceBannerInComplexOperation
     */
    fun createSmartGroupWithBanner(
        operatorUid: Long,
        client: Client,
        campaign: CommonCampaign,
        complexBidModifier: ComplexBidModifier?,
        banners: List<BannerWithSystemFields>,
        brief: UacAdGroupBrief,
    ): MassResult<Long> {
        val clientId = ClientId.fromLong(client.id)
        val group = PerformanceAdGroup()
            .withCampaignId(campaign.id)
            .withName(campaign.name)
            .withType(AdGroupType.PERFORMANCE)
            .withGeo(getGeoForUacGroups(brief.regions, brief.minusRegions))
            .withHyperGeoId(brief.hyperGeoId)
            .withTrackingParams(brief.trackingParams)
            .withFeedId(brief.feedId)
            .withBanners(banners)
        val complexGroup = ComplexPerformanceAdGroup()
            .withAdGroup(group)
            .withComplexBidModifier(complexBidModifier)
        val geoTree: GeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        val operation = complexAdGroupAddOperationFactory.createPerformanceAdGroupAddOperation(
            Applicability.FULL,
            false,
            listOf(complexGroup),
            geoTree,
            operatorUid,
            clientId
        )
        val groupResult = operation.prepareAndApply()

        if (groupResult.validationResult.hasAnyErrors()) {
            return groupResult
        }

        val groupId = groupResult[0].result

        createSmartBanners(groupId, banners, clientId, operatorUid)

        createSmartFilter(client, operatorUid, groupId, brief)

        return groupResult
    }

    private fun createSmartBanners(
        adGroupId: Long,
        banners: List<BannerWithSystemFields>,
        clientId: ClientId,
        operatorUid: Long
    ) {
        banners.forEach { it.adGroupId = adGroupId }
        val bannerOperation = bannersAddOperationFactory
            .createFullAddOperation(banners, clientId, operatorUid, false)
        val bannerResult = bannerOperation.prepareAndApply()
        logResultErrors(bannerResult, adGroupId, "Can't create smart banner for ad group: {}, first error is: {}")
    }

    private fun createSmartFilter(
        client: Client,
        operatorUid: Long,
        groupId: Long,
        brief: UacAdGroupBrief,
    ) {
        val clientId = ClientId.fromLong(client.id)
        val feed = feedService.getFeeds(clientId, setOf(brief.feedId))[0]
        val schema = performanceFilterStorage.getFilterSchema(feed.businessType, feed.feedType, feed.source)
        val filters = brief.feedFilters?.toSmartFilters(groupId, DEFAULT_FILTER_NAME, feed, schema)
            ?: listOf(defaultSmartFilter(groupId, DEFAULT_FILTER_NAME, feed))

        if (featureService.isEnabledForClientId(clientId, FeatureName.ECOM_UC_ADD_DEFAULT_FILTER_CONDITION)) {
            filters.forEach {
                val newConditions = ImmutableList.Builder<PerformanceFilterCondition<Any>>()
                    .addAll(it.conditions)
                    .add(defaultAvailablePerformanceCondition())
                    .build()
                it.conditions = newConditions
            }
        }

        val result = performanceFilterService.add(clientId, operatorUid, filters, Applicability.FULL)
        logResultErrors(result, groupId, "Can't create smart filter for ad group: {}, first error is: {}")
    }

    private fun createDynamicFilter(
        client: Client,
        operatorUid: Long,
        groupId: Long,
        brief: UacAdGroupBrief
    ) {
        val clientId = ClientId.fromLong(client.id)
        val feed = feedService.getFeeds(clientId, setOf(brief.feedId))[0]
        val schema = performanceFilterStorage.getFilterSchema(feed.businessType, feed.feedType, feed.source)
        val filters = brief.feedFilters?.toDynamicFilters(groupId, DEFAULT_FILTER_NAME, feed, schema)
            ?: listOf(defaultDynamicFilter(groupId, DEFAULT_FILTER_NAME, feed))

        if (featureService.isEnabledForClientId(clientId, FeatureName.ECOM_UC_ADD_DEFAULT_FILTER_CONDITION)) {
            filters.forEach {
                val newConditions = ImmutableList.Builder<DynamicFeedRule<Any>>()
                    .addAll(it.condition)
                    .add(defaultAvailableDynamicFeedRule())
                    .build()
                it.condition = newConditions
            }
        }

        val result = dynamicTextAdTargetService.addDynamicFeedAdTargets(clientId, operatorUid, filters)
        logResultErrors(result, groupId, "Can't create dynamic filter for ad group: {}, first error is: {}")
    }

    private fun createFeedFilter(
        client: Client,
        brief: UacAdGroupBrief
    ): FeedFilter {
        val clientId = ClientId.fromLong(client.id)
        val feed = feedService.getFeeds(clientId, setOf(brief.feedId))!![0]
        val schema = performanceFilterStorage.getFilterSchema(feed.businessType, feed.feedType, feed.source)
        val filter = brief.feedFilters?.toFeedFilter(schema) ?: defaultFeedFilter()

        if (featureService.isEnabledForClientId(clientId, FeatureName.ECOM_UC_ADD_DEFAULT_FILTER_CONDITION)) {
            val newConditions = ImmutableList.Builder<FeedFilterCondition<Any?>>()
                .addAll(filter.conditions)
                .add(defaultAvailableFeedFilterCondition())
                .build()
            filter.conditions = newConditions
        }

        return filter
    }

    /**
     * Если есть изменения в фильтрах, полностью пересоздаст их.
     */
    private fun recreateDynamicFilters(
        client: Client, operatorUid: Long,
        groupIds: List<Long>, brief: UacAdGroupBrief
    ) {
        val clientId = ClientId.fromLong(client.clientId)
        val dynamicFeedAdTargets = dynamicTextAdTargetService.getDynamicFeedAdTargets(
            clientId, operatorUid,
            DynamicTextAdTargetSelectionCriteria().withAdGroupIds(groupIds.toSet()), LimitOffset.maxLimited()
        )
        val uacConditionsFromDb = dynamicFeedAdTargets
            .filter { it.condition.isNotEmpty() }
            .map { it.toUacFeedFiltersConditions() }
            .toSet()
        val uacConditionsFromYdb = brief.feedFilters
            ?.map { it.conditions.toSet() }
            ?.toSet() ?: setOf()
        if (uacConditionsFromDb != uacConditionsFromYdb) {
            val dynamicFeedAdTargetIds = dynamicFeedAdTargets.map { it.dynamicConditionId }
            val deletionResult =
                dynamicTextAdTargetService.deleteDynamicAdTargets(operatorUid, clientId, dynamicFeedAdTargetIds)
            logResultErrors(deletionResult, "Can't delete dynamic targets, first error is {}")
            groupIds.forEach { createDynamicFilter(client, operatorUid, it, brief) }
        }
    }

    private fun recreatePerformanceFilters(
        client: Client, operatorUid: Long,
        groupIds: List<Long>, brief: UacAdGroupBrief
    ) {
        val clientId = ClientId.fromLong(client.clientId)
        val smartFilters = performanceFilterService.getPerformanceFilters(clientId, groupIds).values.flatten()
        val uacConditionsFromDb = smartFilters
            .filter { it.conditions.isNotEmpty() }
            .map { it.toUacFeedFiltersConditions() }
            .toSet()
        val uacConditionsFromYdb = brief.feedFilters
            ?.map { it.conditions.toSet() }
            ?.toSet() ?: setOf()
        if (uacConditionsFromDb != uacConditionsFromYdb) {
            val smartFilterIds = smartFilters.map { it.id }
            val deletionResult =
                performanceFilterService.deletePerformanceFilters(clientId, operatorUid, smartFilterIds)
            logResultErrors(deletionResult, "Can't delete smart filter, first error is {}")
            groupIds.forEach { createSmartFilter(client, operatorUid, it, brief) }
        }
    }

    private fun logResultErrors(result: MassResult<Long>, parentId: Long, message: String) {
        if (result.validationResult.hasAnyErrors()) {
            logger.error(message, parentId, result.validationResult.flattenErrors()[0].toString())
        }
    }

    private fun logResultErrors(result: MassResult<Long>, message: String) {
        if (result.validationResult.hasAnyErrors()) {
            logger.error(message, result.validationResult.flattenErrors()[0].toString())
        }
    }

    /**
     * Получаем категории автотаргетинга из заявки
     * Пустой список или !active означает что выбраны все категории
     */
    private fun getRelevanceMatchCategory(
        brief: UacAdGroupBrief
    ): Set<RelevanceMatchCategory> = if (
        brief.relevanceMatch != null
        // Для UC автотаргетинг всегда включен, поэтому для !active считаем что выбраны все
        && brief.relevanceMatch.active
    ) {
        brief.relevanceMatch.categories
            .map { toRelevanceMatchCategory(it) }
            .toSet()
    } else {
        emptySet()
    }

    private fun isLalOrNullRetargeting(retargetingCondition: RetargetingCondition?): Boolean {
        return retargetingCondition
            ?.collectGoalsSafe()
            ?.all { Goal.computeType(it.id) == GoalType.LAL_SEGMENT }
            ?: true
    }
}
