package ru.yandex.direct.core.entity.uac.service

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.AppInfoContent
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.core.entity.uac.service.appinfo.AppInfoGetter
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Lazy
@Component
class UacAppInfoService(
        private val parsers: List<AppInfoGetter>
) {
    private val parserBySource: Map<Store, AppInfoGetter> = parsers.associateBy { it.source }

    private fun getParser(source: Store): AppInfoGetter {
        return parserBySource[source]
            ?: throw IllegalStateException("No parser for source $source")
    }

    fun getAppInfo(ydbAppInfo: UacYdbAppInfo, addRecommendations: Boolean = true): AppInfo {
        val parser = getParser(ydbAppInfo.source)
        return parser.getAppInfo(ydbAppInfo, addRecommendations)
    }

    fun getAppInfoContents(ydbAppInfo: UacYdbAppInfo): List<AppInfoContent> {
        val parser = getParser(ydbAppInfo.source)
        return parser.getContent(ydbAppInfo)
    }

    /**
     * Возвращает минимальную информацию о приложении, которую можно получить из ссылки
     * на стор (без парсинга страницы в сторе) — id приложения, язык, регион и платформа
     * В случае, если для ссылки нашелся парсер, но он не смог ее распарсить, возвращается дефект
     */
    fun getTruncatedAppInfoByUrl(url: String, defect: Defect<Void>): ValidationResult<AppInfo, Defect<*>>? {
        val urlParts = UrlParts.fromUrl(url)
        parsers.forEach {
            if (it.isSuitable(urlParts)) {
                return it.getAppInfoByUrl(url, defect)
            }
        }
        return null
    }
}

