package ru.yandex.direct.core.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.avatars.client.AvatarsClient
import ru.yandex.direct.avatars.client.exception.AvatarsClientCommonException
import ru.yandex.direct.avatars.client.model.AvatarId
import ru.yandex.direct.core.entity.freelancer.service.AvatarsClientPool
import ru.yandex.direct.core.entity.uac.model.UacAvatarsResponseInfo
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import java.util.UUID

@Service
class UacAvatarsService(
    private val uacAvatarsClientPool: AvatarsClientPool
) {
    companion object {
        private const val URI_PATH_SEPARATOR = "/"
        private val logger = LoggerFactory.getLogger(UacAvatarsService::class.java)
    }

    fun uploadImage(image: ByteArray, name: String?): Result<UacAvatarsResponseInfo> {
        return try {
            val avatarInfo = avatarsClient.upload(name ?: randomName(), image)
            Result.successful(UacAvatarsResponseInfo(avatarInfo))
        } catch (e: AvatarsClientCommonException) {
            logger.error("Failure to upload image from data", e)
            Result.broken(ValidationResult.failed(image, CommonDefects.invalidValue()))
        }
    }

    fun uploadImageByUrl(imageUrl: String, name: String?): Result<UacAvatarsResponseInfo> {
        return try {
            val avatarInfo = avatarsClient.uploadByUrl(name ?: randomName(), imageUrl)
            Result.successful(UacAvatarsResponseInfo(avatarInfo))
        } catch (e: AvatarsClientCommonException) {
            logger.error("Failure to upload image from url", e)
            Result.broken(ValidationResult.failed(imageUrl, CommonDefects.invalidValue()))
        }
    }

    fun deleteImage(id: String): Boolean {
        val avatarId = AvatarId.fromIdString(id)
        val result = avatarsClient.delete(avatarId)
        if (result) logger.info("Image {} was successfully deleted", id)
        return result
    }

    private fun randomName(): String {
        return UUID.randomUUID().toString()
    }

    fun getReadUrl(avatarInfo: UacAvatarsResponseInfo, size: String): String {
        return getReadUrl(avatarInfo.groupId, avatarInfo.name, size)
    }

    private fun getReadUrl(groupId: Int, name: String, size: String): String {
        return avatarsClient.getReadUrl(getAvatarsImageExternalId(groupId, name), size)
    }

    private fun getAvatarsImageExternalId(groupId: Int, name: String): String {
        return avatarsClient.conf.uploadNamespace + URI_PATH_SEPARATOR + groupId.toString() + URI_PATH_SEPARATOR + name
    }

    private val avatarsClient: AvatarsClient
        get() = uacAvatarsClientPool.defaultClient
}
