package ru.yandex.direct.core.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.model.TextAdGroup
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService
import ru.yandex.direct.core.entity.banner.container.AdsSelectionCriteria
import ru.yandex.direct.core.entity.banner.model.Banner
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner
import ru.yandex.direct.core.entity.banner.service.BannerArchiveUnarchiveService
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.BannerSuspendResumeService
import ru.yandex.direct.core.entity.banner.service.BannersAddOperationFactory
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory
import ru.yandex.direct.core.entity.banner.service.DatabaseMode
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.model.UpdateAdsJobParams
import ru.yandex.direct.core.entity.uac.repository.mysql.BannerStatusesInfo
import ru.yandex.direct.core.entity.uac.repository.mysql.BannerStatusesRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.grut.GrutUtils.Companion.getDatabaseMode
import ru.yandex.direct.core.grut.api.BannerGrutApi
import ru.yandex.direct.core.grut.converter.calcBannerCreativeStatusModerate
import ru.yandex.direct.core.grut.converter.calcBannerFlags
import ru.yandex.direct.core.grut.converter.calcBannerImageStatusModerate
import ru.yandex.direct.core.grut.converter.calcSiteinksSetStatusModerate
import ru.yandex.direct.core.grut.converter.calcStatusModerate
import ru.yandex.direct.core.grut.converter.convertToAssetModerationReasons
import ru.yandex.direct.core.grut.converter.convertToModerationReasons
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.result.MassResult
import ru.yandex.grut.objects.proto.client.Schema.TBannerV2

data class UacBannerModerationInfo(
    val bannerStatuses: BannerStatusesInfo,
    val moderationReasons: Map<ModerationReasonObjectType, List<ModerationDiag>>,
    val assetModerationReasons: Map<BannerAssetType, List<ModerationDiag>>,
)

@Lazy
@Service
class UacBannerService(
    grutContext: GrutContext,
    private val bannersUpdateOperationFactory: BannersUpdateOperationFactory,
    private var bannersAddOperationFactory: BannersAddOperationFactory,
    private var adGroupService: AdGroupService,
    private val bannerService: BannerService,
    private val bannerArchiveUnarchiveService: BannerArchiveUnarchiveService,
    private val shardHelper: ShardHelper,
    private val bannerSuspendResumeService: BannerSuspendResumeService,
    private val moderationReasonService: ModerationReasonService,
    private val bannerStatusesRepository: BannerStatusesRepository,
    private val dbQueueRepository: DbQueueRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val featureService: FeatureService,
) {

    private val bannerGrutApi = BannerGrutApi(grutContext)

    companion object {
        private val logger = LoggerFactory.getLogger(UacBannerService::class.java)
        private const val CHUNK_ADS_SIZE = 990L
        private const val MAX_COUNT_OF_REQUESTS = 1_000
    }

    data class DeleteAndArchiveBannersResult(
        val deleteMassResult: MassResult<Long>,
        val archiveMassResult: MassResult<Long>,
    )

    fun getMobileAppBanners(
        operatorUid: Long,
        clientId: ClientId,
        bannerIds: Set<Long>,
    ): List<BannerWithSystemFields> {
        val criteria = AdsSelectionCriteria()
            .withAdIds(bannerIds)
            .withTypes(BannersBannerType.mobile_content)
        return getBanners(operatorUid, clientId, criteria)
    }

    fun getBanners(
        operatorUid: Long,
        clientId: ClientId,
        bannerIds: Set<Long>,
    ): List<BannerWithSystemFields> {
        val criteria = AdsSelectionCriteria()
            .withAdIds(bannerIds)
        return getBanners(operatorUid, clientId, criteria)
    }

    fun getMobileAppBannersByAdGroupIds(
        operatorUid: Long,
        clientId: ClientId,
        adGroupIds: Set<Long>,
    ): List<BannerWithSystemFields> {
        val criteria = AdsSelectionCriteria()
            .withAdGroupIds(adGroupIds)
            .withTypes(BannersBannerType.mobile_content)
        return getBanners(operatorUid, clientId, criteria)
    }

    fun getBannersByAdGroupIds(
        operatorUid: Long,
        clientId: ClientId,
        adGroupIds: Set<Long>,
    ): List<BannerWithSystemFields> {
        val criteria = AdsSelectionCriteria()
            .withAdGroupIds(adGroupIds)
        return getBanners(operatorUid, clientId, criteria)
    }

    fun getBannerStatusesInfoByAdGroupIds(
        clientId: ClientId,
        adGroupIds: Set<Long>,
    ): List<BannerStatusesInfo> {
        val shard = shardHelper.getShardByClientId(clientId)
        return bannerStatusesRepository.getBannerStatusesInfoByAdGroupIds(shard, adGroupIds)
    }

    private inline fun <reified T : Banner> getBanners(
        operatorUid: Long,
        clientId: ClientId,
        criteria: AdsSelectionCriteria,
    ): List<T> {
        return bannerService
            .getBannersBySelectionCriteria(operatorUid, clientId, criteria, LimitOffset.maxLimited())
            .filterIsInstance(T::class.java)
    }

    fun getDirectAdsByContentIds(
        uacAdGroupIds: Collection<String>,
    ): List<UacYdbDirectAd> {

        val allYdbDirectAds = mutableListOf<UacYdbDirectAd>()
        if (uacAdGroupIds.isEmpty()) {
            return allYdbDirectAds
        }

        var fromUacAdId = 0L
        var attemptsLimit = MAX_COUNT_OF_REQUESTS
        while (true) {
            val ydbDirectAdsFromYdb = uacYdbDirectAdRepository
                .getByDirectAdGroupId(uacAdGroupIds, fromUacAdId, CHUNK_ADS_SIZE)

            allYdbDirectAds.addAll(ydbDirectAdsFromYdb)
            if (ydbDirectAdsFromYdb.size < CHUNK_ADS_SIZE) {
                break
            }
            if (attemptsLimit-- == 0) {
                logger.error("The limit of $MAX_COUNT_OF_REQUESTS attempts has been reached")
                break
            }

            // Здесь нельзя сравнивать и брать наибольший, т.к. значения id выше long.max - отрицательные
            fromUacAdId = ydbDirectAdsFromYdb.last().id.toIdLong()
        }
        return allYdbDirectAds
    }

    fun createBanner(
        operatorUid: Long,
        clientId: ClientId,
        campaign: BaseCampaign,
        banners: List<BannerWithSystemFields>,
        adGroupId: Long,
    ): MassResult<Long>? {
        val adGroup = adGroupService.getAdGroup(adGroupId)

        if (adGroup == null || (
                adGroup !is MobileContentAdGroup
                    && adGroup !is TextAdGroup
                    && adGroup !is CpmBannerAdGroup
                    && adGroup !is CpmVideoAdGroup
                    && adGroup !is DynamicAdGroup
                )
        ) {
            logger.error(
                "Can't find mobile content or text group or cpm_banner group for add a new banner. " +
                    "ClientId: $clientId, adGroupId: $adGroupId"
            )
            return null
        }

        banners.forEach { banner ->
            banner
                .withAdGroupId(adGroup.id)
                .withCampaignId(campaign.id)
        }

        val databaseMode: DatabaseMode = featureService.getDatabaseMode(clientId)
        return bannersAddOperationFactory
            .createFullAddOperation(banners, clientId, operatorUid, false, databaseMode)
            .prepareAndApply()
    }

    @Deprecated("используется только в 1 контроллере, а контроллер уже не используется")
    fun updateBanners(
        clientId: ClientId,
        operatorUid: Long,
        bannerIds: List<Long>,
        trackingUrl: String?,
        impressionUrl: String?
    ): MassResult<Long> {

        val changes = bannerIds
            .map { id: Long ->
                ModelChanges(id, MobileAppBanner::class.java)
                    .processNotNull(trackingUrl, MobileAppBanner.HREF)
                    .processNotNull(impressionUrl, MobileAppBanner.IMPRESSION_URL)
                    .castModel(BannerWithSystemFields::class.java)
            }
            .toList()

        val operation = bannersUpdateOperationFactory.createFullUpdateOperation(
            changes, operatorUid, clientId, DatabaseMode.ONLY_MYSQL
        )

        return operation.prepareAndApply()
    }

    /**
     * Удаление и архивирование (если нельзя удалить) баннеров по переданному списку {@param bannerIds}
     *
     * @return MassResult удаления и архивирования
     */
    fun deleteAndArchiveBanners(
        operatorUid: Long,
        clientId: ClientId,
        bannerIds: Set<Long>,
    ): DeleteAndArchiveBannersResult {
        if (bannerIds.isEmpty()) {
            return DeleteAndArchiveBannersResult(MassResult.emptyMassAction(), MassResult.emptyMassAction())
        }

        val banners = getBanners(operatorUid, clientId, bannerIds)
        if (banners.isEmpty()) {
            logger.error("No banner found in mysql: $bannerIds")
            return DeleteAndArchiveBannersResult(MassResult.emptyMassAction(), MassResult.emptyMassAction())
        }

        val shard = shardHelper.getShardByClientId(clientId)

        val canDelete = bannerService.getCanBeDeletedBanners(shard, clientId, banners)

        var deleteMassResult: MassResult<Long> = MassResult.emptyMassAction()
        val bannerIdsToDelete = banners
            .filter { canDelete[it.id]!! }
            .map { it.id }
        if (bannerIdsToDelete.isNotEmpty()) {
            deleteMassResult = deleteBanners(operatorUid, clientId, bannerIdsToDelete)
        }

        var archiveMassResult: MassResult<Long> = MassResult.emptyMassAction()
        val bannersToArchive = banners
            .filter { !canDelete[it.id]!! }
        if (bannersToArchive.isNotEmpty()) {
            archiveMassResult = archiveBanners(operatorUid, clientId, bannersToArchive)
        }
        return DeleteAndArchiveBannersResult(deleteMassResult, archiveMassResult)
    }

    private fun deleteBanners(
        operatorUid: Long,
        clientId: ClientId,
        bannerIds: List<Long>,
    ): MassResult<Long> {
        val massDeleteResult = bannerService.deleteBannersPartial(operatorUid, clientId, bannerIds)
        massDeleteResult?.errors?.forEach {
            logger.error("Error when try to delete banner, clientId: $clientId, defect: $it")
        }
        return massDeleteResult
    }

    private fun archiveBanners(
        operatorUid: Long,
        clientId: ClientId,
        banners: List<BannerWithSystemFields>,
    ): MassResult<Long> {
        val bannersToArchive = banners
            .map {
                ModelChanges(it.id, BannerWithSystemFields::class.java)
                    .process(true, BannerWithSystemFields.STATUS_ARCHIVED)
            }

        val massArchiveResult: MassResult<Long> = bannerArchiveUnarchiveService.archiveUnarchiveBanners(
            clientId,
            operatorUid,
            bannersToArchive,
            true
        )
        massArchiveResult.errors?.forEach {
            logger.error("Error when try to archive banner, clientId: $clientId, defect: $it")
        }
        return massArchiveResult
    }

    /**
     * Остановить/запустить баннеры. Обновляет у баннеров {@param statusShow}
     */
    fun suspendResumeBanners(
        clientId: ClientId,
        operatorId: Long,
        statusShow: Boolean,
        bannerIds: Collection<Long>,
    ): MassResult<Long> {
        val changes = bannerIds
            .map {
                ModelChanges(it, BannerWithSystemFields::class.java)
                    .process(statusShow, BannerWithSystemFields.STATUS_SHOW)
            }

        return bannerSuspendResumeService.suspendResumeBanners(
            clientId, operatorId, changes, statusShow
        )
    }

    fun getModerationDiagsByBannerIds(
        clientId: ClientId,
        bannerIds: List<Long>
    ): Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> {
        return moderationReasonService
            .getRejectReasonDiags(clientId, getReasonsTypeToBannerIds(bannerIds))
    }

    fun getBannerModerationData(
        shard: Int,
        bannerIds: List<Long>,
        fetchDataFromGrut: Boolean
    ): Map<Long, UacBannerModerationInfo> {
        val bannerStatuses = bannerStatusesRepository.getBannerStatusesInfo(shard, bannerIds)
        val moderationReasons = getModerationDiagsByBannerIds(shard, bannerIds)
        val assetModerationReasons = getAssetsModerationDiagsByBannerIds(shard, bannerIds)

        val bannerModerationDataFromMysql = bannerStatuses.associate {
            val bid = it.bannerId
            bid to UacBannerModerationInfo(
                bannerStatuses = it,
                moderationReasons = moderationReasons[bid] ?: emptyMap(),
                assetModerationReasons = assetModerationReasons[bid] ?: emptyMap()
            )
        }

        if (!fetchDataFromGrut) {
            return bannerModerationDataFromMysql
        }

        val bannerModerationDataFromGrut = getBannerModerationDataFromGrut(bannerIds)
        return bannerIds.mapNotNull { bannerId ->
            val dataFromMysql = bannerModerationDataFromMysql[bannerId]
            if (dataFromMysql != null) {
                return@mapNotNull bannerId to dataFromMysql
            }
            val dataFromGrut = bannerModerationDataFromGrut[bannerId]
            if (dataFromGrut != null) {
                return@mapNotNull bannerId to dataFromGrut
            }
            return@mapNotNull null
        }.toMap()
    }

    private fun getBannerModerationDataFromGrut(bannerIds: List<Long>): Map<Long, UacBannerModerationInfo> {
        return bannerGrutApi.getBannersByDirectId(bannerIds).associate {
            it.meta.directId to convert(it)
        }
    }

    private fun convert(banner: TBannerV2): UacBannerModerationInfo {
        return UacBannerModerationInfo(
            bannerStatuses = convertToBannerStatuses(banner),
            moderationReasons = convertToModerationReasons(banner),
            assetModerationReasons = convertToAssetModerationReasons(banner),
        )
    }

    private fun convertToBannerStatuses(banner: TBannerV2): BannerStatusesInfo {
        return BannerStatusesInfo(
            bannerId = banner.meta.directId,
            campaignId = banner.meta.campaignId,
            statusModerate = calcStatusModerate(banner),
            imageStatusModerate = calcBannerImageStatusModerate(banner),
            creativeStatusModerate = calcBannerCreativeStatusModerate(banner),
            siteLinksSetStatusModerate = calcSiteinksSetStatusModerate(banner),
            flags = calcBannerFlags(banner),
        )
    }

    fun getModerationDiagsByBannerIds(
        shard: Int,
        bannerIds: List<Long>
    ): Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> {
        return moderationReasonService
            .getRejectReasonDiagsByShard(shard, getReasonsTypeToBannerIds(bannerIds))
    }

    fun getAssetsModerationDiagsByBannerIds(
        clientId: ClientId,
        bannerIds: List<Long>
    ): Map<Long, Map<BannerAssetType, List<ModerationDiag>>> {
        return moderationReasonService.getBannerAssetsRejectReasonDiags(clientId, bannerIds)
    }

    fun getAssetsModerationDiagsByBannerIds(
        shard: Int,
        bannerIds: List<Long>
    ): Map<Long, Map<BannerAssetType, List<ModerationDiag>>> {
        return moderationReasonService.getBannerAssetsRejectReasonDiagsByShard(shard, bannerIds)
    }

    private fun getReasonsTypeToBannerIds(bannerIds: List<Long>) = mapOf(
        ModerationReasonObjectType.BANNER to bannerIds,
        ModerationReasonObjectType.IMAGE to bannerIds,
        ModerationReasonObjectType.VIDEO_ADDITION to bannerIds,
        ModerationReasonObjectType.SITELINKS_SET to bannerIds,
        ModerationReasonObjectType.HTML5_CREATIVE to bannerIds,
    )

    /**
     * Поставить в очередь задание на обновление объявлений
     */
    fun updateAdsDeferred(
        clientId: ClientId,
        operatorUid: Long,
        filledCampaignId: String
    ) {
        val shard = shardHelper.getShardByClientId(clientId)
        val jobParams = UpdateAdsJobParams(operatorUid, filledCampaignId)
        val jobType = DbQueueJobTypes.UAC_UPDATE_ADS
        val jobId = dbQueueRepository.insertJob(shard, jobType, clientId, operatorUid, jobParams).id
        logger.info("Added job with id $jobId")
    }
}
