package ru.yandex.direct.core.entity.uac.service

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory
import ru.yandex.direct.core.entity.uac.converter.UacContentConverter.toSitelink
import ru.yandex.direct.core.entity.uac.model.BaseUacTrackingInfo
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.CampaignByAccountRequest
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectCampaign
import ru.yandex.grut.objects.proto.client.Schema

@Service
@Lazy
class UacCampaignService(
    rmpCampaignService: RmpCampaignService,
    campaignService: CampaignService,
    campaignSubObjectAccessCheckerFactory: CampaignSubObjectAccessCheckerFactory,
    uacClientService: YdbUacClientService,
    private val uacContentService: UacContentService,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val uacYdbCampaignContentRepository: UacYdbCampaignContentRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
) : BaseUacCampaignService(
    rmpCampaignService,
    campaignService,
    campaignSubObjectAccessCheckerFactory,
    uacClientService,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(UacCampaignService::class.java)
    }

    override fun getDirectCampaignIdById(id: String): Long? {
        return uacYdbDirectCampaignRepository.getDirectCampaignById(id)?.directCampaignId
    }

    override fun getCampaignById(id: String): UacYdbCampaign? {
        return uacYdbCampaignRepository.getCampaign(id)
    }

    override fun getLogger(): Logger = logger

    override fun deleteBrief(campaignId: String) {
        uacYdbDirectCampaignRepository.delete(campaignId)
        uacYdbCampaignContentRepository.deleteByCampaignId(campaignId)
        uacYdbCampaignRepository.delete(campaignId)
    }

    /*
    * Т.к. все новые кампании сохраняются в груте, то необходимость YDB'шной реализации отпадает
     */
    override fun getCampaignsWithRetargetingCondition(
        accountId: String,
        retargetingConditionId: Long
    ): List<UacYdbCampaign> {
        throw NotImplementedError()
    }

    override fun getMinBannerIdForCampaign(id: String) = uacYdbDirectAdRepository.getMinDirectBannerIdInCampaign(id)

    private fun getDirectCampaignByDirectCampaignId(directCampaignId: Long): UacYdbDirectCampaign? {
        return uacYdbDirectCampaignRepository.getDirectCampaignByDirectCampaignId(directCampaignId)
    }

    fun getDirectCampaignById(id: String): UacYdbDirectCampaign? {
        return uacYdbDirectCampaignRepository.getDirectCampaignById(id)
    }

    override fun getCampaignIdByDirectCampaignId(directCampaignId: Long) =
        getDirectCampaignByDirectCampaignId(directCampaignId)?.id

    fun fetchContents(campaign: UacYdbCampaign): FetchedContents {
        val contentsContainer = uacContentService.getCampaignContentsContainer(campaign)
        val contents = uacContentService.getContents(contentsContainer.mediaContents.mapNotNull { it.contentId })

        return FetchedContents(
            mediaContents = contents,
            texts = contentsContainer.textContents.mapNotNull { it.text },
            titles = contentsContainer.titleContents.mapNotNull { it.text },
            sitelinks = contentsContainer.sitelinkContents.map { toSitelink(it) },
        )
    }

    override fun getDirectCampaignStatus(ydbCampaign: UacYdbCampaign) =
        getDirectCampaignById(ydbCampaign.id)?.status

    override fun getAppIdsByAccountId(accountId: String): List<String> {
        return uacYdbCampaignRepository.getAppIdsByAccount(CampaignByAccountRequest(accountId = accountId.toIdLong()))
            .map { it.toIdString() }
    }

    override fun getTextContentsByMediaType(campaingId: Long): Map<MediaType, List<String>> {
        val uacCampaignId =
            uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(listOf(campaingId))[campaingId]
                ?: return emptyMap()
        return uacYdbCampaignContentRepository.getTypeToTextsByCampaignId(uacCampaignId)
    }

    override fun getTrackingInfosByAccountIdAndAppId(accountId: String, appId: String): List<BaseUacTrackingInfo> {
        return uacYdbCampaignRepository.getTrackingInfoByAccountAndAppId(accountId, appId)
    }

    override fun getMediaCampaignContentsForCampaign(campaingId: Long): List<UacYdbCampaignContent> {
        return uacYdbCampaignContentRepository.getCampaignContents(campaignId = campaingId.toIdString())
            .filter { it.removedAt == null }
            .filter { it.contentId != null }
    }

    override fun getCampaignProtoByDirectCampaignId(directCampaignId: Long): Schema.TCampaign? {
        throw UnsupportedOperationException()
    }
}
