package ru.yandex.direct.core.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.canvas.client.CanvasClient
import ru.yandex.direct.canvas.client.model.exception.CanvasClientException
import ru.yandex.direct.canvas.client.model.html5.Html5BatchResponse
import ru.yandex.direct.canvas.client.model.html5.Html5SourceResponse
import ru.yandex.direct.canvas.client.model.html5.Html5Tag
import ru.yandex.direct.canvas.client.model.video.AdditionResponse
import ru.yandex.direct.canvas.client.model.video.UacVideoCreativeType
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse
import ru.yandex.direct.core.entity.uac.model.CreativeType
import ru.yandex.direct.core.entity.uac.model.FileData
import ru.yandex.direct.core.entity.uac.validation.CanvasValidationDefectParams
import ru.yandex.direct.core.entity.uac.validation.canvasDefect
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import java.util.*

@Service
class UacCanvasService(
    private val canvasClient: CanvasClient
) {
    companion object {
        private val logger = LoggerFactory.getLogger(UacCanvasService::class.java)
    }

    fun uploadVideo(
        clientId: Long,
        fileData: FileData,
        creativeType: CreativeType,
        locale: Locale?
    ): Result<VideoUploadResponse> {
        return try {
            val response = canvasClient.createVideoFromFile(
                clientId, fileData.data, fileData.name ?: "filename", toCanvasCreativeType(creativeType), locale
            )
            logger.info("Upload video to canvas response: {}", JsonUtils.toJson(response))
            Result.successful(response)
        } catch (e: CanvasClientException) {
            logger.error("Failure to upload video from file", e)
            Result.broken(
                ValidationResult.failed(
                    fileData,
                    canvasDefect(CanvasValidationDefectParams(e.validationErrors))
                )
            )
        }
    }

    fun uploadVideoByUrl(
        clientId: Long,
        url: String,
        creativeType: CreativeType,
        locale: Locale?
    ): Result<VideoUploadResponse> {
        return try {
            val response = canvasClient.createVideoFromUrl(
                clientId, url, toCanvasCreativeType(creativeType), locale
            )
            logger.info("Upload video by url to canvas response: {}", JsonUtils.toJson(response))
            Result.successful(response)
        } catch (e: CanvasClientException) {
            logger.error("Failure to upload video from url", e)
            Result.broken(ValidationResult.failed(url, canvasDefect(CanvasValidationDefectParams(e.validationErrors))))
        }
    }

    fun createCreative(clientId: Long, presetId: Long, videoId: String): Result<AdditionResponse> {
        return try {
            val response = canvasClient.createDefaultAddition(clientId, presetId, videoId)
            logger.info("Create creative response: {}", JsonUtils.toJson(response))
            Result.successful(response)
        } catch (e: CanvasClientException) {
            logger.error("Failure to upload video from url", e)
            Result.broken(ValidationResult.failed(videoId, CommonDefects.invalidValue()))
        }
    }

    fun getVideoById(clientId: Long, videoId: String): Result<VideoUploadResponse> {
        return try {
            val response = canvasClient.getVideoById(clientId, videoId)
            logger.info("Get canvas video file by id response: {}", JsonUtils.toJson(response))
            Result.successful(response)
        } catch (e: CanvasClientException) {
            logger.error("Failure to get canvas video file by id", e)
            Result.broken(ValidationResult.failed(videoId, CommonDefects.invalidValue()))
        }
    }

    private fun toCanvasCreativeType(creativeType: CreativeType): UacVideoCreativeType {
        return when (creativeType) {
            CreativeType.TGO -> UacVideoCreativeType.TEXT
            CreativeType.RMP -> UacVideoCreativeType.MOBILE_CONTENT
            CreativeType.NON_SKIPPABLE_CPM -> UacVideoCreativeType.NON_SKIPPABLE_CPM
            CreativeType.CPM -> UacVideoCreativeType.CPM
            else -> throw IllegalStateException("Illegal creativeType ${creativeType.getType()}")
        }
    }

    fun uploadHtml5(
        clientId: Long,
        fileData: FileData,
        locale: Locale?,
        html5Tag: Html5Tag?
    ): Result<Html5SourceResponse> {
        return try {
            val response = canvasClient.uploadHtml5(
                clientId,
                fileData.data,
                fileData.name ?: "html5_filename",
                html5Tag ?: Html5Tag.PLAYABLE,
                locale
            )
            logger.info("Upload html5 response: {}", JsonUtils.toJson(response))
            Result.successful(response)
        } catch (e: CanvasClientException) {
            logger.error("Failure to upload html5", e)
            Result.broken(
                ValidationResult.failed(
                    fileData,
                    canvasDefect(CanvasValidationDefectParams(e.validationErrors))
                )
            )
        }
    }

    fun createHtml5Creative(clientId: Long, name: String?, sourceId: String): Result<Html5BatchResponse> {
        return try {
            val response = canvasClient.createHtml5Batch(clientId, name ?: "Новый креатив", sourceId)
            logger.info("Create html5 batch response: {}", JsonUtils.toJson(response))
            Result.successful(response)
        } catch (e: CanvasClientException) {
            logger.error("Failure to create html5 batch", e)
            Result.broken(ValidationResult.failed(sourceId, CommonDefects.invalidValue()))
        }
    }

    fun uploadHtml5CreativeToDirect(clientId: Long, userId: Long, batchId: String, creativeId: Long): Result<Long> {
        return try {
            val response = canvasClient.uploadCreativeToDirect(clientId, userId, batchId, creativeId)
            logger.info("Upload html5 creative to direct response: {}", JsonUtils.toJson(response))
            Result.successful(response[0])
        } catch (e: CanvasClientException) {
            logger.error("Failure to upload html5 creative to direct", e)
            Result.broken(ValidationResult.failed(creativeId, CommonDefects.invalidValue()))
        }
    }
}
