package ru.yandex.direct.core.entity.uac.service

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.model.CampaignContentsContainer
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAccountRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.result.Result

@Service
@Lazy
class UacContentService(
    private val uacYdbContentRepository: UacYdbContentRepository,
    private val uacYdbCampaignContentRepository: UacYdbCampaignContentRepository,
    private val uacYdbAccountRepository: UacYdbAccountRepository,
) : BaseUacContentService() {

    override fun deleteContent(id: String): Result<Void> {
        uacYdbContentRepository.deleteContent(id)
        return Result.successful(null)
    }

    override fun checkVisibleContent(clientId: ClientId, content: UacYdbContent): Boolean {
        val account = uacYdbAccountRepository.getAccountByClientId(clientId.asLong()) ?: return false
        return account.id == content.accountId
    }

    override fun getDbContents(contentIds: Collection<String>) = uacYdbContentRepository.getContents(contentIds)

    override fun getCampaignContents(campaign: UacYdbCampaign) =
        uacYdbCampaignContentRepository.getCampaignContents(campaign.id)

    override fun insertContent(content: UacYdbContent): Result<Void> {
        uacYdbContentRepository.saveContents(listOf(content))
        return Result.successful(null)
    }

    override fun insertContents(contents: List<UacYdbContent>): Result<Void> {
        uacYdbContentRepository.saveContents(contents)
        return Result.successful(null)
    }

    fun updateContent(kthModelChanges: KtModelChanges<String, UacYdbContent>) {
        uacYdbContentRepository.updateMetaAndImageHash(kthModelChanges)
    }

    override fun getDbContentByHash(hash: String) = uacYdbContentRepository.getContentByHash(hash)

    fun getCampaignContentsContainer(campaign: UacYdbCampaign): CampaignContentsContainer {
        val result = CampaignContentsContainer()

        val campaignContents = getCampaignContents(campaign)
        for (content in campaignContents) {
            if (content.removedAt == null) {
                when {
                    content.contentId != null -> result.mediaContents.add(content)
                    content.type == MediaType.TEXT -> result.textContents.add(content)
                    content.type == MediaType.TITLE -> result.titleContents.add(content)
                    content.type == MediaType.SITELINK -> result.sitelinkContents.add(content)
                }
            }
        }

        result.mediaContents.sortBy { it.order }
        result.textContents.sortBy { it.order }
        result.titleContents.sortBy { it.order }
        result.sitelinkContents.sortBy { it.order }
        return result
    }

    override fun getDbContentsByDirectCampaignId(
        directCampaignId: Long,
        clientId: ClientId,
        mediaType: MediaType,
    ): Collection<UacYdbContent> {
        val contentIds = uacYdbCampaignContentRepository.getContentIdsByDirectCampaignId(directCampaignId, mediaType)
        return if (contentIds.isEmpty()) emptyList() else uacYdbContentRepository.getContents(contentIds)
    }
}
