package ru.yandex.direct.core.entity.uac.service

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import java.time.Duration
import ru.yandex.direct.core.entity.uac.model.AdvType

/**
 * Определяет, какую базу использовать для работы с заявками
 */
@Component
@Lazy
class UacDbDefineService(
    private val featureService: FeatureService,
    private val ydbCampaignRepository: UacYdbCampaignRepository,
    private val ydbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val ydbContentRepository: UacYdbContentRepository,
) {
    /**
     * Используется GrUT если:
     * - Кампания не лежит в ydb
     */
    fun useGrut(campaignId: String) =
        isCampaignInGrut(campaignId)

    // только для ручки /campaign/direct
    fun useGrutForDirectCampaignId(directCampaignId: Long) =
        isCampaignInGrut(directCampaignId)

    /**
     * Используется GrUT для создания новой заявки если:
     * - Использование GrUT включено по проперте
     * - Использование GrUT включено на клиента
     */
    @Deprecated("Use method with advType param")
    fun useGrutForNewCampaign(clientId: ClientId) =
        featureService.isEnabledForClientId(clientId, FeatureName.UC_UAC_CREATE_BRIEF_IN_GRUT_INSTEAD_OF_YDB)

    /**
     * Используется GrUT для создания новой заявки если:
     * - Тип кампании поддерживается в GrUT
     * - Использование GrUT включено по проперте
     * - Использование GrUT включено на клиента
     */
    fun useGrutForNewCampaign(clientId: ClientId, advType: AdvType, isEcom: Boolean = false): Boolean {
        val isEnabledByAdvType = !isEcom &&
            featureService.isEnabledForClientId(clientId, getFeatureNameByAdvType(advType))

        if (isEnabledByAdvType) {
            return true
        }

        val isEnabledByEcomFlag = isEcom
            && featureService.isEnabledForClientId(clientId, FeatureName.UC_UAC_CREATE_ECOM_BRIEF_IN_GRUT_INSTEAD_OF_YDB)

        if (isEnabledByEcomFlag) {
            return true
        }

        return false
    }

    fun useGrutForContentId(contentId: String) =
        isContentInGrut(contentId)

    // только для ручки content/hash
    fun useGrutForContentHash(hash: String) =
        isContentInGrutByHash(hash)

    private fun isCampaignInGrut(campaignId: String) =
        ydbCampaignRepository.getCampaign(campaignId) == null

    // только для ручки /campaign/direct
    private fun isCampaignInGrut(directCampaignId: Long) =
        ydbDirectCampaignRepository.getDirectCampaignByDirectCampaignId(directCampaignId) == null

    private fun isContentInGrut(contentId: String) =
        ydbContentRepository.getContents(listOf(contentId)).isEmpty()

    // только для ручки content/hash
    private fun isContentInGrutByHash(hash: String) =
        ydbContentRepository.getContentByHash(hash) == null

    private fun getFeatureNameByAdvType(advType: AdvType): FeatureName {
        return when (advType) {
            AdvType.MOBILE_CONTENT  -> FeatureName.UC_UAC_CREATE_MOBILE_CONTENT_BRIEF_IN_GRUT_INSTEAD_OF_YDB
            AdvType.TEXT            -> FeatureName.UC_UAC_CREATE_BRIEF_IN_GRUT_INSTEAD_OF_YDB
            AdvType.CPM_BANNER      -> FeatureName.UC_UAC_CREATE_MEDIA_BRIEF_IN_GRUT_INSTEAD_OF_YDB
        }
    }
}
