package ru.yandex.direct.core.entity.uac.service

import org.apache.commons.lang3.StringUtils
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbRejectReasonRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbRejectReason
import ru.yandex.direct.i18n.Language

@Lazy
@Service
class UacRejectReasonService(
    private val uacYdbRejectReasonRepository: UacYdbRejectReasonRepository,
) {
    companion object {
        val DEFAULT_FORMAT_PARAMS = mapOf(
            "{direct_adv_rules}" to "direct_adv_rules",
            "{adv_rules}" to "adv_rules",
            "{general_adv_rules}" to "general_adv_rules",
            "{direct_display_rules}" to "direct_display_rules",
            "{tld_for_legal}" to "legal",
            "{legal}" to "legal",
        )
        val FORMAT_PARAMS_BY_TLD = mapOf(
            "com" to DEFAULT_FORMAT_PARAMS.mapValues {
                when (it.key) {
                    "{direct_adv_rules}" -> "direct_adv_rules_ch"
                    "{adv_rules}" -> "adv_rules_ch"
                    "{general_adv_rules}" -> "general_adv_rules_ch"
                    "{direct_display_rules}" -> "direct_display_rules_ch"
                    else -> it.value
                }
            }
        )
        val LANGUAGE_TO_TLD = mapOf(
            Language.RU to "ru",
            Language.UK to "ua",
            Language.EN to "com",
            Language.TR to "com.tr"
        )
    }

    fun getRejectReasons(
        uacCampaignContents: List<UacYdbCampaignContent>,
        lang: Language = Language.RU,
    ): Map<String, List<ModerationDiagData>?> {
        val diagIds = uacCampaignContents
            .mapNotNull { it.rejectReasons }
            .flatten()
            .mapNotNull { it.diagId?.toInt() }
            .toSet()
        val titleAndDescriptionByDiagId = uacYdbRejectReasonRepository.getRejectReasons(diagIds, lang)
            .associateBy { it.diagId }
            .mapKeys { it.key.toString() }
            .mapValues {
                it.value.title to getFormattedReasonDescription(it.value, lang)
            }

        return uacCampaignContents
            .associateBy { it.id }
            .mapValues {
                it.value.rejectReasons?.map { rejectReason ->
                    val titleAndDescription = rejectReason.diagId?.let { diagId ->
                        titleAndDescriptionByDiagId[diagId]
                    }
                    rejectReason.shortText = titleAndDescription?.first ?: rejectReason.shortText
                    rejectReason.diagText = titleAndDescription?.second ?: rejectReason.diagText
                    rejectReason
                }
            }
    }

    private fun getFormattedReasonDescription(reason: UacYdbRejectReason, requestedLang: Language): String {
        val reasonLang = Language.fromLangString(reason.lang)
        val lang = if (requestedLang == Language.UK && reasonLang == Language.EN) requestedLang else reasonLang
        val tld = LANGUAGE_TO_TLD.getOrDefault(lang, "com")

        val formatParams = mutableMapOf<String, String>()
        formatParams.putAll(FORMAT_PARAMS_BY_TLD.getOrDefault(tld, DEFAULT_FORMAT_PARAMS))
        formatParams["{lang}"] = lang.langString
        formatParams["{tld}"] = tld

        val searchList = formatParams.keys.toTypedArray()
        val replacementList = formatParams.values.toTypedArray()
        return StringUtils.replaceEach(reason.description, searchList, replacementList)
    }
}
