package ru.yandex.direct.core.entity.uac.service

import java.time.LocalDateTime
import ru.yandex.direct.core.entity.retargeting.Constants
import ru.yandex.direct.core.entity.retargeting.model.ConditionType
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.model.Rule
import ru.yandex.direct.core.entity.retargeting.model.RuleType
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest
import ru.yandex.direct.core.entity.uac.model.AgePoint
import ru.yandex.direct.core.entity.uac.model.Gender
import ru.yandex.direct.core.entity.uac.model.Socdem
import ru.yandex.direct.core.entity.uac.model.SocdemGoal
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRule
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoal
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoalType

object UacRetargetingService {
    private fun <T : SocdemGoal> getSelectedGoals(
        lowerPoint: T,
        highPoint: T,
        orderedGoals: List<T>,
        includeRightBorder: Boolean,
    ): List<Goal> {
        return orderedGoals
            .filter {
                it.order >= lowerPoint.order && ((it.order < highPoint.order)
                    || (includeRightBorder && it.order == highPoint.order))
            }
            .map { idToGoal(it.goalId) }
    }

    private fun idToGoal(id: Long): Goal {
        val x = Goal();
        x.id = id;
        return x;
    }

    private fun convertIncome(lowerPoint: Socdem.IncomeGrade?, upperPoint: Socdem.IncomeGrade?): List<Goal> {
        if (lowerPoint == null || upperPoint == null) {
            return emptyList()
        }

        if (lowerPoint == Socdem.IncomeGrade.LOW && upperPoint == Socdem.IncomeGrade.PREMIUM) {
            return emptyList();
        }

        return getSelectedGoals(lowerPoint, upperPoint, Socdem.IncomeGrade.values().toList(), true);
    }

    private fun convertAge(lowerPoint: AgePoint, upperPoint: AgePoint): List<Goal> {
        if (lowerPoint == AgePoint.AGE_0 && upperPoint == AgePoint.AGE_INF) {
            return emptyList();
        }

        return getSelectedGoals(lowerPoint, upperPoint, AgePoint.values().toList(), false);
    }

    private fun ruleFromGoals(goals: List<Goal>): Rule {
        return Rule().withType(RuleType.OR)
            .withInterestType(null)
            .withGoals(goals)
    }

    private fun convertGenders(genders: List<Gender>): List<Goal> {
        if (genders.contains(Gender.MALE) && genders.contains(Gender.FEMALE)) {
            return emptyList()
        }

        return genders.map { idToGoal(it.goalId) }
    }

    fun convertSocdemToRetargetingRules(socdem: Socdem?): List<Rule> {
        if (socdem == null) {
            return emptyList()
        }

        return listOf(
            ruleFromGoals(convertAge(socdem.ageLower, socdem.ageUpper)),
            ruleFromGoals(convertGenders(socdem.genders)),
            ruleFromGoals(convertIncome(socdem.incomeLower, socdem.incomeUpper))
        ).filter { it.goals.size > 0 }
    }

    fun convertUacToCoreGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
        interestType: CryptaInterestType?,
        parentIdToLalSegmentId: Map<Long, Long>,
    ): Goal {
        val goal = when (uacGoal.type) {
            UacRetargetingConditionRuleGoalType.MOBILE -> convertToUacGoal(uacGoal, uacGoal.time)
            UacRetargetingConditionRuleGoalType.LAL -> convertToUacLalGoal(uacGoal, parentIdToLalSegmentId)
            UacRetargetingConditionRuleGoalType.AUDIENCE -> convertToUacGoal(uacGoal, Constants.MAX_GOAL_TIME)
            UacRetargetingConditionRuleGoalType.LAL_AUDIENCE -> convertToUacLalGoal(uacGoal, parentIdToLalSegmentId)
            UacRetargetingConditionRuleGoalType.SEGMENT -> convertToUacSegmentGoal(uacGoal)
            UacRetargetingConditionRuleGoalType.LAL_SEGMENT -> convertToUacLalSegmentGoal(uacGoal, parentIdToLalSegmentId)
            else -> convertToUacGoal(uacGoal, uacGoal.time)
        }
        return goal.withInterestType(interestType) as Goal
    }

    /**
     * Конвертер с ядрового типа цели в uc
     * В Uc(ТГО) кампании пока подержанны только два типа от custom audience
     */
    fun toUcGoalType(
        goalType: GoalType?,
    ): UacRetargetingConditionRuleGoalType? =
        when (goalType) {
            GoalType.INTERESTS -> UacRetargetingConditionRuleGoalType.INTERESTS
            GoalType.HOST -> UacRetargetingConditionRuleGoalType.HOST
            else -> null
        }

    /**
     * Конвертирует uacGoal с типом MOBILE или цели без типа в директовую цель
     * @param uacGoal: исходная цель
     * @return директовая цель
     **/
    private fun convertToUacGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
        time: Int?
    ) = Goal()
        .withId(uacGoal.id)
        .withTime(time)

    /**
     * Конвертирует uacGoal с типом LAL в директовую цель
     * Изначально в uacGoal есть только id родителя, поэтому нужен маппер parentGoalId -> lalSegmentId
     * @param uacGoal: исходная цель
     * @param parentIdToLalSegmentId: Маппер parentGoalId -> lalSegmentId
     * @return директовая цель
     **/
    private fun convertToUacLalGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
        parentIdToLalSegmentId: Map<Long, Long>
    ) = Goal()
        .withId(parentIdToLalSegmentId[uacGoal.id])
        .withTime(Constants.MAX_GOAL_TIME)
        .withParentId(uacGoal.id)

    /**
     * Конвертирует uacGoal с типом SEGMENT в директовую цель
     * Изначально в uacGoal нет id для сегмента, только его counterId. Для восстановления id цели нужны мапперы
     * counterId -> segmentId
     * @param uacGoal: исходная цель
     * @return директовая цель
     **/
    private fun convertToUacSegmentGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
    ) = Goal()
        .withId(uacGoal.id)
        .withTime(Constants.MAX_GOAL_TIME)
        .withCounterId(uacGoal.segmentInfo?.counterId)
        .withCounterName(uacGoal.segmentInfo?.name)
        .withDomain(uacGoal.segmentInfo?.domain)

    /**
     * Конвертирует uacGoal с типом LAL_SEGMENT в директовую LAL цель, parent которой - сегмент
     * Изначально в uacGoal нет id для сегмента, только его counterId. Для восстановления id цели нужны мапперы
     * counterId -> segmentId, а также parentGoalId -> lalSegmentId для взятия id LAL-сегмента
     * @param uacGoal: исходная цель
     * @param parentIdToLalSegmentId: Маппер parentGoalId -> lalSegmentId
     * @return директовая цель
     **/
    private fun convertToUacLalSegmentGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
        parentIdToLalSegmentId: Map<Long, Long>,
    ) = Goal()
        .withId(parentIdToLalSegmentId[uacGoal.id])
        .withTime(Constants.MAX_GOAL_TIME)
        .withParentId(uacGoal.id)
        .withCounterId(uacGoal.segmentInfo?.counterId)
        .withCounterName(uacGoal.segmentInfo?.name)
        .withDomain(uacGoal.segmentInfo?.domain)

    fun convertUacToCoreRules(
        uacRetargetingCondition: UacRetargetingCondition?,
        parentIdToLalSegmentId: Map<Long, Long>,
    ): List<Rule> {
        if (uacRetargetingCondition == null) {
            return listOf()
        }

        return uacRetargetingCondition.conditionRules.map {
            Rule().withType(convertUacRuleType(it.type))
                .withInterestType(it.interestType)
                .withGoals(it.goals.map { ex -> convertUacToCoreGoal(ex, it.interestType, parentIdToLalSegmentId) })
        }
    }

    fun toCoreRetargetingCondition(
        retargetingCondition: UacRetargetingCondition?,
        clientId: Long,
        socdem: Socdem? = null,
        type: ConditionType = ConditionType.interests,
        parentIdToLalSegmentId: Map<Long, Long> = mapOf(),
    ): RetargetingCondition {
        val retargetingName =
            String.format("UC autoretargeting %d %s", clientId, LocalDateTime.now().toString())

        return (RetargetingCondition()
            .withName(retargetingCondition?.name ?: retargetingName)
            .withType(type)
            .withClientId(clientId)
            .withTargetInterest(TargetInterest().withAdGroupId(0L))
            .withRules(
                convertSocdemToRetargetingRules(socdem)
                    + convertUacToCoreRules(retargetingCondition, parentIdToLalSegmentId)
            )) as RetargetingCondition
    }

    private fun convertUacRuleType(type: UacRetargetingConditionRule.RuleType): RuleType {
        return when (type) {
            UacRetargetingConditionRule.RuleType.OR -> RuleType.OR;
            UacRetargetingConditionRule.RuleType.ALL -> RuleType.ALL
            UacRetargetingConditionRule.RuleType.NOT -> RuleType.NOT
        }
    }
}
