package ru.yandex.direct.core.entity.uac.service.appinfo

import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.AppInfoContent
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.core.entity.uac.validation.invalidAppId
import ru.yandex.direct.core.entity.uac.validation.multipleParams
import ru.yandex.direct.geosearch.model.Lang
import ru.yandex.direct.utils.fromJson
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

interface AppInfoGetter {
    val source: Store
    val platform: Platform
    val storeHosts: List<String>

    companion object {
        private const val GOOGLE_LANG = "hl"
        private const val APPLE_LANG = "l"
        private const val GOOGLE_REGION = "gl"
        private const val GOOGLE_ID = "id"
    }

    fun getAppInfo(model: UacYdbAppInfo, addRecommendations: Boolean = true): AppInfo
    fun getContent(model: UacYdbAppInfo): List<AppInfoContent>

    fun getDataAsMap(model: UacYdbAppInfo): Map<String, Any> {
        return fromJson(model.data)
    }

    fun isSuitable(urlParts: UrlParts) = storeHosts.contains(urlParts.domain)

    fun getAppInfoByUrl(url: String, defaultDefect: Defect<*>): ValidationResult<AppInfo, Defect<*>>? {
        val urlParts = UrlParts.fromUrl(url)

        if (!storeHosts.contains(urlParts.domain)) {
            throw IllegalStateException("URL host not in " + storeHosts.joinToString(", "))
        }

        val storeUrlOptional = MobileAppStoreUrlParser.parse(url)
        if (storeUrlOptional.isPresent) {
            val paramCheck = checkParams(urlParts)
            if (paramCheck != null) {
                return ValidationResult.failed(null, paramCheck)
            }

            val storeUrl = storeUrlOptional.get()
            val appInfo = AppInfo(
                id = null,
                appId = storeUrl.storeContentId,
                bundleId = null,
                title = null,
                subtitle = null,
                description = null,
                iconUrl = null,
                language = getLanguageByUrl(urlParts),
                platform = platform,
                region = storeUrl.storeCountry,
                vendor = null,
                rating = null,
                reviews = null,
                ageLimit = null,
                minOsVersion = null,
                url = url,
                currency = null,
                price = null,
                interests = null,
                recommendedCpa = null,
                recommendedCpi = null
            )
            return ValidationResult.success(appInfo)
        }
        return ValidationResult.failed(null, defaultDefect)
    }

    private fun checkParams(urlParts: UrlParts): Defect<*>? {
        if (platform == Platform.ANDROID) {
            val parameterMap = urlParts.parameters
                ?.filter { it.right?.isNotEmpty() ?: false }
                ?.groupBy({ it.left }, { it.right })

            if (parameterMap == null || !parameterMap.contains(GOOGLE_ID))
                return invalidAppId()

            val multiple = sequenceOf(
                parameterMap.getOrDefault(GOOGLE_LANG, emptyList()).size,
                parameterMap.getOrDefault(GOOGLE_ID, emptyList()).size,
                parameterMap.getOrDefault(GOOGLE_REGION, emptyList()).size
            ).any { it > 1 }

            if (multiple) {
                return multipleParams()
            }
        }
        return null
    }

    private fun getLanguageByUrl(urlParts: UrlParts): String {
        val parametersMap = urlParts.parameters
            ?.filter { it.right?.isNotEmpty() ?: false }
            ?.associate { it.left to it.right }
        return parametersMap?.get(if (platform == Platform.ANDROID) GOOGLE_LANG else APPLE_LANG) ?: Lang.RU.language
    }
}
