package ru.yandex.direct.core.entity.uac.service.appinfo

import org.jsoup.Jsoup
import org.springframework.stereotype.Component
import org.springframework.web.util.UriComponentsBuilder
import ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants
import ru.yandex.direct.core.entity.uac.operation.UacUploadImageByUrlToAvatarsOperation
import ru.yandex.direct.core.entity.uac.service.UacAvatarsService
import ru.yandex.direct.tracing.Trace
import java.net.MalformedURLException
import java.net.URL

@Component
class AppInfoInputProcessor(
    private val uacAvatarsService: UacAvatarsService,
) {

    companion object {
        val NOT_ALLOW_BANNER_LETTERS_REGEX = BannerLettersConstants.NOT_ALLOW_BANNER_LETTERS_RE.toRegex()
        val SPACE_BEFORE_COMMA_REGEX = """ +,""".toRegex()
        val COMMA_WITHOUT_SPACE_REGEX = """,([^ ])""".toRegex()

        // todo вынести куда-нибудь
        val avatarsHosts = listOf(
            "avatars.mdst.yandex.net",
            "avatars.mds.yandex.net",
        )
    }

    fun processText(text: String?): String? {
        return text
            ?.replace("－", " - ")
            ?.replace(NOT_ALLOW_BANNER_LETTERS_REGEX, "")
            ?.replace(SPACE_BEFORE_COMMA_REGEX, ",")
            ?.replace(COMMA_WITHOUT_SPACE_REGEX, """, $1""")
    }

    fun fixDescription(description: String?): String? {
        val formatted = description
            ?.replace("<p>", "\n")
            ?.replace("</p>", "\n")
            ?.replace("<br>", "\n")

        return Trace.current().profile("jsoup_parse").use {
            try {
                Jsoup.parse(formatted).text()
            } catch (e: RuntimeException) {
                formatted
            }
        }
    }

    fun fixAvatarsUrl(iconUrl: String?): String? {
        val parsed = try {
            URL(iconUrl)
        } catch (e: MalformedURLException) {
            null
        }
        if (parsed != null && parsed.host in avatarsHosts) {
            return UriComponentsBuilder.fromUri(parsed.toURI())
                .scheme("https")
                .build()
                .toString()
        }
        return iconUrl
    }

    fun isAvatarsUrl(iconUrl: String?): Boolean {
        val parsed = try {
            URL(iconUrl)
        } catch (e: MalformedURLException) {
            null
        }
        return parsed != null && parsed.host in avatarsHosts
    }

    fun fixAgeLimit(ageLimit: String?): Int? {
        return ageLimit
            ?.removeSuffix("+")
            ?.toIntOrNull()
    }

    fun fixIconUrl(iconUrl: String?): String? {
        if (isAvatarsUrl(iconUrl)) {
            return fixAvatarsUrl(iconUrl)
        } else if (iconUrl != null) {
            val uploadOperationResult = UacUploadImageByUrlToAvatarsOperation(uacAvatarsService, iconUrl).prepareAndApplyWithCheck()
            if (!uploadOperationResult.isSuccessful) {
                return iconUrl
            }
            return uacAvatarsService.getReadUrl(uploadOperationResult.result, "orig")
        } else {
            return iconUrl
        }
    }
}
