package ru.yandex.direct.core.entity.uac.service.appinfo

import com.fasterxml.jackson.databind.JsonMappingException
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.image.service.ImageConstants.BANNER_REGULAR_IMAGE_MIN_SIZE
import ru.yandex.direct.core.entity.image.service.ImageConstants.MAX_IMAGE_LINEAR_SIZE_IN_PIXEL
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser.GOOGLE_PLAY_DOMAIN
import ru.yandex.direct.core.entity.uac.CATEGORY_TO_INTEREST
import ru.yandex.direct.core.entity.uac.GOOGLE_PLAY_CATEGORIES
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.AppInfoContent
import ru.yandex.direct.core.entity.uac.model.AppInfoScreenMeta
import ru.yandex.direct.core.entity.uac.model.GooglePlayAppInfoData
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.RecommendedCostType
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbRecommendedCostRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.utils.fromJson
import java.net.URLEncoder

@Lazy
@Component
class GooglePlayAppInfoGetter(
    private val appInfoInputProcessor: AppInfoInputProcessor,
    private val uacYdbRecommendedCostRepository: UacYdbRecommendedCostRepository,
) : AppInfoGetter {

    companion object {
        private val logger: Logger = LoggerFactory.getLogger(GooglePlayAppInfoGetter::class.java)

        private val DESCRIPTION_REGEX = """^\d+?\.\s*""".toRegex()
    }

    override val source = Store.GOOGLE_PLAY
    override val platform = Platform.ANDROID
    override val storeHosts = listOf(GOOGLE_PLAY_DOMAIN)

    private fun getData(model: UacYdbAppInfo): GooglePlayAppInfoData? {
        return try {
            fromJson(model.data)
        } catch (e: JsonMappingException) {
            logger.error("Failed to parse google play app info", e)
            null
        }
    }

    fun subtitle(description: String?): String? {
        val text = appInfoInputProcessor.fixDescription(description)
            ?.replace(DESCRIPTION_REGEX, "")
            ?: return null

        var subtitle = ""
        for (char in text) {
            when {
                char == '\n' -> break
                char == ' ' && subtitle.isNotBlank() && subtitle.last() in ".?!" -> break
                else -> subtitle += char
            }
        }

        if (subtitle.isEmpty()) {
            return null
        }

        return appInfoInputProcessor.processText(subtitle)
    }

    fun appPageUrl(appId: String, region: String, language: String): String {
        val encodedAppId = URLEncoder.encode(appId, Charsets.UTF_8)
        val encodedRegion = URLEncoder.encode(region, Charsets.UTF_8)
        val encodedLanguage = URLEncoder.encode(language, Charsets.UTF_8)
        return "https://play.google.com/store/apps/details?hl=$encodedLanguage&gl=$encodedRegion&id=$encodedAppId"
    }

    fun interests(data: GooglePlayAppInfoData?): Set<Int> {
        val categories = data?.subcategoryEng ?: return setOf()

        return categories.map { GOOGLE_PLAY_CATEGORIES[it] }.mapNotNull { CATEGORY_TO_INTEREST[it] }.toSet()
    }

    fun getCategory(data: GooglePlayAppInfoData?): String {
        return data?.categories?.firstOrNull()?.lowercase() ?: ""
    }

    override fun getAppInfo(model: UacYdbAppInfo, addRecommendations: Boolean): AppInfo {
        val data = getData(model)
        val recommendedCost =
            if (addRecommendations) uacYdbRecommendedCostRepository.getRecommendedCostByCategoryAndPlatform(
                getCategory(data), model.platform
            ) else null

        return AppInfo(
            id = model.id,
            appId = model.appId,
            region = model.region,
            bundleId = model.bundleId,
            title = appInfoInputProcessor.processText(data?.name),
            subtitle = subtitle(data?.htmlDescription),
            description = data?.description,
            iconUrl = appInfoInputProcessor.fixIconUrl(data?.icon),
            language = model.language,
            platform = model.platform,
            vendor = data?.author?.name,
            rating = data?.rating?.value,
            reviews = data?.rating?.votes,
            ageLimit = appInfoInputProcessor.fixAgeLimit(data?.adult),
            minOsVersion = null,
            url = appPageUrl(model.appId, model.region, model.language),
            currency = data?.price?.currency,
            price = data?.price?.value,
            interests = interests(data),
            recommendedCpa = recommendedCost?.getOrDefault(RecommendedCostType.CPA, null),
            recommendedCpi = recommendedCost?.getOrDefault(RecommendedCostType.CPI, null),
        )
    }

    fun getImageContents(data: GooglePlayAppInfoData?): List<AppInfoContent> {

        fun isValidSize(meta: AppInfoScreenMeta?): Boolean {
            return meta?.origSize == null ||
                meta.origSize.x in BANNER_REGULAR_IMAGE_MIN_SIZE..MAX_IMAGE_LINEAR_SIZE_IN_PIXEL &&
                meta.origSize.y in BANNER_REGULAR_IMAGE_MIN_SIZE..MAX_IMAGE_LINEAR_SIZE_IN_PIXEL
        }

        val contents = data?.screens
            ?.mapIndexedNotNull { index, screen ->
                val url = appInfoInputProcessor.fixAvatarsUrl(screen as String?)
                    ?: return@mapIndexedNotNull null

                val screenEx = data.screensEx?.get(index)
                if (!isValidSize(screenEx?.meta)) {
                    null
                } else {
                    AppInfoContent(
                        type = MediaType.IMAGE,
                        sourceUrl = url,
                    )
                }
            }

        return contents ?: listOf()
    }

    override fun getContent(model: UacYdbAppInfo): List<AppInfoContent> {
        val data = getData(model)
        val imageContents = getImageContents(data)
        val videoContents = data?.videosS3
            ?.entries
            ?.map {
                AppInfoContent(
                    type = MediaType.VIDEO,
                    sourceUrl = it.key,
                    mdsUrl = it.value
                )
            } ?: listOf()
        return imageContents + videoContents
    }
}
