package ru.yandex.direct.core.entity.uac.service.appinfo

import com.fasterxml.jackson.databind.JsonMappingException
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser.ITUNES_APP_STORE_DOMAIN
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser.ITUNES_APP_STORE_OLD_DOMAIN
import ru.yandex.direct.core.entity.uac.CATEGORY_TO_INTEREST
import ru.yandex.direct.core.entity.uac.ITUNES_CATEGORIES
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.AppInfoContent
import ru.yandex.direct.core.entity.uac.model.ITunesAppInfoData
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.RecommendedCostType
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbRecommendedCostRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.utils.fromJson
import java.net.URLEncoder
import kotlin.math.abs

@Lazy
@Component
class ITunesAppInfoGetter(
    private val appInfoInputProcessor: AppInfoInputProcessor,
    private val uacYdbRecommendedCostRepository: UacYdbRecommendedCostRepository,
) : AppInfoGetter {

    override val source = Store.ITUNES
    override val platform = Platform.IOS
    override val storeHosts = listOf(ITUNES_APP_STORE_OLD_DOMAIN, ITUNES_APP_STORE_DOMAIN)

    companion object {
        private val logger = LoggerFactory.getLogger(ITunesAppInfoGetter::class.java)

        private const val ICON_KEY_PREFIX = "artworkUrl"
        private const val CANONICAL_ICON_SIZE = 88
    }

    private fun getData(model: UacYdbAppInfo): ITunesAppInfoData? {
        return try {
            fromJson(model.data)
        } catch (e: JsonMappingException) {
            logger.error("Failed to parse itunes app info", e)
            null
        }
    }

    fun extractIcon(dataMap: Map<String, Any>): String? {
        var delta: Int? = null
        var iconUrl: String? = null
        for ((key, value) in dataMap) {
            if (!key.startsWith(ICON_KEY_PREFIX)) {
                continue
            }
            val size = key.removePrefix(ICON_KEY_PREFIX).toIntOrNull()
                ?: continue
            val currentDelta = abs(CANONICAL_ICON_SIZE - size)
            if (delta == null || currentDelta < delta) {
                delta = currentDelta
                iconUrl = value as? String
            }
            if (delta == 0) {
                break
            }
        }

        return iconUrl
    }

    fun appPageUrl(appId: String, region: String, language: String): String {
        val encodedAppId = URLEncoder.encode(appId, Charsets.UTF_8)
        val encodedRegion = URLEncoder.encode(region, Charsets.UTF_8)
        val encodedLanguage = URLEncoder.encode(language, Charsets.UTF_8)
        return "https://apps.apple.com/$encodedRegion/app/$encodedAppId?l=$encodedLanguage"
    }

    fun interests(data: ITunesAppInfoData?): Set<Int> {
        val categories = data?.genreIds?.map { it.toInt() } ?: return setOf()

        return categories.map { ITUNES_CATEGORIES[it] }.mapNotNull { CATEGORY_TO_INTEREST[it] }.toSet()
    }

    fun getCategory(data: ITunesAppInfoData?): String {
        return data?.primaryGenreName?.lowercase() ?: ""
    }

    override fun getAppInfo(model: UacYdbAppInfo, addRecommendations: Boolean): AppInfo {
        val data = getData(model)
        val dataMap = getDataAsMap(model)
        val icon = data?.iconMds ?: extractIcon(dataMap)
        val recommendedCost =
            if (addRecommendations) uacYdbRecommendedCostRepository.getRecommendedCostByCategoryAndPlatform(
                getCategory(data), model.platform
            ) else null

        return AppInfo(
            id = model.id,
            appId = model.appId,
            bundleId = model.bundleId,
            title = appInfoInputProcessor.processText(data?.trackName),
            subtitle = appInfoInputProcessor.processText(data?.additional?.get("subtitle") as String?),
            description = data?.description,
            iconUrl = appInfoInputProcessor.fixIconUrl(icon),
            language = model.language,
            region = model.region,
            platform = model.platform,
            vendor = data?.artistName,
            rating = data?.averageUserRating,
            reviews = data?.userRatingCount,
            ageLimit = appInfoInputProcessor.fixAgeLimit(data?.contentAdvisoryRating),
            minOsVersion = data?.minimumOsVersion,
            url = appPageUrl(model.appId, model.region, model.language),
            currency = data?.currency,
            price = data?.price,
            interests = interests(data),
            recommendedCpa = recommendedCost?.getOrDefault(RecommendedCostType.CPA, null),
            recommendedCpi = recommendedCost?.getOrDefault(RecommendedCostType.CPI, null),
        )
    }

    override fun getContent(model: UacYdbAppInfo): List<AppInfoContent> {
        val data = getData(model)
        val imageContents = (data?.bigScreenshotUrls ?: data?.screenshotUrls ?: listOf())
            .map { url ->
                AppInfoContent(
                    type = MediaType.IMAGE,
                    sourceUrl = appInfoInputProcessor.fixAvatarsUrl(url),
                )
            }
        val videoContents = data?.videosS3
            ?.entries
            ?.map {
                AppInfoContent(
                    type = MediaType.VIDEO,
                    sourceUrl = it.key,
                    mdsUrl = it.value
                )
            } ?: listOf()
        return imageContents + videoContents
    }
}
