package ru.yandex.direct.core.entity.uac.service.appinfo

import org.apache.http.client.utils.URIBuilder
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl
import ru.yandex.direct.core.entity.uac.model.MobileAppStoreUrlWithOriginal
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.core.entity.uac.service.appinfo.ParseAppStoreContentService.ParseAppStoreContentException
import ru.yandex.direct.core.entity.zora.ZoraService
import ru.yandex.direct.http.smart.error.ErrorUtils
import ru.yandex.direct.utils.JsonUtils.toJson
import ru.yandex.direct.utils.fromJson
import java.net.URI

@Service
class ItunesContentProcessor(
    private val zoraService: ZoraService,
    private val parseAppStoreContentService: ParseAppStoreContentService
) {
    companion object {
        const val LOOKUP_URL = "https://itunes.apple.com/lookup"
        const val BIGGEST_PNG_SCREEN_SIZE = "2778x0w.png"
    }

    fun getAppInfoModel(urls: MobileAppStoreUrlWithOriginal): UacYdbAppInfo {
        val parsedContent = parseAppStoreContentService.parseAppStoreContent(urls)
        if (!parsedContent.hasItunes()) {
            throw ProcessItunesContentException("Parser returned empty itunes result for $urls.originalUrl")
        }

        val data = getLookupInfo(urls.parsedUrl)

        // using JsonMapper on plain Java types fails to serialize protobuf inside it,
        // so convert it to Map
        data["additional"] = fromJson<Map<String, *>>(toJson(parsedContent.itunes))

        return UacYdbAppInfo(
            appId = urls.parsedUrl.storeContentId,
            bundleId = data["bundleId"] as? String,
            language = urls.parsedUrl.storeLanguage.lowercase(),
            region = urls.parsedUrl.storeCountry.lowercase(),
            platform = Platform.IOS,
            source = Store.ITUNES,
            data = toJson(data),
        )
    }

    private fun getLookupInfo(parsedUrl: MobileAppStoreUrl): MutableMap<String, Any> {
        val lookupInfo = downloadLookupInfo(getLookupInfoUrl(parsedUrl))
            .toMutableMap()

        getIconUrl(lookupInfo)?.let {
            lookupInfo["icon"] = it
        }

        getBigScreenshotUrl(lookupInfo["screenshotUrls"] as? List<*>).let {
            lookupInfo["bigScreenshotUrls"] = it as Any
        }

        return lookupInfo
    }

    private fun getIconUrl(lookupInfo: Map<String, Any>): Any? {
        val iconUrlsBySize = lookupInfo
            .filter { it.key.startsWith("artworkUrl") }
            .mapKeys { it.key.removePrefix("artworkUrl").toIntOrNull() }

        return iconUrlsBySize.filter { it.key != null }
            .maxByOrNull { it.key!! }
            ?.value
    }

    private fun getBigScreenshotUrl(screens: List<*>?) =
        screens?.mapNotNull { (it as? String)?.replaceAfterLast('/', BIGGEST_PNG_SCREEN_SIZE) }

    private fun getLookupInfoUrl(parsedUrl: MobileAppStoreUrl): URI =
        with(URIBuilder(LOOKUP_URL)) {
            addParameter("id", parsedUrl.storeContentId.substring(2))
            addParameter("country", parsedUrl.storeCountry)
            addParameter("l", parsedUrl.storeLanguage)
            return build()
        }

    data class ItunesLookupResponse(
        val resultsCount: Int,
        val results: List<Map<String, Any>>
    )

    private fun downloadLookupInfo(url: URI): Map<String, Any> {
        val res = zoraService.fetchByUrl(url.toString(), true)
        ErrorUtils.checkResultForErrors(res, ItunesContentProcessor::ProcessItunesContentException)
        if (!res.success.isOk) {
            throw ProcessItunesContentException("Itunes returned empty lookup response for $url")
        }

        val lookupResponse: ItunesLookupResponse = fromJson(res.success.response.responseBody)

        if (lookupResponse.results.isEmpty()) {
            throw ProcessItunesContentException("Itunes returned response with 0 resultsCount for $url")
        }
        return lookupResponse.results.first()
    }

    class ProcessItunesContentException(message: String?) : ParseAppStoreContentException(message)
}

