package ru.yandex.direct.core.entity.uac.service.appinfo

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl
import ru.yandex.direct.core.entity.uac.model.MobileAppStoreUrlWithOriginal
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.zora.ZoraService
import ru.yandex.direct.http.smart.error.ErrorUtils
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.grut.client.GrutClient
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqParseAppStoreContent
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TRspParseAppStoreContent
import ru.yandex.yabs.rmp.tools.libs.app_stores_parser.library.proto.GooglePlay.TGooglePlayParserResult
import ru.yandex.yabs.rmp.tools.libs.app_stores_parser.library.proto.Itunes.TItunesParserResult

typealias ParsedAppStoreContent = TRspParseAppStoreContent
typealias ParsedGooglePlayContent = TGooglePlayParserResult
typealias ParsedItunesContent = TItunesParserResult

@Service
class ParseAppStoreContentService(
    private val grutClient: GrutClient,
    private val zoraService: ZoraService,
) {

    fun parseAppStoreContent(url: MobileAppStoreUrlWithOriginal): ParsedAppStoreContent {
        val pageHtml = downloadAppStorePage(fixUrl(url.originalUrl, url.parsedUrl.store))
        val req = when (url.parsedUrl.store) {
            Store.GOOGLE_PLAY -> TReqParseAppStoreContent.newBuilder()
                .setGooglePlayContent(pageHtml)
                .setRegion(url.parsedUrl.storeCountry)
                .build()
            Store.ITUNES -> TReqParseAppStoreContent.newBuilder()
                .setItunesContent(pageHtml)
                .build()
            else -> throw ParseAppStoreContentException("Unknown or empty Store for ${url.parsedUrl}")
        }
        return grutClient.parseAppStoreContent(req)
    }

    private fun downloadAppStorePage(url: String): String {
        val res = zoraService.fetchByUrl(url, true)
        ErrorUtils.checkResultForErrors(res, ParseAppStoreContentService::ParseAppStoreContentException)
        if (!res.success.isOk) {
            throw ParseAppStoreContentException("Store returned ${res.success.statusCode} status code for $url")
        }

        return res.success.response.responseBody
    }


    private fun fixUrl(url: String, store: Store): String {
        val builder = UrlParts.fromUrl(url).toBuilder()
        if (store == Store.GOOGLE_PLAY) {
            builder.addParamIfNotExists("hl", MobileAppStoreUrl.DEFAULT_LANGUAGE.toLowerCase())
            builder.addParamIfNotExists("gl", MobileAppStoreUrl.DEFAULT_COUNTRY.toLowerCase())
        }
        return builder.build().toUrl()
    }

    open class ParseAppStoreContentException(message: String?) : RuntimeException(message)
}
