package ru.yandex.direct.core.entity.uac.service.appinfo

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser
import ru.yandex.direct.core.entity.uac.model.MobileAppStoreUrlWithOriginal
import ru.yandex.direct.core.entity.uac.service.trackingurl.ParserType
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService
import ru.yandex.direct.core.entity.uac.validation.invalidAppUrl
import ru.yandex.direct.core.entity.uac.validation.invalidTrackingUrl
import ru.yandex.direct.core.entity.zora.ZoraService
import ru.yandex.direct.core.service.urlchecker.RedirectChecker.extractLocation
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Service
class ParseAppStoreUrlService(
    private val trackingUrlParseService: TrackingUrlParseService,
    private val zoraService: ZoraService,
) {

    // Returns parsed app store url from tracking or store url or defect
    fun getAppStoreUrl(requestUrl: String): ValidationResult<MobileAppStoreUrlWithOriginal, Defect<*>> {
        val trackingUrl = trackingUrlParseService.parse(requestUrl, null, ParserType.TRACKING_URL)
        return if (trackingUrl != null) {
            val redirectLocation = getRedirectLocation(trackingUrl.getUrl())
            parseStoreUrl(
                url = redirectLocation,
                isTrackingUrl = true,
                defect = invalidTrackingUrl()
            )
        } else {
            parseStoreUrl(
                url = requestUrl,
                isTrackingUrl = false,
                defect = invalidAppUrl()
            )
        }
    }

    private fun parseStoreUrl(
        url: String?,
        isTrackingUrl: Boolean,
        defect: Defect<Void>
    ): ValidationResult<MobileAppStoreUrlWithOriginal, Defect<*>> {
        val parsedUrl = MobileAppStoreUrlParser.parseWithDefect(url, defect)
        return if (!parsedUrl.hasAnyErrors()) {
            val parsedResult = parsedUrl.value
            ValidationResult.success(MobileAppStoreUrlWithOriginal(parsedResult, url!!, isTrackingUrl))
        } else {
            ValidationResult.failed(null, parsedUrl.errors.first())
        }
    }

    private fun getRedirectLocation(url: String): String? =
        zoraService.fetchByUrl(url, false)
            ?.takeIf { it.errors.isNullOrEmpty() }
            ?.success
            ?.takeIf { it.isRedirect }
            ?.let { extractLocation(url, it) }
}
