package ru.yandex.direct.core.entity.uac.service.shopinshop

import ru.yandex.direct.core.entity.uac.model.Source
import ru.yandex.direct.utils.model.UrlParts

class MarketUrlResolver : ShopInShopUrlResolver {
    companion object {
        private val MARKET_DOMAINS: Set<String> = setOf(
            "m.market.yandex.ru", "market.yandex.ru", "pokupki.market.yandex.ru"
        )

        private const val BUSINESS_ID_URL_PARAM = "businessId"
        private const val BUSINESS_PATH = "/business--"
        private const val STORE_PATH = "/store--"
        private const val DIRECT_ACCOUNT_URL_PATTERN = "https://partner.market.yandex.ru/business/%d/direct-bonus"
    }

    override val source: Source = Source.MARKET

    override val targetDomain: String = "market.yandex.ru"

    override fun isApplicableByUrl(url: UrlParts): Boolean {
        val domain = url.domain
        return MARKET_DOMAINS.contains(domain)
    }

    override fun getBusinessIdFromUrl(url: UrlParts): Long? {
        val params = url.parameters?.associate { it.left to it.right } ?: emptyMap()
        val path = url.path

        val businessId = if (path.startsWith(BUSINESS_PATH)) {
            path.takeLastWhile { it != '/' }
        } else {
            params[BUSINESS_ID_URL_PARAM]
        }

        return businessId?.toLongOrNull()
    }

    override fun isAdditionalCommission(url: UrlParts): Boolean {
        // Считаем, что ссылки на витрину в Маркете (https://market.yandex.ru/store--{store}?businessId={bussinessId})
        // не облагаются дополнительной комиссией. Для остальных необходимо показывать плашку про комиссию Маркета
        return !url.path.startsWith(STORE_PATH)
    }

    override fun getUrlWithoutCommission(url: UrlParts, businessId: Long): String? {
        val path = url.path

        if (path.startsWith(STORE_PATH)) {
            return url.toUrl()
        } else if (path.startsWith(BUSINESS_PATH)) {
            val businessName = path.substringAfter(BUSINESS_PATH).takeWhile { it != '/' }
            if (businessName.isNotBlank()) {
                return getUrlByStore(url.protocol, url.domain, businessName, businessId)
            }
        }

        return null
    }

    override fun getUrlToShopInShopAccount(url: UrlParts, businessId: Long): String {
        return String.format(DIRECT_ACCOUNT_URL_PATTERN, businessId)
    }

    private fun getUrlByStore(protocol: String, domain: String, store: String, businessId: Long): String {
        return "$protocol://$domain$STORE_PATH$store?$BUSINESS_ID_URL_PARAM=$businessId"
    }
}
