package ru.yandex.direct.core.entity.uac.service.stat

import java.math.BigDecimal
import java.math.RoundingMode
import java.time.Duration
import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.bs.common.repository.BsCommonRepository
import ru.yandex.direct.core.entity.uac.average
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.campaign_content.AssetStat
import ru.yandex.direct.core.entity.uac.model.campaign_content.ContentEfficiency
import ru.yandex.direct.core.entity.uac.repository.stat.AssetStatRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.utils.CommonUtils.nvl

data class ContentEfficiencyWithExistenceAndStat(
    val contentEfficiency: ContentEfficiency,
    val existence: Boolean,
    val assetStat: AssetStat?,
)

class ThresholdClicksCosts(
    val assetStats: List<AssetStat>,
    val countOfActiveAssets: Int,
) {
    val thresholdClicksCost1: BigDecimal
    val thresholdClicksCost2: BigDecimal
    val thresholdClicksCost3: BigDecimal
    val thresholdClicksCost4: BigDecimal

    init {
        val averageClicksCost = assetStats
            .map { it.clicksCost }
            .average(countOfActiveAssets)

        thresholdClicksCost1 = averageClicksCost.divide(BigDecimal(2.25), RoundingMode.HALF_UP)
        thresholdClicksCost2 = averageClicksCost.divide(BigDecimal(1.5), RoundingMode.HALF_UP)
        thresholdClicksCost3 = averageClicksCost
        thresholdClicksCost4 = averageClicksCost.multiply(BigDecimal(1.5))
    }

    fun doesNotHaveThresholds() = thresholdClicksCost4.compareTo(BigDecimal.ZERO) == 0
}

@Service
class AssetStatService(
    private val assetStatRepository: AssetStatRepository,
    private val bsCommonRepository: BsCommonRepository,
    private val shardHelper: ShardHelper,
) {
    companion object {
        private val THRESHOLD_SHOWS_OF_ACTIVE_ASSET = BigDecimal.valueOf(1000L)
        private const val THRESHOLD_DAYS_OF_ACTIVE_ASSET = 7L
        private val logger = LoggerFactory.getLogger(AssetStatService::class.java)
    }

    /**
     * Возвращает эффективность и флаг частичного существования ассета в переданный промежуток времени по id ассета.
     * Частичное существование ассета - когда он существовал не во всём промежутке времени [fromTime] - [toTime]
     */
    fun getAssetsEfficiencyAndExistence(
        uacCampaignContents: List<UacYdbCampaignContent>,
        campaignId: Long,
        fromTime: Long,
        toTime: Long,
    ): Map<String, ContentEfficiencyWithExistenceAndStat> {
        val shard = shardHelper.getShardByCampaignId(campaignId)
        val orderId = bsCommonRepository.getOrderIdForCampaigns(shard, listOf(campaignId))[campaignId]
            ?: return emptyMap()

        val now = LocalDateTime.now()
        val assetStats = getAssetStats(orderId, uacCampaignContents, fromTime, toTime)
        val assetStatsById = assetStats.associateBy { it.id }
        val typeToActiveAssetIds = getActiveAssetIdsByType(uacCampaignContents, assetStats, now)
        val typeToThresholdClicksCosts = getThresholdClicksCostsByType(assetStats, typeToActiveAssetIds)

        // Отбираем существовавшие ассеты, которые существовали не во всем промежутке времени
        val campaignContentIdsPartlyInPeriod = getCampaignContentIdsThatNotCompletelyInPeriod(
            uacCampaignContents,
            fromTime,
            toTime,
        )

        return uacCampaignContents
            .associateBy { it.id }
            .mapValues {

                val activeAssetIds = typeToActiveAssetIds[it.value.type]
                val thresholdClicksCosts = typeToThresholdClicksCosts[it.value.type]

                if (activeAssetIds != null && activeAssetIds.contains(it.key)) {
                    val clicksCost = assetStatsById[it.key]?.clicksCost ?: BigDecimal.ZERO
                    calcEfficiency(clicksCost, thresholdClicksCosts!!)
                } else {
                    ContentEfficiency.COLLECTING
                }
            }
            .mapValues {
                ContentEfficiencyWithExistenceAndStat(
                    it.value,
                    campaignContentIdsPartlyInPeriod.contains(it.key),
                    assetStatsById[it.key],
                )
            }
    }

    private fun getActiveAssetIdsByType(
        uacCampaignContents: Collection<UacYdbCampaignContent>,
        assetStats: List<AssetStat>,
        now: LocalDateTime,
    ): Map<MediaType, List<String>> {
        val assetIdToStat = assetStats
            .associateBy { it.id }
        return uacCampaignContents
            .filter { isAssetActive(assetIdToStat[it.id]?.shows, it, now) }
            .groupBy({ it.type!! }, { it.id })
    }

    private fun getThresholdClicksCostsByType(
        assetStats: List<AssetStat>,
        typeToActiveAssetIds: Map<MediaType, List<String>>,
    ): Map<MediaType, ThresholdClicksCosts> {
        val typeToAssetStats = assetStats
            .groupBy { it.mediaType }
        return typeToActiveAssetIds.keys
            .associateBy({ it }, {
                ThresholdClicksCosts(
                    nvl(typeToAssetStats[it], emptyList()),
                    typeToActiveAssetIds[it]!!.size
                )
            })
    }

    private fun getAssetStats(
        orderId: Long,
        uacCampaignContent: Collection<UacYdbCampaignContent>,
        fromTime: Long,
        toTime: Long,
    ): List<AssetStat> {
        val assetIds = uacCampaignContent
            .map { it.id }
            .toSet()
        val assetStats = assetStatRepository.getStatsByAssetIds(assetIds, orderId, fromTime, toTime)
        logger.info("Received ${assetStats.size} statistical data by order id $orderId, asset ids $assetIds" +
            " and in the period from $fromTime to $toTime"
        )
        return assetStats
    }

    /**
     * Возвращает true если ассет был показан >= 1000 раз или прошло 7 дней с момента создания ассета (DIRECT-149218)
     */
    private fun isAssetActive(
        shows: BigDecimal?,
        uacCampaignContent: UacYdbCampaignContent,
        now: LocalDateTime,
    ) = nvl(shows, BigDecimal.ZERO) >= THRESHOLD_SHOWS_OF_ACTIVE_ASSET
        || uacCampaignContent.createdAt + Duration.ofDays(THRESHOLD_DAYS_OF_ACTIVE_ASSET) < now

    /**
     * Возвращает эффективность ассета по его стоимости, в зависимости
     * от порогов [thresholdClicksCost1] - [thresholdClicksCost4]
     */
    private fun calcEfficiency(
        clicksClicksCost: BigDecimal,
        thresholdClicksCosts: ThresholdClicksCosts,
    ): ContentEfficiency {
        if (thresholdClicksCosts.doesNotHaveThresholds()) {
            return ContentEfficiency.ONE
        }

        return when {
            clicksClicksCost > thresholdClicksCosts.thresholdClicksCost4 -> ContentEfficiency.FIVE
            clicksClicksCost > thresholdClicksCosts.thresholdClicksCost3 -> ContentEfficiency.FOUR
            clicksClicksCost > thresholdClicksCosts.thresholdClicksCost2 -> ContentEfficiency.THREE
            clicksClicksCost > thresholdClicksCosts.thresholdClicksCost1 -> ContentEfficiency.TWO
            else -> ContentEfficiency.ONE
        }
    }

    /**
     * Возвращает ассеты campaign_content
     * которые существовали в любое время, в течении периода [fromTime] - [toTime]
     */
    fun getExistedCampaignContentAtPeriod(
        uacCampaignContent: List<UacYdbCampaignContent>,
        fromTime: Long,
        toTime: Long,
    ): List<UacYdbCampaignContent> {

        val localDateTimeFrom = UacYdbUtils.fromEpochSecond(fromTime)
        val localDateTimeTo = UacYdbUtils.fromEpochSecond(toTime)

        return uacCampaignContent
            .filter { it.createdAt <= localDateTimeTo }
            .filter { it.removedAt == null || it.removedAt >= localDateTimeFrom }
    }

    /**
     * Возвращает ассеты campaign_content которые существовали не всё время периода [fromTime] - [toTime]
     */
    private fun getCampaignContentIdsThatNotCompletelyInPeriod(
        uacCampaignContent: List<UacYdbCampaignContent>,
        fromTime: Long,
        toTime: Long,
    ): Set<String> {

        // Считаем что у ассета есть статистика за полный период если он был создан в день [fromTime] или раньше
        // и если удален, то в день [toTime] или позже
        val localDateTimeFrom = UacYdbUtils.fromEpochSecond(fromTime).toLocalDate()
            .atStartOfDay().plusDays(1).minusSeconds(1)
        val localDateTimeTo = UacYdbUtils.fromEpochSecond(toTime).toLocalDate().atStartOfDay()

        return uacCampaignContent
            .filter {
                it.createdAt > localDateTimeFrom
                    || (it.removedAt != null && it.removedAt < localDateTimeTo)
            }
            .map { it.id }
            .toSet()
    }
}
