package ru.yandex.direct.core.entity.uac.service.trackingurl

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.model.Parameter
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.TrackingSystem
import ru.yandex.direct.core.entity.uac.validation.cannotMakeImpressionUrl
import ru.yandex.direct.utils.model.UrlParts
import org.apache.commons.lang3.tuple.Pair as ParamPair

abstract class AdjustBaseUrlParser(type: ParserType) : UrlParser(type) {
    override val trackingSystem = TrackingSystem.ADJUST

    val baseParameters = listOf(
        Parameter(name = "idfa", value = IOS_IFA_MACROS, platform = Platform.IOS, required = true),

        Parameter(name = "gps_adid", value = GOOGLE_AID_MACROS, platform = Platform.ANDROID, required = true),
        Parameter(name = "oaid", value = OAID_MACROS, platform = Platform.ANDROID, required = true),

        Parameter(name = "ya_click_id", value = LOGID_MACROS, platform = null, required = true),
        Parameter(
            name = "campaign", value = CAMPAIGN_MACROS, platform = null, required = false,
            description = "Название кампании (латиницей)",
        ),
        Parameter(
            name = "adgroup", value = ADGROUP_MACROS, platform = null, required = false,
            description = "Идентификатор группы объявлений",
        ),
        Parameter(
            name = "creative", value = CREATIVE_MACROS, platform = null, required = false,
            description = "Объявление, ключевое слово или условие ретаргетинга",
        ),
    )

    override val skadNetworkIntegrated = true
    override val externalLink = "https://www.adjust.com"

    override fun transformParams(builder: UrlParts.Builder) {
        listOf("install_callback", "conversion_callback").forEach { paramToTransform ->
            builder.removeParamIf(paramToTransform) { paramValue ->
                paramValue.contains("postback.yandexadexchange.net")
            }
        }
    }
}

@Component
class AdjustTrackingUrlParser : AdjustBaseUrlParser(ParserType.TRACKING_URL) {
    override val hostPatterns = listOf(
        """(^|\.)adjust\.(com|io)$""".toRegex(),
    )
    override val hasImpression = true
    override val trackerIdFetcher = InPathTrackerIdFetcher()
    override val parameters = baseParameters

    override fun createImpressionUrl(urlParts: UrlParts) = CreateImpressionUrlResult(
        urlParts = urlParts.toBuilder()
            .withDomain("view.adjust.com")
            .withPath("/impression/${trackerIdFetcher.getTrackerId(urlParts)}")
            .build()
    )
}

@Component
class AdjustUniversalTrackingUrlParser : AdjustBaseUrlParser(ParserType.TRACKING_URL) {
    override val hostPatterns = listOf(
        """(^|\.)adj\.st$""".toRegex(),
    )
    override val hasImpression = true
    override val trackerIdFetcher = AdjustUniversalTrackerIdFetcher()
    override val parameters = listOf(
        Parameter(name = "adj_idfa", value = IOS_IFA_MACROS, platform = Platform.IOS, required = true),

        Parameter(name = "adj_gps_adid", value = GOOGLE_AID_MACROS, platform = Platform.ANDROID, required = true),
        Parameter(name = "adj_oaid", value = OAID_MACROS, platform = Platform.ANDROID, required = true),

        Parameter(name = "adj_ya_click_id", value = LOGID_MACROS, platform = null, required = true),
        Parameter(
            name = "adj_campaign", value = CAMPAIGN_MACROS, platform = null, required = false,
            description = "Название кампании (латиницей)",
        ),
        Parameter(
            name = "adj_adgroup", value = ADGROUP_MACROS, platform = null, required = false,
            description = "Идентификатор группы объявлений",
        ),
        Parameter(
            name = "adj_creative", value = CREATIVE_MACROS, platform = null, required = false,
            description = "Объявление, ключевое слово или условие ретаргетинга",
        ),
    )

    private fun getImpressionParams(params: List<ParamPair<String, String>>?): List<ParamPair<String, String>>? {
        if (params == null) {
            return null
        }
        return params.filter {
            it.left !in listOf("adjust_t", "adj_t")
        }.map {
            if (it.left.startsWith("adjust_")) {
                ParamPair.of(it.left.removePrefix("adjust_"), it.right)
            } else if (it.left.startsWith("adj_")) {
                ParamPair.of(it.left.removePrefix("adj_"), it.right)
            } else {
                it
            }
        }
    }

    override fun createImpressionUrl(urlParts: UrlParts): CreateImpressionUrlResult {
        val trackerId = trackerIdFetcher.getTrackerId(urlParts)
        return if (trackerId != null) {
            CreateImpressionUrlResult(
                urlParts = urlParts.toBuilder()
                    .withDomain("view.adjust.com")
                    .withPath("/impression/${trackerIdFetcher.getTrackerId(urlParts)}")
                    .withParameters(getImpressionParams(urlParts.parameters))
                    .build()
            )
        } else {
            CreateImpressionUrlResult(
                defect = cannotMakeImpressionUrl()
            )
        }
    }
}

class AdjustUniversalTrackerIdFetcher : TrackerIdFetcher {
    override fun getTrackerId(urlParts: UrlParts): String? {
        return urlParts.parameters?.firstOrNull { it.left in listOf("adjust_t", "adj_t") }?.right
    }
}

@Component
class AdjustImpressionUrlParser : AdjustBaseUrlParser(ParserType.IMPRESSION_URL) {
    override val hostPatterns = listOf(
        """^view\.adjust\.(com|io)$""".toRegex(),
    )
    override val pathPattern = """^/impression/""".toRegex()
    override val parameters = baseParameters + listOf(
        Parameter(name = "user_agent", value = "{user_agent}", platform = null, required = true),
        Parameter(name = "ip_address", value = "{client_ip}", platform = null, required = true),
        Parameter(name = "language", value = "{device_lang}", platform = null, required = true)
    )

    override val trackerIdFetcher = InPathTrackerIdFetcher()
}
