package ru.yandex.direct.core.entity.uac.service.trackingurl

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.model.Parameter
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.TrackingSystem
import ru.yandex.direct.utils.model.UrlParts

abstract class AppsflyerBaseUrlParser(type: ParserType) : UrlParser(type) {
    companion object {
        private const val YANDEXDIRECT_INT = "yandexdirect_int"
        private const val IS_RETARGETING = "true"
        private const val CAMPAIGN_NAME_MACROS = "{campaign_name}"
        private const val CAMPAIGN_ID_MACROS = "{campaign_id}"
        private const val ADGROUP_MACROS = "{gbid}"
        private const val AD_MACROS = "{ad_id}"
        private const val KEYWORDS_MACROS = "{phrase_id}{retargeting_id}_{keyword}{adtarget_name}"
        private const val SITE_MACROS = "{source_type}_{source}"
    }

    override val trackingSystem = TrackingSystem.APPSFLYER
    override val skadNetworkIntegrated = true

    val baseParameters = listOf(
        Parameter(name = "idfa", value = IOS_IFA_MACROS, platform = Platform.IOS, required = true),

        Parameter(name = "advertising_id", value = GOOGLE_AID_MACROS, platform = Platform.ANDROID, required = true),
        Parameter(name = "oaid", value = OAID_MACROS, platform = Platform.ANDROID, required = true),

        Parameter(name = "pid", value = YANDEXDIRECT_INT, platform = null, required = true),
        Parameter(name = "clickid", value = LOGID_MACROS, platform = null, required = true),
        Parameter(name = "is_retargeting", value = IS_RETARGETING, platform = null, required = false),
        Parameter(
            name = "c", value = CAMPAIGN_NAME_MACROS, platform = null, required = true,
            description = "Название кампании", overwrite = false
        ),
        Parameter(
            name = "af_c_id", value = CAMPAIGN_ID_MACROS, platform = null, required = true,
            description = "Идентификатор кампании"
        ),
        Parameter(
            name = "af_adset_id", value = ADGROUP_MACROS, platform = null, required = false,
            description = "Идентификатор группы объявлений"
        ),
        Parameter(
            name = "af_ad_id", value = AD_MACROS, platform = null, required = false,
            description = "Идентификатор объявления"
        ),
        Parameter(
            name = "af_keywords", value = KEYWORDS_MACROS, platform = null, required = false,
            description = "Ключевые слова"
        ),
        Parameter(
            name = "af_siteid", value = SITE_MACROS, platform = null, required = false,
            description = "Сайт сети, где был показ (место показа)"
        )
    )

    override val externalLink = "https://www.appsflyer.com"
}

@Component
class AppsflyerTrackingUrlParser : AppsflyerBaseUrlParser(ParserType.TRACKING_URL) {
    override val hostPatterns = listOf(
        """^app\.appsflyer\.com$""".toRegex(),
    )
    override val hasImpression = true
    override val trackerIdFetcher = InPathTrackerIdFetcher()
    override val parameters = baseParameters

    override fun createImpressionUrl(urlParts: UrlParts) = CreateImpressionUrlResult(
        urlParts = urlParts.toBuilder()
            .withDomain("impression.appsflyer.com")
            .removeParamIf("af_click_lookback") {true}
            .addParamIfNotExists("af_viewthrough_lookback", "1d")
            .build()
    )
}

@Component
class AppsflyerUniversalTrackingUrlParser : AppsflyerBaseUrlParser(ParserType.TRACKING_URL) {
    override val hostPatterns = listOf(
        """.+\.onelink\.me$""".toRegex(),
    )
    override val trackingSystem = TrackingSystem.APPSFLYERUL
    override val hasImpression = true
    override val trackerIdFetcher = InPathTrackerIdFetcher()
    override val parameters = baseParameters
}

@Component
class AppsflyerImpressionUrlParser : AppsflyerBaseUrlParser(ParserType.IMPRESSION_URL) {
    override val hostPatterns = listOf(
        """^impression\.appsflyer\.com$""".toRegex(),
    )
    override val parameters = baseParameters + listOf(
        Parameter(name = "af_ip", value = "{client_ip}", platform = null, required = true),
        Parameter(name = "af_ua", value = "{user_agent}", platform = null, required = true),
        Parameter(name = "af_lang", value = "{device_lang}", platform = null, required = true)
    )

    override val trackerIdFetcher = InPathTrackerIdFetcher()
}
