package ru.yandex.direct.core.entity.uac.service.trackingurl

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.model.Parameter
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.TrackingSystem
import ru.yandex.direct.utils.model.UrlParts

abstract class KochavaBaseUrlParser(type: ParserType) : UrlParser(type) {
    companion object {
        private const val ADID = "adid"
        private const val NETWORK_ID_ANDROID = "1517"
        private const val NETWORK_ID_IOS = "1516"
        private const val IDFA = "idfa"
        private const val SITE_MACROS = "{source_type}_{source}"
        private const val AD_MACROS = "{ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}"
    }

    override val trackingSystem: TrackingSystem = TrackingSystem.KOCHAVA

    val baseParameters = listOf(
        Parameter(name = "network_id", value = NETWORK_ID_IOS, platform = Platform.IOS, required = true),
        Parameter(name = "device_id_type", value = IDFA, platform = Platform.IOS, required = true),
        Parameter(name = "idfa", value = IOS_IFA_MACROS, platform = Platform.IOS, required = true),
        Parameter(name = "device_id", value = IOS_IFA_MACROS, platform = Platform.IOS, required = true),
        Parameter(name = "ios_idfa", value = IOS_IFA_MACROS, platform = Platform.IOS, required = true),

        Parameter(name = "network_id", value = NETWORK_ID_ANDROID, platform = Platform.ANDROID, required = true),
        Parameter(name = "device_id_type", value = ADID, platform = Platform.ANDROID, required = true),
        Parameter(name = "adid", value = GOOGLE_AID_MACROS, platform = Platform.ANDROID, required = true),
        Parameter(name = "device_id", value = GOOGLE_AID_MACROS, platform = Platform.ANDROID, required = true),
        Parameter(name = "android_id", value = ANDROID_ID_MACROS, platform = Platform.ANDROID, required = true),

        Parameter(
            name = "site_id", value = SITE_MACROS, platform = null, required = false,
            description = "Сайт сети, где был показ (место показа)"
        ),
        Parameter(
            name = "creative_id", value = AD_MACROS, platform = null, required = false,
            description = "Объявление, ключевое слово или условие ретаргетинга"
        )
    )

    override val externalLink = "https://www.kochava.com"
}

@Component
class KochavaTrackingUrlParser : KochavaBaseUrlParser(ParserType.TRACKING_URL) {
    override val hostPatterns = listOf(
        """^control\.kochava\.com$""".toRegex(),
    )
    override val parameters = baseParameters + listOf(
        Parameter(name = "click_id", value = LOGID_MACROS, platform = null, required = true)
    )
    override val hasImpression = true
    override val trackerIdFetcher = KochavaTrackerIdFetcher()
}

class KochavaTrackerIdFetcher: TrackerIdFetcher {
    override fun getTrackerId(urlParts: UrlParts): String? {
        return urlParts.parameters?.firstOrNull{it.left == "campaign_id"}?.right
    }
}

@Component
class KochavaImpressionUrlParser : KochavaBaseUrlParser(ParserType.IMPRESSION_URL) {
    override val hostPatterns = listOf(
        """^imp\.control\.kochava\.com$""".toRegex(),
    )
    override val parameters = baseParameters + listOf(
        Parameter(name = "impression_id", value = LOGID_MACROS, platform = null, required = true)
    )

    override val trackerIdFetcher = KochavaTrackerIdFetcher()
}
